"""
Collect test items at filesystem and python module levels. 

Collectors and test items form a tree.  The difference
between a collector and a test item as seen from the session 
is smalll.  Collectors usually return a list of child 
collectors/items whereas items usually return None 
indicating a successful test run.  

The is a schematic example of a tree of collectors and test items:: 

    Directory
        Module 
            Class 
                Instance   
                    Function  
                    Generator 
                        ... 
            Function 
            Generator 
                Function 
        Directory       
            ... 

""" 
from __future__ import generators 
import py
isclass = py.std.inspect.isclass 

def configproperty(name):
    def fget(self):
        #print "retrieving %r property from %s" %(name, self.fspath)
        return py.test.Config.getvalue(name, self.fspath) 
    return property(fget)

def getfscollector(fspath):
    if isinstance(fspath, str): 
        fspath = py.path.local(fspath)
    if not fspath.check(): 
       raise py.error.ENOENT(fspath) 
    pkgpath = fspath.pypkgpath() 
    if pkgpath is None: 
        pkgpath = fspath.dirpath() 
    Directory = py.test.Config.getvalue('Directory', pkgpath)
    current = Directory(pkgpath) 
    #print "pkgpath", pkgpath
    names = filter(None, fspath.relto(pkgpath).split(fspath.sep))
    for name in names: 
        current = current.join(name) 
        assert current, "joining %r resulted in None!" % (names,)
    top = current.listchain()[0]
    #top._config = config 
    return current 
 
class Collector(object): 
    def __init__(self, name, parent=None): 
        self.name = name 
        self.parent = parent 
        self.option = getattr(parent, 'option', None)
        self.fspath = getattr(parent, 'fspath', None) 

    Module = configproperty('Module')
    Directory = configproperty('Directory')
    Class = configproperty('Class')
    Instance = configproperty('Instance')
    Function = configproperty('Function')
    Generator = configproperty('Generator')

    _stickyfailure = None
    class Outcome: 
        def __init__(self, msg=None, excinfo=None): 
            self.msg = msg 
            self.excinfo = excinfo 
        def __repr__(self):
            if self.msg: 
                return self.msg 
            return "<%s instance>" %(self.__class__.__name__,)
        __str__ = __repr__

    class Passed(Outcome): 
        pass
    class Failed(Outcome): 
        pass
    class ExceptionFailure(Failed): 
        def __init__(self, expr, expected, msg=None, excinfo=None): 
            Collector.Failed.__init__(self, msg=msg, excinfo=excinfo) 
            self.expr = expr 
            self.expected = expected  
    class Skipped(Outcome): 
        pass

    def __repr__(self): 
        return "<%s %r>" %(self.__class__.__name__, self.name) 

    def __eq__(self, other): 
        try: 
            return self.name == other.name and self.parent == other.parent 
        except AttributeError: 
            return False 

    def __cmp__(self, other): 
        s1 = self.getsortvalue()
        s2 = other.getsortvalue()
        #print "cmp", s1, s2
        return cmp(s1, s2) 

    def obj(): 
        def fget(self):
            try: 
                return self._obj   
            except AttributeError: 
                self._obj = obj = self._getobj() 
                return obj 
        def fset(self, value): 
            self._obj = value 
        return property(fget, fset, None, "underlying object") 
    obj = obj()

    def _getobj(self): 
        return getattr(self.parent.obj, self.name) 

    def multijoin(self, namelist): 
        """ return a list of colitems for the given namelist. """ 
        return [self.join(name) for name in namelist]

    def getpathlineno(self): 
        return self.fspath, py.std.sys.maxint 

    def setup(self): 
        pass 

    def teardown(self): 
        pass 

    def listchain(self): 
        """ return list of all parent collectors up to ourself. """ 
        l = [self]
        while 1: 
            x = l[-1]
            if x.parent is not None: 
                l.append(x.parent) 
            else: 
                l.reverse() 
                return l 

    def listnames(self): 
        return [x.name for x in self.listchain()]

    def haskeyword(self, keyword): 
        return self.name.startswith(keyword)

    def getmodpath(self):
        """ return dotted module path (relative to the containing). """ 
        inmodule = False 
        newl = []
        for x in self.listchain(): 
            if not inmodule and not isinstance(x, Module): 
                continue
            if not inmodule:  
                inmodule = True
                continue
            if newl and x.name[:1] in '([': 
                newl[-1] += x.name 
            else: 
                newl.append(x.name) 
        return ".".join(newl) 

    def tryiter(self, stopitems=None): 
        """ yield stop item instances from flattening the collector. 
        """ 
        if stopitems is None: 
            stopitems = py.test.Item 
        if isinstance(self, stopitems): 
            yield self
        else:
            for x in self.run(): 
                for y in self.join(x).tryiter(stopitems): 
                    yield y

    def _prepare(self): 
        if not hasattr(self, '_name2items'): 
            ex = getattr(self, '_name2items_exception', None)
            if ex is not None: 
                raise ex[0], ex[1], ex[2]
            try: 
                self._name2items = self.buildname2items()
            except (SystemExit, KeyboardInterrupt): 
                raise 
            except:
                self._name2items_exception = py.std.sys.exc_info()
                raise 

    def buildname2items(self): 
        raise NotImplementedError, "abstract" 

    def getsortvalue(self): 
        return self.name 

    def run(self): 
        self._prepare()
        itemlist = self._name2items.values()
        itemlist.sort()
        return [x.name for x in itemlist]

    def join(self, name): 
        self._prepare()
        return self._name2items.get(name, None) 

    captured_out = captured_err = None
    def startcapture(self): 
        return None # by default collectors don't capture output 
    def finishcapture(self): 
        return None # by default collectors don't capture output 
    def getouterr(self): 
        return self.captured_out, self.captured_err 

class FSCollector(Collector): 
    def __init__(self, fspath, parent=None): 
        fspath = py.path.local(fspath) 
        super(FSCollector, self).__init__(fspath.basename, parent) 
        self.fspath = fspath 

class Directory(FSCollector): 
    def filefilter(self, path): 
        pb = path.purebasename 
        return (path.check(fnmatch="*.py") and 
               (pb.startswith('test_') or pb.endswith('_test')))
    
    def recfilter(self, path): 
        return path.check(dotfile=0) and \
               path.basename not in ('CVS', '_darcs', '{arch}')

    def buildname2items(self): 
        d = {} 
        for p in self.fspath.listdir(): 
            x = self.makeitem(p.basename, self.filefilter, self.recfilter)
            if x is not None: 
                d[p.basename] = x
        return d 

    def makeitem(self, basename, filefilter=None, recfilter=None): 
        p = self.fspath.join(basename)
        if p.check(file=1) and (not filefilter or filefilter(p)): 
            return self.Module(p, parent=self) 
        elif p.check(dir=1) and (not recfilter or recfilter(p)): 
            Directory = py.test.Config.getvalue('Directory', p) 
            return Directory(p, parent=self) 

    def join(self, name): 
        x = super(Directory, self).join(name)
        if x is None:    
            x = self.makeitem(name)
        return x 

class PyCollectorMixin(object): 
    def funcnamefilter(self, name): 
        return name.startswith('test') 
    def classnamefilter(self, name): 
        return name.startswith('Test')

    def buildname2items(self): 
        # NB. we avoid random getattrs and peek in the __dict__ instead
        d = {}
        dicts = [getattr(self.obj, '__dict__', {})]
        for basecls in py.std.inspect.getmro(self.obj.__class__):
            dicts.append(basecls.__dict__)
        seen = {}
        for dic in dicts:
            for name, obj in dic.items():
                if name in seen:
                    continue
                seen[name] = True
                if self.classnamefilter(name) and isclass(obj): 
                    d[name] = self.Class(name, parent=self) 
                elif self.funcnamefilter(name) and callable(obj): 
                    if obj.func_code.co_flags & 32: # generator function 
                        d[name] = self.Generator(name, parent=self) 
                    else: 
                        d[name] = self.Function(name, parent=self) 
        return d

class Module(PyCollectorMixin, FSCollector): 
    
    def run(self):
        if getattr(self.obj, 'disabled', 0):
            return []
        return FSCollector.run(self)
    
    def startcapture(self): 
        if not self.option.nocapture and not self.option.usepdb: 
            assert not hasattr(self, '_capture')
            #self._capture = py.io.OutErrCapture() 
            # XXX integrate this into py.io / refactor
            #     execnet/py.test capturing mechanisms
            from py.__.misc.simplecapture import SimpleOutErrCapture
            self._capture = SimpleOutErrCapture()

    def finishcapture(self): 
        if hasattr(self, '_capture'): 
            capture = self._capture 
            del self._capture 
            self.captured_out, self.captured_err = capture.reset()

    def __repr__(self): 
        return "<%s %r>" % (self.__class__.__name__, self.name)

    def obj(self): 
        try: 
            return self._obj    
        except AttributeError:
            failure = getattr(self, '_stickyfailure', None)
            if failure is not None: 
                raise failure[0], failure[1], failure[2]
            try: 
                self._obj = obj = self.fspath.pyimport() 
            except KeyboardInterrupt: 
                raise
            except: 
                self._stickyfailure = py.std.sys.exc_info()
                raise 
            return obj 
    obj = property(obj, None, None, "module object")

    def setup(self): 
        if hasattr(self.obj, 'setup_module'): 
            self.obj.setup_module(self.obj) 

    def teardown(self): 
        if hasattr(self.obj, 'teardown_module'): 
            self.obj.teardown_module(self.obj) 


class Class(PyCollectorMixin, Collector): 

    def run(self): 
        if getattr(self.obj, 'disabled', 0):
            return []
        return ["()"]

    def join(self, name): 
        assert name == '()' 
        return self.Instance(name, self) 

    def setup(self): 
        setup_class = getattr(self.obj, 'setup_class', None)
        if setup_class is not None: 
            setup_class = getattr(setup_class, 'im_func', setup_class) 
            setup_class(self.obj) 

    def teardown(self): 
        teardown_class = getattr(self.obj, 'teardown_class', None) 
        if teardown_class is not None: 
            teardown_class = getattr(teardown_class, 'im_func', teardown_class) 
            teardown_class(self.obj) 

    def getsortvalue(self): 
        for x in self.tryiter((py.test.collect.Generator, py.test.Item)): 
            return x.getsortvalue()

class Instance(PyCollectorMixin, Collector): 
    def _getobj(self): 
        return self.parent.obj()  
    def Function(self): 
        return getattr(self.obj, 'Function', 
                       Collector.Function.__get__(self)) # XXX for python 2.2 
    Function = property(Function)

class Generator(Collector): 
    def buildname2items(self): 
        d = {} 
        for i, x in py.builtin.enumerate(self.obj()): 
            call, args = self.getcallargs(x)
            if not callable(call): 
                raise TypeError("yielded test %r not callable" %(call,))
            name = "[%d]" % i
            #XXX name = "%s(%r)" %(call.__name__, str(args and args[0] or i)) # # XXX
            d[name] = self.Function(name, self, args, obj=call, sort_value = i)
        return d 
                
    def getcallargs(self, obj):
        if isinstance(obj, (tuple, list)):
            call, args = obj[0], obj[1:]
        else:
            call, args = obj, ()
        return call, args 

    def getpathlineno(self): 
        code = py.code.Code(self.obj) 
        return code.path, code.firstlineno 

    def getsortvalue(self):  
        return self.getpathlineno() 
