/*
 *
 * CLEX File Manager
 *
 * Copyright (C) 2001-2005 Vlado Potisk <vlado_potisk@clex.sk>
 *
 * CLEX is free software without warranty of any kind; see the
 * GNU General Public License as set out in the "COPYING" document
 * which accompanies the CLEX File Manager package.
 *
 * CLEX can be downloaded from http://www.clex.sk
 *
 */

#include <config.h>

#include <sys/types.h>		/* clex.h */
#include <sys/stat.h>		/* umask() */
#include <errno.h>			/* errno */
#include <stdarg.h>			/* va_list */
#include <stdio.h>			/* vprint() */
#include <stdlib.h>			/* free() */
#include <string.h>			/* strerror() */

#include "clex.h"
#include "bookmarks.h"

#include "control.h"		/* control_loop() */
#include "edit.h"			/* edit_setprompt() */
#include "filepanel.h"		/* changedir() */
#include "inout.h"			/* txt_printf() */
#include "panel.h"			/* pan_adjust() */
#include "ustring.h"		/* us_reset() */
#include "util.h"			/* read_file() */

/* limits to protect resources */
#define BM_FILESIZE_LIMIT	10000
#define BM_ENTRIES_LIMIT	200

#define BM_LIST (panel_bm.bm)	/* list of bookmarks */
static int bm_cnt = 0;			/* number of bookmarks */
static FLAG changed = 0;		/* bookmarks have changed */
static const char *user_bm_file;	/* personal bookmarks filename */
static time_t bm_file_mod = 0;		/* last modification of the file */

extern int errno;

static void
bm_error(const char *format, ...)
{
	va_list argptr;

	txt_printf("BOOKMARKS: Reading the bookmark file \"%s\"\n",
	  user_bm_file);

	va_start(argptr,format);
	fputs("BOOKMARKS: ",stdout);
	vprintf(format,argptr);
	va_end(argptr);
}

/*
 * return value:
 *  -1 = failure, original bookmarks are unchanged
 *   0 = bookmarks loaded (with or without problems)
 */
static int
read_bm_file(void)
{
	int  error;
	size_t filesize;
	char *source, *ptr;

	filesize = BM_FILESIZE_LIMIT;
	source = read_file(user_bm_file,&filesize,&error);
	if (source == 0) {
		switch (error) {
		case 1:
			if (errno == ENOENT) {
				bm_cnt = 0;
				return 0;	/* no file is OK */
			}
			bm_error("Cannot open the file (%s)\n",strerror(errno));
			break;
		case 2:
			bm_error("File is too big, limit is "
			  STR(BM_FILESIZE_LIMIT) " bytes\n");
			break;
		case 3:
			bm_error("File read error (%s)\n",strerror(errno));
		}
		return -1;
	}

	bm_cnt = 0;
	for (ptr = source; ptr < source + filesize; ptr++)
		if (*ptr == '\n')
			*ptr = '\0';

	for (ptr = source; ptr < source + filesize; ) {
		if (*ptr == '/') {
			if (bm_cnt == BM_ENTRIES_LIMIT) {
				bm_error("Too many lines, limit is "
				  STR(BM_ENTRIES_LIMIT) "\n");
				break;
			}
			us_copy(&BM_LIST[bm_cnt],ptr);
			bm_cnt++;
		}
		/* advance to the next line */
		while (*ptr++)
			;
	}
	free(source);
	return 0;
}

void
bm_initialize(void)
{
	int i;
	time_t mod;
	static USTRING list[BM_ENTRIES_LIMIT];

	BM_LIST = list;
	for (i = 0; i < BM_ENTRIES_LIMIT; i++)
		US_INIT(BM_LIST[i]);

	pathname_set_directory(clex_data.homedir);
	user_bm_file = estrdup(pathname_join(".clexbm"));

	mod = mod_time(user_bm_file);
	if (read_bm_file() == 0)
		bm_file_mod = mod;
}

void
bm_prepare(void)
{
	time_t mod;

	mod = mod_time(user_bm_file);
	if (bm_file_mod != mod && read_bm_file() == 0) {
		bm_file_mod = mod;
		win_warning("New version of the bookmarks was loaded.");
	}

	panel_bm.pd->cnt = bm_cnt;
	panel_bm.pd->min = -3;
	panel_bm.pd->type = PANEL_TYPE_BM;
	if (panel_bm.pd->curs < 0)
		panel_bm.pd->curs = panel_bm.pd->min;
	if (panel_bm.pd->top < 0)
		panel_bm.pd->top = panel_bm.pd->min;
	/* otherwise leave the cursor where it was */

	panel = panel_bm.pd;
	textline = 0;
}

void
bm_manager_prepare(void)
{
	panel_bm.pd->cnt = bm_cnt;
	panel_bm.pd->top = panel_bm.pd->min = panel_bm.pd->curs = -1;
	panel_bm.pd->type = PANEL_TYPE_BM_MANAGER;

	panel = panel_bm.pd;
	/*
	 * this won't be recognized as a panel change
	 * (same panel, just different type), that's why
	 * we must redraw the screen explicitely
	 */
	pan_adjust(panel_bm.pd);
	win_panel();

	textline = 0;
}

void
bm_edit_prepare(void)
{
	/* panel = panel_bm.pd; */
	textline = &line_tmp;
	edit_setprompt(textline,"bookmark: ");
	edit_nu_putstr(USTR(BM_LIST[panel_bm.pd->curs]));
}

static void
bm_save(void)
{
	int i;
	FLAG errflag;
	FILE *fp;

	umask(clex_data.umask | 022);
	fp = fopen(user_bm_file,"w");
	umask(clex_data.umask);

	if (fp == 0) {
		win_warning("BOOKMARKS: Cannot open the bookmark file "
		  "for writing.");
		return;
	}
	fprintf(fp,	"#\n"
				"# CLEX bookmark file\n"
				"#\n");
	for (i = 0; i < bm_cnt; i++)
		fprintf(fp,"%s\n",USTR(BM_LIST[i]));
	errflag = ferror(fp) != 0;
	if (fclose(fp) || errflag)
		win_warning("BOOKMARKS: File write error occurred.");
	else {
		win_remark("bookmarks updated");
		changed = 0;
	}
	bm_file_mod = mod_time(user_bm_file);
}

static void
bm_add(const char *dir)
{
	int i;

	for (i = 0; i < bm_cnt; i++)
		if (strcmp(USTR(BM_LIST[i]),dir) == 0) {
			win_remark("already bookmarked");
			return;
		}

	if (bm_cnt == BM_ENTRIES_LIMIT) {
		win_warning("Bookmark list is full.");
		return;
	}

	us_copy(&BM_LIST[bm_cnt],dir);
	panel_bm.pd->curs = bm_cnt;
	panel_bm.pd->cnt = ++bm_cnt;
	pan_adjust(panel_bm.pd);
	win_panel();

	changed = 1;
	bm_save();
}

void
cx_bm_enter(void)
{
	if (panel_bm.pd->curs == -3)
		next_mode = MODE_SPECIAL_RETURN;
	else if (panel_bm.pd->curs == -2)
		next_mode = MODE_BM_MANAGER;
	else if (panel_bm.pd->curs == -1)
		bm_add(USTR(ppanel_file->dir));
	else if (changedir(USTR(BM_LIST[panel_bm.pd->curs])) == 0)
		next_mode = MODE_SPECIAL_RETURN;
}

void
cx_bm_manager_save(void)
{
	if (changed)
		bm_save();
	next_mode = MODE_SPECIAL_RETURN;
}

void
cx_bm_manager_enter(void)
{
	if (panel_bm.pd->curs == -1)
		cx_bm_manager_save();
	else {
		control_loop(MODE_BM_EDIT);
		win_panel_opt();
	}
}

void
cx_bm_edit_enter(void)
{
	if (*USTR(textline->line) != '/') {
		win_warning("Directory name must start with a slash / . ");
		return;
	}
	us_xchg(&textline->line,&BM_LIST[panel_bm.pd->curs]);
	changed = 1;
	next_mode = MODE_SPECIAL_RETURN;
}

void
cx_bm_manager_up(void)
{
	int pos;

	if ((pos = panel_bm.pd->curs) == 0)
		return;

	us_xchg(&BM_LIST[pos],&BM_LIST[panel_bm.pd->curs = pos - 1]);
	changed = 1;

	win_panel();
}

void
cx_bm_manager_down(void)
{
	int pos;

	if ((pos = panel_bm.pd->curs) == bm_cnt - 1)
		return;

	us_xchg(&BM_LIST[pos],&BM_LIST[panel_bm.pd->curs = pos + 1]);
	changed = 1;

	win_panel();
}

void
cx_bm_manager_del(void)
{
	int i, del;
	USTRING x;

	del = panel_bm.pd->curs;

	panel_bm.pd->cnt = --bm_cnt;
	if (panel_bm.pd->curs == bm_cnt) {
		panel_bm.pd->curs--;
		pan_adjust(panel_bm.pd);
	}

	/* these are struct copy operations */
	x = BM_LIST[del];
	for (i = del; i < bm_cnt; i++)
		BM_LIST[i] = BM_LIST[i + 1];
	BM_LIST[bm_cnt] = x;
	changed = 1;

	win_panel();
}

void
cx_bm_manager_new(void)
{
	int i, ins;
	USTRING x;

	if (bm_cnt == BM_ENTRIES_LIMIT) {
		win_warning("Bookmark list is full.");
		return;
	}

	ins = ++panel_bm.pd->curs;

	/* these are struct copy operations */
	x = BM_LIST[bm_cnt];
	for (i = bm_cnt; i > ins; i--)
		BM_LIST[i] = BM_LIST[i - 1];
	BM_LIST[ins] = x;
	panel_bm.pd->cnt = ++bm_cnt;
	us_copy(&BM_LIST[ins],"-- new bookmark --");
	changed = 1;

	pan_adjust(panel_bm.pd);
	win_panel();
	us_copy(&BM_LIST[ins],"/");
	cx_bm_manager_enter();
}
