;;;;
;;;; AspectL
;;;;
;;;; Copyright (c) 2005 Pascal Costanza
;;;;
;;;; Permission is hereby granted, free of charge, to any person
;;;; obtaining a copy of this software and associated documentation
;;;; files (the "Software"), to deal in the Software without
;;;; restriction, including without limitation the rights to use,
;;;; copy, modify, merge, publish, distribute, sublicense, and/or
;;;; sell copies of the Software, and to permit persons to whom the
;;;; Software is furnished to do so, subject to the following
;;;; conditions:
;;;;
;;;; The above copyright notice and this permission notice shall be
;;;; included in all copies or substantial portions of the Software.
;;;;
;;;; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
;;;; EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
;;;; OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
;;;; NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
;;;; HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
;;;; WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
;;;; FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
;;;; OTHER DEALINGS IN THE SOFTWARE.
;;;;

(in-package #:al.dynaclos)

(defclass special-class (standard-class)
  (old-slot-definitions)
  (:documentation
   "The metaclass for classes that allow for dynamic slot bindings via dletf and dletf*.
    Special slots can be rebound via slot-value, slot-value-using-class and the reader functions
    automatically generated by a defclass form. Example:

    (defclass person ()
      ((name :accessor person-name :initarg :name :special t))
      (:metaclass special class))

    (defvar *p* (make-instance 'person :name \"Dr. Jekyll\"))

    (dletf (((person-name *p*) \"Mr. Hide\"))
      (print (person-name *p*)))"))

(defmethod validate-superclass ((class special-class) (superclass standard-class))
  t)

(defclass special-object ()
  ()
  (:documentation
   "Implicitly added as a superclass to each class whose metaclass is special-class."))

(defmethod initialize-instance :around
  ((class special-class) &rest initargs)
  (declare (dynamic-extent initargs))
  (apply #'initialize-class-metaobject #'call-next-method
         class (find-class 'special-class) (find-class 'special-object)
         #+lispworks :optimize-slot-access #+lispworks nil
         initargs))

(defmethod reinitialize-instance :around
  ((class special-class) &rest initargs)
  (declare (dynamic-extent initargs))
  (apply #'reinitialize-class-metaobject #'call-next-method
         class (find-class 'special-class) (find-class 'special-object)
         #+lispworks :optimize-slot-access #+lispworks nil
         initargs))

(defgeneric special-slot-p (slotd)
  (:documentation "Is the :special flag set for slotd?")
  (:method ((slotd slot-definition)) nil))

(defclass special-class-slot-definition-mixin ()
  ((special-slot-p :accessor special-slot-p
		   :initarg :special
		   :initform nil)))

(defclass special-class-direct-slot-definition
          (special-class-slot-definition-mixin
           standard-direct-slot-definition)
  ())

(defmethod direct-slot-definition-class
           ((class special-class)
            &key &allow-other-keys)
  (find-class 'special-class-direct-slot-definition))

(defclass special-class-effective-slot-definition
          (special-class-slot-definition-mixin
           standard-effective-slot-definition)
  ())

(defmethod effective-slot-definition-class
           ((class special-class) &key &allow-other-keys)
  (find-class 'special-class-effective-slot-definition))

(defmethod compute-effective-slot-definition
           ((class special-class) slot-name direct-slot-definitions)
  (declare (ignore slot-name))
  (let ((effective-slotd (call-next-method)))
    (setf (special-slot-p effective-slotd)
          (special-slot-p (first direct-slot-definitions)))
    effective-slotd))

(defgeneric ensure-slot-symbol-using-class (class object slot)
  (:method ((class special-class) (object special-object) slot)
	   "Ensure that the special slot is bound to a special symbol."
	   (with-symbol-access
	    (if (slot-boundp-using-class class object slot)
		(slot-value-using-class class object slot)
	      (setf (slot-value-using-class class object slot)
		    (make-special-symbol))))))

(defmethod slot-value-using-class
           ((class special-class) (object special-object) slot)
  (if *symbol-access* (call-next-method)
    (let ((slotd (the-effective-slot-definition class slot)))
      (if (special-slot-p slotd)
          (let ((slot-symbol (ensure-slot-symbol-using-class class object slot)))
            (if (boundp slot-symbol) (symbol-value slot-symbol)
              (slot-unbound class object (slot-definition-name slotd))))
        (call-next-method)))))

(defmethod (setf slot-value-using-class)
           ((new-value t) (class special-class) (object special-object) slot)
  (if *symbol-access* (call-next-method)
    (let ((slotd (the-effective-slot-definition class slot)))
      (if (special-slot-p slotd)
          (let ((slot-symbol (ensure-slot-symbol-using-class class object slot)))
            (setf (symbol-value slot-symbol) new-value))
        (call-next-method)))))

(defmethod slot-boundp-using-class
           ((class special-class) (object special-object) slot)
  (if *symbol-access* (call-next-method)
    (let ((slotd (the-effective-slot-definition class slot)))
      (if (special-slot-p slotd)
          (boundp (ensure-slot-symbol-using-class class object slot))
        (call-next-method)))))

(defmethod slot-makunbound-using-class
           ((class special-class) (object special-object) slot)
  (if *symbol-access* (call-next-method)
    (let ((slotd (the-effective-slot-definition class slot)))
      (if (special-slot-p slotd)
          (progn
            (makunbound (ensure-slot-symbol-using-class class object slot))
            object)
        (call-next-method)))))

(defmethod shared-initialize :after
  ((instance special-object)
   slot-names &rest all-keys)
  "ensure that all initialized special slots are indeed bound to
   a special slot; circumvents possible optimizations in the
   initialization of standard-class objects"
  (declare (ignore slot-names all-keys))
  (flet ((shift-slot (slotd)
           (when (special-slot-p slotd)
             (let ((slot-name (slot-definition-name slotd)))
               (with-symbol-access
                 (when (slot-boundp instance slot-name)
                   (let ((slot-value (slot-value instance slot-name)))
                     (unless (special-symbol-p slot-value)
                       (slot-makunbound instance slot-name)
                       (without-symbol-access
                         (setf (slot-value instance slot-name) slot-value))))))))))
    (mapc #'shift-slot (class-slots (class-of instance)))))

(defmethod reinitialize-instance :before
  ((class special-class) &rest initargs)
  (declare (ignore initargs))
  (setf (slot-value class 'old-slot-definitions)
        (class-slots class)))

(defmethod finalize-inheritance :after
  ((class special-class))
  "ensure that special slots remain special after class redefinition
   (there is no protocol for collapsing multiple values in different
   dynamic scopes for the same special slot); make instances obsolete
   when non-special slots have been turned into special slots"
  (assert (or (not (slot-boundp class 'old-slot-definitions))
              (every (lambda (old-slot)
                       (let ((new-slot (the-effective-slot-definition
                                        class (slot-definition-name old-slot))))
			 #+allegro
			 (cond ((null new-slot) t)
			       (t (eql (special-slot-p old-slot)
				       (special-slot-p new-slot))))
			 #-allegro
                         (cond ((null new-slot) t)
                               ((special-slot-p old-slot)
                                (special-slot-p new-slot))
                               (t (when (special-slot-p new-slot)
                                    (make-instances-obsolete class))
                                  t))))
                     (slot-value class 'old-slot-definitions)))
      ()
    #+allegro "The (non-)special slots in class ~S must remain (non-)special."
    #-allegro "The special slots in class ~S must remain special."
    (class-name class))
  (slot-makunbound class 'old-slot-definitions))

(defmethod slot-options ((class special-class))
  (cons :special (call-next-method)))

(defmethod get-slot-option
           ((class special-class)
            (slot special-class-slot-definition-mixin)
            (option (eql :special)))
  (special-slot-p slot))

(defmethod default-slot-option
           ((class special-class) (option (eql :special)) &optional ignore)
  (declare (ignore ignore))
  nil)
