/*
 * ProgressBar.hpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright (c) 2005-2006  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
  @file ProgressBar.hpp
   
  Several variants (attempts) of concretisations of `ProgressInfo' via Fltk.
  Content:
   - ProgressBar     - derived from Fl_Progress, i.e. no top level window.
   - ProgressWindow  - derived from Fl_Window, Fl_Progress as element.
   - ProgressNewWin  - no Fltk-Widget (Fl_Window and Fl_Progress as elements);
                        each show() creates a Window, each() hide() deletes it.
			
   @todo  Fix permanently debug output.
*/
#ifndef ProgressBar_hpp
#define ProgressBar_hpp

#include <iostream>                  // debug output
#include <FL/Fl_Progress.H>
#include "../br_core/ProgressInfo.hpp"


//#define BR_DEBUG_PROGRESSBAR

/**======================================================================
* 
*   @class ProgressBar  -  derived from Fl_Progess, no top level window
* 
*   Konkretisierung eines `ProgressInfo' mittels Fltk: erbt von ProgressInfo
*   unten konkretisierte abstrakte Funktionen und von Fl_Progress den Rest.
* 
*   Ctor with Fl_Widget interface, so usable in FLUID.
* 
*========================================================================*/
class ProgressBar : public ProgressInfo, public Fl_Progress
{
public:
    ProgressBar (int X, int Y, int W, int H, const char* la=0)
      : Fl_Progress(X,Y,W,H,la)
      { 
        Fl_Progress::minimum(0.0);    // accordingly to the ProgressInfo
        Fl_Progress::maximum(1.0);    //  convention "v \in [0,1]"
      }
  
    void show()               {
#      ifdef BR_DEBUG_PROGRESSBAR    
       std::cout << "\tProgressBar::show()...\n";
#      endif       
       Fl_Progress::show(); Fl::flush(); }
    
    void hide()               {
#      ifdef BR_DEBUG_PROGRESSBAR    
       std::cout << "\tProgressBar::hide()...\n";
#      endif       
       Fl_Progress::hide(); Fl_Progress::value(0.0); }
    
    void value (float v)      {
#      ifdef BR_DEBUG_PROGRESSBAR    
       std::cout << "\tProgressBar::value( "<< v <<" )...\n"; 
#      endif       
       Fl_Progress::value(v); Fl::flush(); }
    
    void text (const char* s) {
#      ifdef BR_DEBUG_PROGRESSBAR    
       std::cout << "\tProgressBar::text(\""<< s <<"\")...\n";
#      endif       
       Fl_Progress::label(s); Fl::flush(); }
};


/**======================================================================
* 
*   @class ProgressWindow  -  ProgressBar in a separate top level window
* 
*   Ctor with Fl_Widget interface, so usable in FLUID.
* 
*========================================================================*/
class ProgressWindow : public ProgressInfo, public Fl_Double_Window
{
    Fl_Progress progress_;
    
    void init_() { 
        progress_.minimum(0.0);    // accordingly to the ProgressInfo
        progress_.maximum(1.0);    //  convention "v \in [0,1]"
      }
    
public:
    ProgressWindow (int X, int Y, int W, int H, const char* la=0)
      : Fl_Double_Window (X,Y,W,H,la),
        progress_(0,0,W,H)
      { init_(); } 
  
    ProgressWindow (int W, int H, const char* la=0)
      : Fl_Double_Window (W,H,la),
        progress_(0,0,W,H)
      { init_(); } 
    
    void show() {
        std::cout << "\tProgressWindow::show()...\n";
        Fl_Double_Window::show(); }
    void hide() {
        std::cout << "\tProgressWindow::hide()...\n";
        Fl_Double_Window::hide(); progress_.value(0.0); }
    void value (float v) {
        std::cout << "\tProgressWindow::value( "<< v <<" )...\n"; 
        progress_.value(v); redraw(); Fl::flush(); }
    void text (const char* s) {
        std::cout << "\tProgressWindow::text(\""<< s <<"\")...\n";
        progress_.label(s); Fl::flush(); }
};


/**======================================================================
* 
*   @class ProgressNewWin  
*
*   Each show() creates a new Window, and each hide() delete()s it. Would 
*    allow hotspot positioning.
* 
*   Ctor with Fl_Widget interface, usable (via make_window()) in FLUID.
* 
*========================================================================*/
class ProgressNewWin : public ProgressInfo
{
    int               w_, h_;
    const char*       label_;
    Fl_Double_Window* window_;
    Fl_Progress*      progress_;
    
public:
    ProgressNewWin (int W, int H, const char* la=0)
      : w_(W), h_(H), label_(la),
        window_ (0),
        progress_(0)
      {} 
    
    void show() { //const Fl_Widget* spot=0) {
        std::cout << "\tProgressNewWin::show()...\n";
        if (!window_) {
          window_ = new Fl_Double_Window(w_,h_,label_);
          { progress_ = new Fl_Progress(0,0,w_,h_);
            progress_->minimum(0.0);    // accordingly to the ProgressInfo
            progress_->maximum(1.0);    //  convention "v \in [0,1]"
          }
          //if (spot) window_->hotspot(spot);
          window_->end();
        }
        window_->show(); 
      }
    
    void hide() {
        std::cout << "\tProgressNewWin::hide()...\n";
        if (window_) {
          window_->hide(); delete window_; window_=0; // Fl::flush();
        } 
      }
    
    void value (float v) {
        std::cout << "\tProgressNewWin::value( "<< v <<" )...\n"; 
        progress_->value(v); window_->redraw(); Fl::flush(); 
      }
    
    void text (const char* s) {
        std::cout << "\tProgressNewWin::text(\""<< s <<"\")...\n";
        progress_->label(s); Fl::flush(); 
      }
};

#endif  // ProgressBar_hpp

// END OF FILE
