// Copyright (c) 2010, Google Inc.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//     * Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above
// copyright notice, this list of conditions and the following disclaimer
// in the documentation and/or other materials provided with the
// distribution.
//     * Neither the name of Google Inc. nor the names of its
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE

#include <stdlib.h>
#include <unistd.h>

#include <string>

#include "breakpad_googletest_includes.h"
#include "common/using_std_string.h"
#include "google_breakpad/processor/basic_source_line_resolver.h"
#include "google_breakpad/processor/call_stack.h"
#include "google_breakpad/processor/code_module.h"
#include "google_breakpad/processor/code_modules.h"
#include "google_breakpad/processor/minidump.h"
#include "google_breakpad/processor/minidump_processor.h"
#include "google_breakpad/processor/process_state.h"
#include "google_breakpad/processor/stack_frame.h"
#include "google_breakpad/processor/symbol_supplier.h"

namespace google_breakpad {
class MockMinidump : public Minidump {
 public:
  MockMinidump() : Minidump("") {
  }

  MOCK_METHOD0(Read, bool());
  MOCK_CONST_METHOD0(path, string());
  MOCK_CONST_METHOD0(header, const MDRawHeader*());
  MOCK_METHOD0(GetThreadList, MinidumpThreadList*());
};
}

namespace {

using google_breakpad::BasicSourceLineResolver;
using google_breakpad::CallStack;
using google_breakpad::CodeModule;
using google_breakpad::MinidumpProcessor;
using google_breakpad::MinidumpThreadList;
using google_breakpad::MinidumpThread;
using google_breakpad::MockMinidump;
using google_breakpad::ProcessState;
using google_breakpad::SymbolSupplier;
using google_breakpad::SystemInfo;

class TestSymbolSupplier : public SymbolSupplier {
 public:
  TestSymbolSupplier() : interrupt_(false) {}

  virtual SymbolResult GetSymbolFile(const CodeModule *module,
                                     const SystemInfo *system_info,
                                     string *symbol_file);

  virtual SymbolResult GetSymbolFile(const CodeModule *module,
                                     const SystemInfo *system_info,
                                     string *symbol_file,
                                     string *symbol_data);

  virtual SymbolResult GetCStringSymbolData(const CodeModule *module,
                                            const SystemInfo *system_info,
                                            string *symbol_file,
                                            char **symbol_data,
                                            size_t *symbol_data_size);

  virtual void FreeSymbolData(const CodeModule *module) { }
  // When set to true, causes the SymbolSupplier to return INTERRUPT
  void set_interrupt(bool interrupt) { interrupt_ = interrupt; }

 private:
  bool interrupt_;
};

SymbolSupplier::SymbolResult TestSymbolSupplier::GetSymbolFile(
    const CodeModule *module,
    const SystemInfo *system_info,
    string *symbol_file) {

  if (interrupt_) {
    return INTERRUPT;
  }

  return NOT_FOUND;
}

SymbolSupplier::SymbolResult TestSymbolSupplier::GetCStringSymbolData(
    const CodeModule *module,
    const SystemInfo *system_info,
    string *symbol_file,
    char **symbol_data,
    size_t *symbol_data_size) {
  return GetSymbolFile(module, system_info, symbol_file);
}

SymbolSupplier::SymbolResult TestSymbolSupplier::GetSymbolFile(
    const CodeModule *module,
    const SystemInfo *system_info,
    string *symbol_file,
    string *symbol_data) {
  return GetSymbolFile(module, system_info, symbol_file);
}

TEST(ExploitabilityTest, TestWindowsEngine) {
  TestSymbolSupplier supplier;
  BasicSourceLineResolver resolver;
  MinidumpProcessor processor(&supplier, &resolver, true);
  ProcessState state;

  string minidump_file = string(getenv("srcdir") ? getenv("srcdir") : ".") +
      "/src/processor/testdata/ascii_read_av.dmp";
  ASSERT_EQ(processor.Process(minidump_file, &state),
            google_breakpad::PROCESS_OK);
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            state.exploitability());

  minidump_file = string(getenv("srcdir") ? getenv("srcdir") : ".") +
      "/src/processor/testdata/ascii_read_av_block_write.dmp";
  ASSERT_EQ(processor.Process(minidump_file, &state),
            google_breakpad::PROCESS_OK);
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            state.exploitability());

  minidump_file = string(getenv("srcdir") ? getenv("srcdir") : ".") +
      "/src/processor/testdata/ascii_read_av_clobber_write.dmp";
  ASSERT_EQ(processor.Process(minidump_file, &state),
            google_breakpad::PROCESS_OK);
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            state.exploitability());

  minidump_file = string(getenv("srcdir") ? getenv("srcdir") : ".") +
      "/src/processor/testdata/ascii_read_av_conditional.dmp";
  ASSERT_EQ(processor.Process(minidump_file, &state),
            google_breakpad::PROCESS_OK);
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            state.exploitability());

  minidump_file = string(getenv("srcdir") ? getenv("srcdir") : ".") +
      "/src/processor/testdata/ascii_read_av_then_jmp.dmp";
  ASSERT_EQ(processor.Process(minidump_file, &state),
            google_breakpad::PROCESS_OK);
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            state.exploitability());

  minidump_file = string(getenv("srcdir") ? getenv("srcdir") : ".") +
      "/src/processor/testdata/ascii_read_av_xchg_write.dmp";
  ASSERT_EQ(processor.Process(minidump_file, &state),
            google_breakpad::PROCESS_OK);
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            state.exploitability());

  minidump_file = string(getenv("srcdir") ? getenv("srcdir") : ".") +
      "/src/processor/testdata/ascii_write_av.dmp";
  ASSERT_EQ(processor.Process(minidump_file, &state),
            google_breakpad::PROCESS_OK);
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            state.exploitability());

  minidump_file = string(getenv("srcdir") ? getenv("srcdir") : ".") +
      "/src/processor/testdata/ascii_write_av_arg_to_call.dmp";
  ASSERT_EQ(processor.Process(minidump_file, &state),
            google_breakpad::PROCESS_OK);
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            state.exploitability());

  minidump_file = string(getenv("srcdir") ? getenv("srcdir") : ".") +
      "/src/processor/testdata/null_read_av.dmp";
  ASSERT_EQ(processor.Process(minidump_file, &state),
            google_breakpad::PROCESS_OK);
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_NONE,
            state.exploitability());

  minidump_file = string(getenv("srcdir") ? getenv("srcdir") : ".") +
      "/src/processor/testdata/null_write_av.dmp";
  ASSERT_EQ(processor.Process(minidump_file, &state),
            google_breakpad::PROCESS_OK);
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_NONE,
            state.exploitability());

  minidump_file = string(getenv("srcdir") ? getenv("srcdir") : ".") +
      "/src/processor/testdata/stack_exhaustion.dmp";
  ASSERT_EQ(processor.Process(minidump_file, &state),
            google_breakpad::PROCESS_OK);
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_NONE,
            state.exploitability());

  minidump_file = string(getenv("srcdir") ? getenv("srcdir") : ".") +
      "/src/processor/testdata/exec_av_on_stack.dmp";
  ASSERT_EQ(processor.Process(minidump_file, &state),
            google_breakpad::PROCESS_OK);
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            state.exploitability());

  minidump_file = string(getenv("srcdir") ? getenv("srcdir") : ".") +
      "/src/processor/testdata/write_av_non_null.dmp";
  ASSERT_EQ(processor.Process(minidump_file, &state),
            google_breakpad::PROCESS_OK);
  ASSERT_EQ(google_breakpad::EXPLOITABLITY_MEDIUM,
            state.exploitability());

  minidump_file = string(getenv("srcdir") ? getenv("srcdir") : ".") +
      "/src/processor/testdata/read_av_non_null.dmp";
  ASSERT_EQ(processor.Process(minidump_file, &state),
            google_breakpad::PROCESS_OK);
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_LOW,
            state.exploitability());

  minidump_file = string(getenv("srcdir") ? getenv("srcdir") : ".") +
      "/src/processor/testdata/read_av_clobber_write.dmp";
  ASSERT_EQ(processor.Process(minidump_file, &state),
            google_breakpad::PROCESS_OK);
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_LOW,
            state.exploitability());

  minidump_file = string(getenv("srcdir") ? getenv("srcdir") : ".") +
      "/src/processor/testdata/read_av_conditional.dmp";
  ASSERT_EQ(processor.Process(minidump_file, &state),
            google_breakpad::PROCESS_OK);
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_LOW,
            state.exploitability());
}
}
