// Copyright (c) 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/renderer/translate/translate_helper.h"

#include "base/strings/utf_string_conversions.h"
#include "chrome/common/chrome_constants.h"
#include "testing/gtest/include/gtest/gtest.h"

typedef testing::Test TranslateHelperTest;

// Tests that well-known language code typos are fixed.
TEST_F(TranslateHelperTest, LanguageCodeTypoCorrection) {
  std::string language;

  // Strip the second and later codes.
  language = std::string("ja,en");
  TranslateHelper::CorrectLanguageCodeTypo(&language);
  EXPECT_EQ("ja", language);

  // Replace dash with hyphen.
  language = std::string("ja_JP");
  TranslateHelper::CorrectLanguageCodeTypo(&language);
  EXPECT_EQ("ja-JP", language);

  // Correct wrong cases.
  language = std::string("JA-jp");
  TranslateHelper::CorrectLanguageCodeTypo(&language);
  EXPECT_EQ("ja-JP", language);
}

// Tests that invalid language code is reset to empty string.
TEST_F(TranslateHelperTest, ResetInvalidLanguageCode) {
  std::string language;

  language = std::string("ja");
  TranslateHelper::ResetInvalidLanguageCode(&language);
  EXPECT_EQ("ja", language);

  language = std::string("ja-JP");
  TranslateHelper::ResetInvalidLanguageCode(&language);
  EXPECT_EQ("ja-JP", language);

  // Invalid because of three characters before hyphen.
  language = std::string("utf-8");
  TranslateHelper::ResetInvalidLanguageCode(&language);
  EXPECT_TRUE(language.empty());

  // Invalid because of six characters after hyphen.
  language = std::string("ja-YUKARI");
  TranslateHelper::ResetInvalidLanguageCode(&language);
  EXPECT_TRUE(language.empty());

  // Invalid because of three characters.
  language = std::string("YMO");
  TranslateHelper::ResetInvalidLanguageCode(&language);
  EXPECT_TRUE(language.empty());
}

// Tests that similar language table works.
TEST_F(TranslateHelperTest, SimilarLanguageCode) {
  EXPECT_TRUE(TranslateHelper::IsSameOrSimilarLanguages("en", "en"));
  EXPECT_FALSE(TranslateHelper::IsSameOrSimilarLanguages("en", "ja"));
  EXPECT_TRUE(TranslateHelper::IsSameOrSimilarLanguages("bs", "hr"));
  EXPECT_TRUE(TranslateHelper::IsSameOrSimilarLanguages("sr-ME", "sr"));
}

// Tests that well-known languages which often have wrong server configuration
// are handles.
TEST_F(TranslateHelperTest, WellKnownWrongConfiguration) {
  EXPECT_TRUE(TranslateHelper::MaybeServerWrongConfiguration("en", "ja"));
  EXPECT_TRUE(TranslateHelper::MaybeServerWrongConfiguration("en-US", "ja"));
  EXPECT_TRUE(TranslateHelper::MaybeServerWrongConfiguration("en", "zh-CN"));
  EXPECT_FALSE(TranslateHelper::MaybeServerWrongConfiguration("ja", "en"));
  EXPECT_FALSE(TranslateHelper::MaybeServerWrongConfiguration("en", "he"));
}

// Tests that the language meta tag providing wrong information is ignored by
// TranslateHelper due to disagreement between meta tag and CLD.
TEST_F(TranslateHelperTest, CLDDisagreeWithWrongLanguageCode) {
  string16 contents = ASCIIToUTF16(
      "<html><head><meta http-equiv='Content-Language' content='ja'></head>"
      "<body>This is a page apparently written in English. Even though "
      "content-language is provided, the value will be ignored if the value "
      "is suspicious.</body></html>");
  std::string cld_language;
  bool is_cld_reliable;
  std::string language =
      TranslateHelper::DeterminePageLanguage(std::string("ja"), std::string(),
                                             contents, &cld_language,
                                             &is_cld_reliable);
  EXPECT_EQ(chrome::kUnknownLanguageCode, language);
  EXPECT_EQ("en", cld_language);
  EXPECT_TRUE(is_cld_reliable);
}

// Tests that the language meta tag providing "en-US" style information is
// agreed by CLD.
TEST_F(TranslateHelperTest, CLDAgreeWithLanguageCodeHavingCountryCode) {
  string16 contents = ASCIIToUTF16(
      "<html><head><meta http-equiv='Content-Language' content='en-US'></head>"
      "<body>This is a page apparently written in English. Even though "
      "content-language is provided, the value will be ignored if the value "
      "is suspicious.</body></html>");
  std::string cld_language;
  bool is_cld_reliable;
  std::string language =
      TranslateHelper::DeterminePageLanguage(std::string("en-US"),
                                             std::string(), contents,
                                             &cld_language, &is_cld_reliable);
  EXPECT_EQ("en-US", language);
  EXPECT_EQ("en", cld_language);
  EXPECT_TRUE(is_cld_reliable);
}

// Tests that the language meta tag providing wrong information is ignored and
// CLD's language will be adopted by TranslateHelper due to an invalid meta tag.
TEST_F(TranslateHelperTest, InvalidLanguageMetaTagProviding) {
  string16 contents = ASCIIToUTF16(
      "<html><head><meta http-equiv='Content-Language' content='utf-8'></head>"
      "<body>This is a page apparently written in English. Even though "
      "content-language is provided, the value will be ignored and CLD's"
      " language will be adopted if the value is invalid.</body></html>");
  std::string cld_language;
  bool is_cld_reliable;
  std::string language =
      TranslateHelper::DeterminePageLanguage(std::string("utf-8"),
                                             std::string(), contents,
                                             &cld_language, &is_cld_reliable);
  EXPECT_EQ("en", language);
  EXPECT_EQ("en", cld_language);
  EXPECT_TRUE(is_cld_reliable);
}

// Tests that the language meta tag providing wrong information is ignored
// because of valid html lang attribute.
TEST_F(TranslateHelperTest, AdoptHtmlLang) {
  string16 contents = ASCIIToUTF16(
      "<html lang='en'><head><meta http-equiv='Content-Language' content='ja'>"
      "</head><body>This is a page apparently written in English. Even though "
      "content-language is provided, the value will be ignored if the value "
      "is suspicious.</body></html>");
  std::string cld_language;
  bool is_cld_reliable;
  std::string language =
      TranslateHelper::DeterminePageLanguage(std::string("ja"),
                                             std::string("en"),
                                             contents, &cld_language,
                                             &is_cld_reliable);
  EXPECT_EQ("en", language);
  EXPECT_EQ("en", cld_language);
  EXPECT_TRUE(is_cld_reliable);
}
