// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include <limits>
#include <string>

#include "base/basictypes.h"
#include "base/guid.h"
#include "base/strings/string_number_conversions.h"
#include "base/strings/stringprintf.h"
#include "base/strings/utf_string_conversions.h"
#include "base/test/test_reg_util_win.h"
#include "base/time.h"
#include "base/win/registry.h"
#include "chrome/installer/gcapi/gcapi.h"
#include "chrome/installer/util/google_update_constants.h"
#include "testing/gtest/include/gtest/gtest.h"

using base::Time;
using base::TimeDelta;
using base::win::RegKey;


class GCAPILastRunTest : public ::testing::Test {
 protected:
  void SetUp() {
    // Override keys - this is undone during destruction.
    std::wstring hkcu_override = base::StringPrintf(
        L"hkcu_override\\%ls", ASCIIToWide(base::GenerateGUID()));
    override_manager_.OverrideRegistry(HKEY_CURRENT_USER, hkcu_override);

    // Create the client state key in the right places.
    std::wstring reg_path(google_update::kRegPathClientState);
    reg_path += L"\\";
    reg_path += google_update::kChromeUpgradeCode;
    RegKey client_state(HKEY_CURRENT_USER,
                        reg_path.c_str(),
                        KEY_CREATE_SUB_KEY);
    ASSERT_TRUE(client_state.Valid());

    // Place a bogus "pv" value in the right places to make the last run
    // checker believe Chrome is installed.
    std::wstring clients_path(google_update::kRegPathClients);
    clients_path += L"\\";
    clients_path += google_update::kChromeUpgradeCode;
    RegKey client_key(HKEY_CURRENT_USER,
                      clients_path.c_str(),
                      KEY_CREATE_SUB_KEY | KEY_SET_VALUE);
    ASSERT_TRUE(client_key.Valid());
    client_key.WriteValue(L"pv", L"1.2.3.4");
  }

  bool SetLastRunTime(int64 last_run_time) {
    return SetLastRunTimeString(base::Int64ToString16(last_run_time));
  }

  bool SetLastRunTimeString(const string16& last_run_time_string) {
    const wchar_t* base_path = google_update::kRegPathClientState;
    std::wstring path(base_path);
    path += L"\\";
    path += google_update::kChromeUpgradeCode;

    RegKey client_state(HKEY_CURRENT_USER, path.c_str(), KEY_SET_VALUE);
    return (client_state.Valid() &&
            client_state.WriteValue(
                google_update::kRegLastRunTimeField,
                last_run_time_string.c_str()) == ERROR_SUCCESS);
  }

 private:
  registry_util::RegistryOverrideManager override_manager_;
};

TEST_F(GCAPILastRunTest, Basic) {
  Time last_run = Time::NowFromSystemTime() - TimeDelta::FromDays(10);
  EXPECT_TRUE(SetLastRunTime(last_run.ToInternalValue()));

  int days_since_last_run = GoogleChromeDaysSinceLastRun();
  EXPECT_EQ(10, days_since_last_run);
}

TEST_F(GCAPILastRunTest, NoLastRun) {
  int days_since_last_run = GoogleChromeDaysSinceLastRun();
  EXPECT_EQ(-1, days_since_last_run);
}

TEST_F(GCAPILastRunTest, InvalidLastRun) {
  EXPECT_TRUE(SetLastRunTimeString(L"Hi Mum!"));
  int days_since_last_run = GoogleChromeDaysSinceLastRun();
  EXPECT_EQ(-1, days_since_last_run);
}

TEST_F(GCAPILastRunTest, OutOfRangeLastRun) {
  Time last_run = Time::NowFromSystemTime() - TimeDelta::FromDays(-42);
  EXPECT_TRUE(SetLastRunTime(last_run.ToInternalValue()));

  int days_since_last_run = GoogleChromeDaysSinceLastRun();
  EXPECT_EQ(-1, days_since_last_run);
}
