// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_UI_VIEWS_COOKIE_INFO_VIEW_H_
#define CHROME_BROWSER_UI_VIEWS_COOKIE_INFO_VIEW_H_

#include <string>
#include <vector>

#include "base/basictypes.h"
#include "base/compiler_specific.h"
#include "base/strings/string16.h"
#include "ui/base/models/combobox_model.h"
#include "ui/views/controls/combobox/combobox_listener.h"
#include "ui/views/view.h"

namespace views {
class GridLayout;
class Label;
class Textfield;
}

namespace net {
class CanonicalCookie;
}

///////////////////////////////////////////////////////////////////////////////
// CookieInfoViewDelegate
//
class CookieInfoViewDelegate {
 public:
  virtual void ModifyExpireDate(bool session_expire) = 0;

 protected:
  virtual ~CookieInfoViewDelegate() {}
};

///////////////////////////////////////////////////////////////////////////////
// CookieInfoView
//
//  Responsible for displaying a tabular grid of Cookie information.
class CookieInfoView : public views::View,
                       public views::ComboboxListener,
                       public ui::ComboboxModel {
 public:
  explicit CookieInfoView(bool editable_expiration_date);
  virtual ~CookieInfoView();

  // Update the display from the specified CookieNode.
  void SetCookie(const std::string& domain,
                 const net::CanonicalCookie& cookie_node);

  // Update the display from the specified cookie string.
  void SetCookieString(const GURL& url, const std::string& cookie_line);

  // Clears the cookie display to indicate that no or multiple cookies are
  // selected.
  void ClearCookieDisplay();

  // Enables or disables the cookie property text fields.
  void EnableCookieDisplay(bool enabled);

  void set_delegate(CookieInfoViewDelegate* delegate) { delegate_ = delegate; }

 protected:
  // views::View:
  virtual void ViewHierarchyChanged(
      const ViewHierarchyChangedDetails& details) OVERRIDE;

  // views::ComboboxListener:
  virtual void OnSelectedIndexChanged(views::Combobox* combobox) OVERRIDE;

  // ui::ComboboxModel:
  virtual int GetItemCount() const OVERRIDE;
  virtual string16 GetItemAt(int index) OVERRIDE;

 private:
  // Layout helper routines.
  void AddLabelRow(int layout_id, views::GridLayout* layout,
                   views::View* label, views::View* value);
  void AddControlRow(int layout_id, views::GridLayout* layout,
                     views::View* label, views::View* control);

  // Sets up the view layout.
  void Init();

  // Individual property labels
  views::Label* name_label_;
  views::Textfield* name_value_field_;
  views::Label* content_label_;
  views::Textfield* content_value_field_;
  views::Label* domain_label_;
  views::Textfield* domain_value_field_;
  views::Label* path_label_;
  views::Textfield* path_value_field_;
  views::Label* send_for_label_;
  views::Textfield* send_for_value_field_;
  views::Label* created_label_;
  views::Textfield* created_value_field_;
  views::Label* expires_label_;
  views::Textfield* expires_value_field_;
  views::Combobox* expires_value_combobox_;
  views::View* expire_view_;

  // Option values for expires_value_combobox_.
  std::vector<string16> expire_combo_values_;

  // True if expiration date can be edited. In this case we will show
  // expires_value_combobox_ instead of expires_value_field_. The cookie's
  // expiration date is editable only this class is used in
  // CookiesPromptView (alert before cookie is set), in all other cases we
  // don't let user directly change cookie setting.
  bool editable_expiration_date_;

  CookieInfoViewDelegate* delegate_;

  DISALLOW_COPY_AND_ASSIGN(CookieInfoView);
};

#endif  // CHROME_BROWSER_UI_VIEWS_COOKIE_INFO_VIEW_H_
