// Copyright (c) 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_UI_COCOA_AUTOFILL_AUTOFILL_DIALOG_COCOA_H_
#define CHROME_BROWSER_UI_COCOA_AUTOFILL_AUTOFILL_DIALOG_COCOA_H_

#include "base/mac/scoped_nsobject.h"
#include "base/memory/scoped_ptr.h"
#include "chrome/browser/ui/autofill/autofill_dialog_types.h"
#include "chrome/browser/ui/autofill/autofill_dialog_view.h"
#import "chrome/browser/ui/cocoa/autofill/autofill_layout.h"
#include "chrome/browser/ui/cocoa/constrained_window/constrained_window_mac.h"

namespace content {
  class NavigationController;
}

namespace autofill {
  class AutofillDialogController;
}

@class AutofillAccountChooser;
@class AutofillDialogWindowController;
@class AutofillSignInContainer;
@class AutofillMainContainer;

namespace autofill {

class AutofillDialogCocoa : public AutofillDialogView,
                            public ConstrainedWindowMacDelegate {
 public:
  explicit AutofillDialogCocoa(AutofillDialogController* controller);
  virtual ~AutofillDialogCocoa();

  // AutofillDialogView implementation:
  virtual void Show() OVERRIDE;
  virtual void Hide() OVERRIDE;
  virtual void UpdateAccountChooser() OVERRIDE;
  virtual void UpdateButtonStrip() OVERRIDE;
  virtual void UpdateDetailArea() OVERRIDE;
  virtual void UpdateForErrors() OVERRIDE;
  virtual void UpdateNotificationArea() OVERRIDE;
  virtual void UpdateAutocheckoutStepsArea() OVERRIDE;
  virtual void UpdateSection(DialogSection section) OVERRIDE;
  virtual void FillSection(DialogSection section,
                           const DetailInput& originating_input) OVERRIDE;
  virtual void GetUserInput(DialogSection section,
                            DetailOutputMap* output) OVERRIDE;
  virtual string16 GetCvc() OVERRIDE;
  virtual bool SaveDetailsLocally() OVERRIDE;
  virtual const content::NavigationController* ShowSignIn() OVERRIDE;
  virtual void HideSignIn() OVERRIDE;
  virtual void UpdateProgressBar(double value) OVERRIDE;
  virtual void ModelChanged() OVERRIDE;
  virtual void OnSignInResize(const gfx::Size& pref_size) OVERRIDE;

  // ConstrainedWindowMacDelegate implementation.
  virtual void OnConstrainedWindowClosed(
      ConstrainedWindowMac* window) OVERRIDE;

  AutofillDialogController* controller() { return controller_; }

  void PerformClose();

 private:

  scoped_ptr<ConstrainedWindowMac> constrained_window_;
  base::scoped_nsobject<AutofillDialogWindowController> sheet_controller_;

  // The controller |this| queries for logic and state.
  AutofillDialogController* controller_;
};

}  // autofill

@interface AutofillDialogWindowController :
    NSWindowController<NSWindowDelegate, AutofillLayout> {
 @private
  content::WebContents* webContents_;  // weak.
  autofill::AutofillDialogCocoa* autofillDialog_;  // weak.

  base::scoped_nsobject<AutofillMainContainer> mainContainer_;
  base::scoped_nsobject<AutofillSignInContainer> signInContainer_;
  base::scoped_nsobject<AutofillAccountChooser> accountChooser_;
}

// Designated initializer. The WebContents cannot be NULL.
- (id)initWithWebContents:(content::WebContents*)webContents
      autofillDialog:(autofill::AutofillDialogCocoa*)autofillDialog;

// A child view request re-layouting.
- (void)requestRelayout;

// Validate data. If it is valid, notify the controller that the user would
// like to use the data.
- (IBAction)accept:(id)sender;

// User cancels dialog.
- (IBAction)cancel:(id)sender;

// Forwarding AutofillDialogView calls.
- (void)updateAccountChooser;
- (void)updateSection:(autofill::DialogSection)section;
- (void)getInputs:(autofill::DetailOutputMap*)outputs
       forSection:(autofill::DialogSection)section;
- (content::NavigationController*)showSignIn;
- (void)hideSignIn;
- (void)modelChanged;

@end

#endif  // CHROME_BROWSER_UI_COCOA_AUTOFILL_AUTOFILL_DIALOG_COCOA_H_
