// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/policy/cloud/policy_builder.h"

#include <vector>

#include "base/stl_util.h"
#include "chrome/browser/policy/cloud/cloud_policy_constants.h"
#include "chrome/browser/policy/proto/cloud/chrome_extension_policy.pb.h"
#include "crypto/signature_creator.h"
#include "policy/proto/cloud_policy.pb.h"

namespace em = enterprise_management;

namespace policy {

namespace {

// Signing key test data in DER-encoded PKCS8 format.
const uint8 kSigningKey[] = {
    0x30, 0x82, 0x01, 0x55, 0x02, 0x01, 0x00, 0x30, 0x0d, 0x06, 0x09, 0x2a,
    0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x82,
    0x01, 0x3f, 0x30, 0x82, 0x01, 0x3b, 0x02, 0x01, 0x00, 0x02, 0x41, 0x00,
    0xd9, 0xcd, 0xca, 0xcd, 0xc3, 0xea, 0xbe, 0x72, 0x79, 0x1c, 0x29, 0x37,
    0x39, 0x99, 0x1f, 0xd4, 0xb3, 0x0e, 0xf0, 0x7b, 0x78, 0x77, 0x0e, 0x05,
    0x3b, 0x65, 0x34, 0x12, 0x62, 0xaf, 0xa6, 0x8d, 0x33, 0xce, 0x78, 0xf8,
    0x47, 0x05, 0x1d, 0x98, 0xaa, 0x1b, 0x1f, 0x50, 0x05, 0x5b, 0x3c, 0x19,
    0x3f, 0x80, 0x83, 0x63, 0x63, 0x3a, 0xec, 0xcb, 0x2e, 0x90, 0x4f, 0xf5,
    0x26, 0x76, 0xf1, 0xd5, 0x02, 0x03, 0x01, 0x00, 0x01, 0x02, 0x40, 0x64,
    0x29, 0xc2, 0xd9, 0x6b, 0xfe, 0xf9, 0x84, 0x75, 0x73, 0xe0, 0xf4, 0x77,
    0xb5, 0x96, 0xb0, 0xdf, 0x83, 0xc0, 0x4e, 0x57, 0xf1, 0x10, 0x6e, 0x91,
    0x89, 0x12, 0x30, 0x5e, 0x57, 0xff, 0x14, 0x59, 0x5f, 0x18, 0x86, 0x4e,
    0x4b, 0x17, 0x56, 0xfc, 0x8d, 0x40, 0xdd, 0x74, 0x65, 0xd3, 0xff, 0x67,
    0x64, 0xcb, 0x9c, 0xb4, 0x14, 0x8a, 0x06, 0xb7, 0x13, 0x45, 0x94, 0x16,
    0x7d, 0x3f, 0xe1, 0x02, 0x21, 0x00, 0xf6, 0x0f, 0x31, 0x6d, 0x06, 0xcc,
    0x3b, 0xa0, 0x44, 0x1f, 0xf5, 0xc2, 0x45, 0x2b, 0x10, 0x6c, 0xf9, 0x6f,
    0x8f, 0x87, 0x3d, 0xc0, 0x3b, 0x55, 0x13, 0x37, 0x80, 0xcd, 0x9f, 0xe1,
    0xb7, 0xd9, 0x02, 0x21, 0x00, 0xe2, 0x9a, 0x5f, 0xbf, 0x95, 0x74, 0xb5,
    0x7a, 0x6a, 0xa6, 0x97, 0xbd, 0x75, 0x8c, 0x97, 0x18, 0x24, 0xd6, 0x09,
    0xcd, 0xdc, 0xb5, 0x94, 0xbf, 0xe2, 0x78, 0xaa, 0x20, 0x47, 0x9f, 0x68,
    0x5d, 0x02, 0x21, 0x00, 0xaf, 0x8f, 0x97, 0x8c, 0x5a, 0xd5, 0x4d, 0x95,
    0xc4, 0x05, 0xa9, 0xab, 0xba, 0xfe, 0x46, 0xf1, 0xf9, 0xe7, 0x07, 0x59,
    0x4f, 0x4d, 0xe1, 0x07, 0x8a, 0x76, 0x87, 0x88, 0x2f, 0x13, 0x35, 0xc1,
    0x02, 0x20, 0x24, 0xc3, 0xd9, 0x2f, 0x13, 0x47, 0x99, 0x3e, 0x20, 0x59,
    0xa1, 0x1a, 0xeb, 0x1c, 0x81, 0x53, 0x38, 0x7e, 0xc5, 0x9e, 0x71, 0xe5,
    0xc0, 0x19, 0x95, 0xdb, 0xef, 0xf6, 0x46, 0xc8, 0x95, 0x3d, 0x02, 0x21,
    0x00, 0xaa, 0xb1, 0xff, 0x8a, 0xa2, 0xb2, 0x2b, 0xef, 0x9a, 0x83, 0x3f,
    0xc5, 0xbc, 0xd4, 0x6a, 0x07, 0xe8, 0xc7, 0x0b, 0x2e, 0xd4, 0x0f, 0xf8,
    0x98, 0x68, 0xe1, 0x04, 0xa8, 0x92, 0xd0, 0x10, 0xaa,
};

// New signing key test data in DER-encoded PKCS8 format.
const uint8 kNewSigningKey[] = {
    0x30, 0x82, 0x01, 0x54, 0x02, 0x01, 0x00, 0x30, 0x0d, 0x06, 0x09, 0x2a,
    0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x82,
    0x01, 0x3e, 0x30, 0x82, 0x01, 0x3a, 0x02, 0x01, 0x00, 0x02, 0x41, 0x00,
    0x99, 0x98, 0x6b, 0x79, 0x5d, 0x38, 0x33, 0x79, 0x27, 0x0a, 0x2e, 0xb0,
    0x89, 0xba, 0xf8, 0xf6, 0x80, 0xde, 0xb0, 0x79, 0xf2, 0xd4, 0x6d, 0xf7,
    0x3c, 0xa3, 0x97, 0xf6, 0x4a, 0x3c, 0xa5, 0xcc, 0x40, 0x8a, 0xef, 0x59,
    0xaa, 0xc2, 0x82, 0x8f, 0xbc, 0x0d, 0x5b, 0x63, 0xc6, 0xaa, 0x72, 0xe2,
    0xf3, 0x57, 0xdd, 0x74, 0x00, 0xb0, 0x42, 0xd6, 0x27, 0xe7, 0x17, 0x61,
    0x0a, 0xdc, 0xc1, 0xf7, 0x02, 0x03, 0x01, 0x00, 0x01, 0x02, 0x40, 0x34,
    0xcf, 0xc9, 0xb4, 0x73, 0x2f, 0x0d, 0xd3, 0xcc, 0x6e, 0x9d, 0xdb, 0x29,
    0xa0, 0x56, 0x56, 0x3b, 0xbd, 0x56, 0x24, 0xb8, 0x2f, 0xfe, 0x97, 0x92,
    0x0c, 0x16, 0x06, 0x23, 0x44, 0x73, 0x25, 0x1d, 0x65, 0xf4, 0xda, 0x77,
    0xe7, 0x91, 0x2e, 0x91, 0x05, 0x10, 0xc1, 0x1b, 0x39, 0x5e, 0xb2, 0xf7,
    0xbd, 0x14, 0x19, 0xcb, 0x6b, 0xc3, 0xa9, 0xe8, 0x91, 0xf7, 0xa7, 0xa9,
    0x90, 0x08, 0x51, 0x02, 0x21, 0x00, 0xcc, 0x9e, 0x03, 0x54, 0x8f, 0x24,
    0xde, 0x90, 0x25, 0xec, 0x21, 0xaf, 0xe6, 0x27, 0x2a, 0x16, 0x42, 0x74,
    0xda, 0xf8, 0x84, 0xc4, 0x8c, 0x1e, 0x86, 0x12, 0x04, 0x5c, 0x17, 0x01,
    0xea, 0x9d, 0x02, 0x21, 0x00, 0xc0, 0x2a, 0x6c, 0xe9, 0xa1, 0x1a, 0x41,
    0x11, 0x94, 0x50, 0xf7, 0x1a, 0xd3, 0xbc, 0xf3, 0xa2, 0xf8, 0x46, 0xbc,
    0x26, 0x77, 0x78, 0xef, 0xc0, 0x54, 0xec, 0x22, 0x3f, 0x2c, 0x57, 0xe0,
    0xa3, 0x02, 0x20, 0x31, 0xf2, 0xc8, 0xa1, 0x55, 0xa8, 0x0c, 0x64, 0x67,
    0xbd, 0x72, 0xa3, 0xbb, 0xad, 0x07, 0xcb, 0x13, 0x41, 0xef, 0x4a, 0x07,
    0x2e, 0xeb, 0x7d, 0x70, 0x00, 0xe9, 0xeb, 0x88, 0xfa, 0x40, 0xc9, 0x02,
    0x20, 0x3a, 0xe0, 0xc4, 0xde, 0x10, 0x6e, 0x6a, 0xe1, 0x68, 0x00, 0x26,
    0xb6, 0x21, 0x8a, 0x13, 0x5c, 0x2b, 0x96, 0x00, 0xb0, 0x08, 0x8b, 0x15,
    0x6a, 0x68, 0x9a, 0xb1, 0x23, 0x8a, 0x02, 0xa2, 0xe1, 0x02, 0x21, 0x00,
    0xa3, 0xf2, 0x2d, 0x55, 0xc1, 0x6d, 0x40, 0xfa, 0x1d, 0xf7, 0xba, 0x86,
    0xef, 0x50, 0x98, 0xfc, 0xee, 0x09, 0xcc, 0xe7, 0x22, 0xb9, 0x4e, 0x80,
    0x32, 0x1a, 0x6b, 0xb3, 0x5f, 0x35, 0xbd, 0xf3,
};

}  // namespace

// Constants used as dummy data for filling the PolicyData protobuf.
const char PolicyBuilder::kFakeDeviceId[] = "device-id";
const char PolicyBuilder::kFakeDomain[] = "example.com";
const char PolicyBuilder::kFakeMachineName[] = "machine-name";
const char PolicyBuilder::kFakePolicyType[] = "policy type";
const int PolicyBuilder::kFakePublicKeyVersion = 17;
const int64 PolicyBuilder::kFakeTimestamp = 365LL * 24 * 60 * 60 * 1000;
const char PolicyBuilder::kFakeToken[] = "token";
const char PolicyBuilder::kFakeUsername[] = "username@example.com";
const char PolicyBuilder::kFakeServiceAccountIdentity[] = "robot4test@g.com";

PolicyBuilder::PolicyBuilder()
    : policy_data_(new em::PolicyData()),
      signing_key_(CreateTestSigningKey()) {
  policy_data_->set_policy_type(kFakePolicyType);
  policy_data_->set_timestamp(kFakeTimestamp);
  policy_data_->set_request_token(kFakeToken);
  policy_data_->set_machine_name(kFakeMachineName);
  policy_data_->set_public_key_version(kFakePublicKeyVersion);
  policy_data_->set_username(kFakeUsername);
  policy_data_->set_device_id(kFakeDeviceId);
  policy_data_->set_state(em::PolicyData::ACTIVE);
  policy_data_->set_service_account_identity(kFakeServiceAccountIdentity);
}

PolicyBuilder::~PolicyBuilder() {}

void PolicyBuilder::Build() {
  if (policy_data_.get())
    CHECK(policy_data_->SerializeToString(policy_.mutable_policy_data()));

  // Generate signatures if applicable.
  crypto::RSAPrivateKey* policy_signing_key = signing_key_.get();
  if (new_signing_key_.get()) {
    policy_signing_key = new_signing_key_.get();

    // Add the new public key.
    std::vector<uint8> raw_new_public_signing_key;
    CHECK(new_signing_key_->ExportPublicKey(&raw_new_public_signing_key));
    policy_.set_new_public_key(vector_as_array(&raw_new_public_signing_key),
                               raw_new_public_signing_key.size());

    // New public key signature.
    if (signing_key_.get()) {
      SignData(policy_.new_public_key(), signing_key_.get(),
               policy_.mutable_new_public_key_signature());
    }
  }

  // PolicyData signature.
  if (policy_signing_key) {
    SignData(policy_.policy_data(), policy_signing_key,
             policy_.mutable_policy_data_signature());
  }
}

std::string PolicyBuilder::GetBlob() {
  return policy_.SerializeAsString();
}

scoped_ptr<em::PolicyFetchResponse> PolicyBuilder::GetCopy() {
  scoped_ptr<em::PolicyFetchResponse> result(new em::PolicyFetchResponse());
  result->CopyFrom(policy_);
  return result.Pass();
}

// static
scoped_ptr<crypto::RSAPrivateKey> PolicyBuilder::CreateTestSigningKey() {
  std::vector<uint8> raw_signing_key(
      kSigningKey, kSigningKey + arraysize(kSigningKey));
  return scoped_ptr<crypto::RSAPrivateKey>(
      crypto::RSAPrivateKey::CreateFromPrivateKeyInfo(raw_signing_key));
}

// static
scoped_ptr<crypto::RSAPrivateKey> PolicyBuilder::CreateTestNewSigningKey() {
  std::vector<uint8> raw_new_signing_key(
      kNewSigningKey, kNewSigningKey + arraysize(kNewSigningKey));
  return scoped_ptr<crypto::RSAPrivateKey>(
      crypto::RSAPrivateKey::CreateFromPrivateKeyInfo(raw_new_signing_key));
}

void PolicyBuilder::SignData(const std::string& data,
                             crypto::RSAPrivateKey* key,
                             std::string* signature) {
  scoped_ptr<crypto::SignatureCreator> signature_creator(
      crypto::SignatureCreator::Create(key));
  signature_creator->Update(reinterpret_cast<const uint8*>(data.c_str()),
                            data.size());
  std::vector<uint8> signature_bytes;
  CHECK(signature_creator->Final(&signature_bytes));
  signature->assign(
      reinterpret_cast<const char*>(vector_as_array(&signature_bytes)),
      signature_bytes.size());
}

template<>
TypedPolicyBuilder<em::CloudPolicySettings>::TypedPolicyBuilder()
    : payload_(new em::CloudPolicySettings()) {
  policy_data().set_policy_type(dm_protocol::kChromeUserPolicyType);
}

template<>
TypedPolicyBuilder<em::ExternalPolicyData>::TypedPolicyBuilder()
    : payload_(new em::ExternalPolicyData()) {
  policy_data().set_policy_type(dm_protocol::kChromeExtensionPolicyType);
}


// Have the instantiations compiled into the module.
template class TypedPolicyBuilder<em::CloudPolicySettings>;
template class TypedPolicyBuilder<em::ExternalPolicyData>;

}  // namespace policy
