// Copyright (c) 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_CHROMEOS_NET_CONNECTIVITY_STATE_HELPER_H_
#define CHROME_BROWSER_CHROMEOS_NET_CONNECTIVITY_STATE_HELPER_H_

#include "base/observer_list.h"
#include "chrome/browser/chromeos/cros/network_library.h"
#include "chrome/browser/chromeos/net/connectivity_state_helper_observer.h"
#include "chromeos/network/network_state_handler.h"

namespace chromeos {

// This class provides an interface for consumers to query the connectivity
// state on Chrome OS. Subclasses must implement the query methods using an
// appropriate source (e.g. NetworkStateHandler).
class ConnectivityStateHelper {
 public:
  virtual ~ConnectivityStateHelper();

  // Initializes the state helper singleton to use the default (network state
  // handler) implementation or the network library implementation based
  // on the value of command line flag.
  static void Initialize();
  static bool IsInitialized();
  static void Shutdown();
  static ConnectivityStateHelper* Get();

  // Sets up Get() to return |impl| for testing (e.g. with a mock
  // implementation). Call SetForTest(NUL) when |impl| is deleted.
  static void SetForTest(ConnectivityStateHelper* impl);

  // Returns true if in a connected state.
  virtual bool IsConnected() = 0;

  // Returns true if in a connecting state.
  virtual bool IsConnecting() = 0;

  // Returns true if there's a network of |type| in connected state.
  virtual bool IsConnectedType(const std::string& type) = 0;

  // Returns true if there's a network of |type| in connecting state.
  virtual bool IsConnectingType(const std::string& type) = 0;

  // Get the name for the primary network of type |type| which is not
  // in non-idle state (i.e. connecting or connected state).
  virtual std::string NetworkNameForType(const std::string& type) = 0;

  // Returns the name of the default network.
  virtual std::string DefaultNetworkName() = 0;

  // Returns true if we have a default network and are in online state.
  virtual bool DefaultNetworkOnline() = 0;

  // Request a network scan.
  virtual void RequestScan() const = 0;

  // Add/remove observers for listening to connection manager changes.
  virtual void AddNetworkManagerObserver(
      ConnectivityStateHelperObserver* observer);
  virtual void RemoveNetworkManagerObserver(
      ConnectivityStateHelperObserver* observer);

 protected:
  ConnectivityStateHelper();
  ObserverList<ConnectivityStateHelperObserver> connectivity_observers_;
};

}  // namespace chromeos

#endif  // CHROME_BROWSER_CHROMEOS_NET_CONNECTIVITY_STATE_HELPER_H_
