// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/chromeos/drive/file_write_helper.h"

#include "base/threading/sequenced_worker_pool.h"
#include "content/public/browser/browser_thread.h"

using content::BrowserThread;

namespace drive {

namespace {

// Emits debug log when FileSystem::CloseFile() is complete.
void EmitDebugLogForCloseFile(const base::FilePath& file_path,
                              FileError file_error) {
  if (file_error != FILE_ERROR_OK) {
    LOG(WARNING) << "CloseFile failed: " << file_path.AsUTF8Unsafe() << ": "
                 << file_error;
  }
}

}  // namespace

FileWriteHelper::FileWriteHelper(FileSystemInterface* file_system)
    : file_system_(file_system),
      weak_ptr_factory_(this) {
  // Must be created in DriveIntegrationService.
  DCHECK(BrowserThread::CurrentlyOn(BrowserThread::UI));
}

FileWriteHelper::~FileWriteHelper() {
  // Must be destroyed in DriveIntegrationService.
  DCHECK(BrowserThread::CurrentlyOn(BrowserThread::UI));
}

void FileWriteHelper::PrepareWritableFileAndRun(
    const base::FilePath& file_path,
    const OpenFileCallback& callback) {
  DCHECK(BrowserThread::CurrentlyOn(BrowserThread::UI));
  DCHECK(!callback.is_null());

  file_system_->CreateFile(
      file_path,
      false,  // it is not an error, even if the path already exists.
      base::Bind(&FileWriteHelper::PrepareWritableFileAndRunAfterCreateFile,
                 weak_ptr_factory_.GetWeakPtr(),
                 file_path,
                 callback));
}

void FileWriteHelper::PrepareWritableFileAndRunAfterCreateFile(
    const base::FilePath& file_path,
    const OpenFileCallback& callback,
    FileError error) {
  DCHECK(BrowserThread::CurrentlyOn(BrowserThread::UI));
  DCHECK(!callback.is_null());

  if (error != FILE_ERROR_OK) {
    content::BrowserThread::GetBlockingPool()->PostTask(
        FROM_HERE,
        base::Bind(callback, error, base::FilePath()));
    return;
  }
  file_system_->OpenFile(
      file_path,
      base::Bind(&FileWriteHelper::PrepareWritableFileAndRunAfterOpenFile,
                 weak_ptr_factory_.GetWeakPtr(),
                 file_path,
                 callback));
}

void FileWriteHelper::PrepareWritableFileAndRunAfterOpenFile(
    const base::FilePath& file_path,
    const OpenFileCallback& callback,
    FileError error,
    const base::FilePath& local_cache_path) {
  DCHECK(BrowserThread::CurrentlyOn(BrowserThread::UI));
  DCHECK(!callback.is_null());

  if (error != FILE_ERROR_OK) {
    content::BrowserThread::GetBlockingPool()->PostTask(
        FROM_HERE,
        base::Bind(callback, error, base::FilePath()));
    return;
  }

  content::BrowserThread::GetBlockingPool()->PostTaskAndReply(
      FROM_HERE,
      base::Bind(callback, FILE_ERROR_OK, local_cache_path),
      base::Bind(&FileWriteHelper::PrepareWritableFileAndRunAfterCallback,
                 weak_ptr_factory_.GetWeakPtr(),
                 file_path));
}

void FileWriteHelper::PrepareWritableFileAndRunAfterCallback(
    const base::FilePath& file_path) {
  DCHECK(BrowserThread::CurrentlyOn(BrowserThread::UI));
  file_system_->CloseFile(file_path,
                          base::Bind(&EmitDebugLogForCloseFile, file_path));
}

}  // namespace drive
