// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_CHROMEOS_AUDIO_AUDIO_HANDLER_H_
#define CHROME_BROWSER_CHROMEOS_AUDIO_AUDIO_HANDLER_H_

#include "base/basictypes.h"
#include "base/memory/ref_counted.h"
#include "base/observer_list.h"
#include "base/threading/thread.h"
#include "chromeos/audio/audio_pref_observer.h"

template <typename T> struct DefaultSingletonTraits;

class PrefChangeRegistrar;
class PrefRegistrySimple;
class PrefService;

namespace chromeos {

class AudioMixer;
class AudioPrefHandler;

// TODO(jennyz): crbug.com/233301.
// Retire the old AudioHandler and the old audio library code once
// the new audio dbus and handler code stabilizes.
class AudioHandler : public AudioPrefObserver {
 public:
  class VolumeObserver {
   public:
    virtual void OnVolumeChanged() = 0;
    virtual void OnMuteToggled() = 0;
   protected:
    VolumeObserver() {}
    virtual ~VolumeObserver() {}
    DISALLOW_COPY_AND_ASSIGN(VolumeObserver);
  };

  static void Initialize(scoped_refptr<AudioPrefHandler> audio_pref_handler);
  static void Shutdown();

  // Same as Initialize but using the specified audio mixer.  It takes
  // ownership of |mixer|.
  static void InitializeForTesting(
      AudioMixer* mixer,
      scoped_refptr<AudioPrefHandler> audio_pref_handler);

  // GetInstance returns NULL if not initialized or if already shutdown.
  static AudioHandler* GetInstance();

  // Gets volume level in our internal 0-100% range, 0 being pure silence.
  double GetVolumePercent();

  // Sets volume level from 0-100%. If less than kMuteThresholdPercent, then
  // mutes the sound. If it was muted, and |volume_percent| is larger than
  // the threshold, then the sound is unmuted.
  void SetVolumePercent(double volume_percent);

  // Adjusts volume up (positive percentage) or down (negative percentage).
  void AdjustVolumeByPercent(double adjust_by_percent);

  // Is the volume currently muted?
  bool IsMuted();

  // Mutes or unmutes all audio.
  void SetMuted(bool do_mute);

  // Is the capture volume currently muted?
  bool IsCaptureMuted();

  // Mutes or unmutes all capture devices. If unmutes and the volume was set
  // to 0, then increases volume to a minimum value (5%).
  void SetCaptureMuted(bool do_mute);

  void AddVolumeObserver(VolumeObserver* observer);
  void RemoveVolumeObserver(VolumeObserver* observer);

 private:
  // Defines the delete on exit Singleton traits we like.  Best to have this
  // and constructor/destructor private as recommended for Singletons.
  friend struct DefaultSingletonTraits<AudioHandler>;

  // Takes ownership of |mixer|.
  explicit AudioHandler(AudioMixer* mixer,
                        scoped_refptr<AudioPrefHandler> audio_pref_handler);
  virtual ~AudioHandler();

  // Applies the audio muting policies whenever the user logs in or policy
  // change notification is received.
  void ApplyAudioPolicy();

  // Sets volume/muted to specified value and notifies observers if |notify|
  // is true.
  void SetVolumePercentInternal(double volume_percent, bool notify);
  void SetMutedInternal(bool do_mute, bool notify);

  // Overriden from AudioPrefObserver.
  virtual void OnAudioPolicyPrefChanged() OVERRIDE;

  scoped_ptr<AudioMixer> mixer_;

  ObserverList<VolumeObserver> volume_observers_;

  // Track state for triggering callbacks in ApplyAudioPolicy().
  bool muted_;

  scoped_refptr<AudioPrefHandler> audio_pref_handler_;

  DISALLOW_COPY_AND_ASSIGN(AudioHandler);
};

}  // namespace chromeos

#endif  // CHROME_BROWSER_CHROMEOS_AUDIO_AUDIO_HANDLER_H_
