%{
#include <sepol/policydb.h>
#include <sepol/services.h>
#include <sepol/conditional.h>
#include <stdint.h>
#include "queue.h"
#include "checkpolicy.h"

#define TRUE 1
#define FALSE 0

policydb_t *policydbp;
queue_t id_queue = 0;
unsigned int pass;
char *curfile = 0;
unsigned int curline; 

extern unsigned long policydb_lineno;

extern char yytext[];
extern int yywarn(char *msg);
extern int yyerror(char *msg);

static char errormsg[255];

static int insert_separator(int push);
static int insert_id(char *id,int push);
static int define_user(void);
%}

%token ROLES
%token RANGES
%token USER
%token IDENTIFIER
%token USER_IDENTIFIER

%%
users			: user_def
			| users user_def
			;
user_id			: identifier
			| user_identifier
			;
user_def		: USER user_id ROLES names opt_user_ranges ';'
	                {if (define_user()) return -1;}
			;
opt_user_ranges		: RANGES user_ranges 
			|
			;
user_ranges		: mls_range_def
			| '{' user_range_def_list '}' 
			;
user_range_def_list	: mls_range_def
			| user_range_def_list mls_range_def
			;
mls_range_def		: mls_level_def '-' mls_level_def
			{if (insert_separator(0)) return -1;}
	                | mls_level_def
			{if (insert_separator(0)) return -1;}
	                ;
mls_level_def		: identifier ':' id_comma_list
			{if (insert_separator(0)) return -1;}
	                | identifier 
			{if (insert_separator(0)) return -1;}
	                ;
id_comma_list           : identifier
			| id_comma_list ',' identifier
			;
tilde			: '~'
			;
asterisk		: '*'
			;
names           	: identifier
			{ if (insert_separator(0)) return -1; }
			| nested_id_set
			{ if (insert_separator(0)) return -1; }
			| asterisk
                        { if (insert_id("*", 0)) return -1; 
			  if (insert_separator(0)) return -1; }
			| tilde identifier
                        { if (insert_id("~", 0)) return -1;
			  if (insert_separator(0)) return -1; }
			| tilde nested_id_set
	 		{ if (insert_id("~", 0)) return -1; 
			  if (insert_separator(0)) return -1; }
                        | identifier '-' { if (insert_id("-", 0)) return -1; } identifier 
			{ if (insert_separator(0)) return -1; }
			;
nested_id_set           : '{' nested_id_list '}'
                        ;
nested_id_list          : nested_id_element | nested_id_list nested_id_element
                        ;
nested_id_element       : identifier | '-' { if (insert_id("-", 0)) return -1; } identifier | nested_id_set
                        ;
identifier		: IDENTIFIER
			{ if (insert_id(yytext,0)) return -1; }
			;
user_identifier		: USER_IDENTIFIER
			{ if (insert_id(yytext,0)) return -1; }
			;

%%
#define DEBUG 1

static int insert_separator(int push)
{
	int error;

	if (push)
		error = queue_push(id_queue, 0);
	else
		error = queue_insert(id_queue, 0);

	if (error) {
		yyerror("queue overflow");
		return -1;
	}
	return 0;
}

static int insert_id(char *id, int push)
{
	char *newid = 0;
	int error;

	newid = (char *) malloc(strlen(id) + 1);
	if (!newid) {
		yyerror("out of memory");
		return -1;
	}
	strcpy(newid, id);
	if (push)
		error = queue_push(id_queue, (queue_element_t) newid);
	else
		error = queue_insert(id_queue, (queue_element_t) newid);

	if (error) {
		yyerror("queue overflow");
		free(newid);
		return -1;
	}
	return 0;
}

static int set_user_roles(ebitmap_t *set,
			  char *id)
{
	role_datum_t *r;
	unsigned int i;

	if (strcmp(id, "*") == 0) {
		/* set all roles */
		for (i = 0; i < policydbp->p_roles.nprim; i++) 
			ebitmap_set_bit(set, i, TRUE);
		free(id);
		return 0;
	}

	if (strcmp(id, "~") == 0) {
		/* complement the set */
		for (i = 0; i < policydbp->p_roles.nprim; i++) {
			if (ebitmap_get_bit(set, i))
				ebitmap_set_bit(set, i, FALSE);
			else 
				ebitmap_set_bit(set, i, TRUE);
		}
		free(id);
		return 0;
	}

	r = hashtab_search(policydbp->p_roles.table, id);
	if (!r) {
		sprintf(errormsg, "unknown role %s", id);
		yyerror(errormsg);
		free(id);
		return -1;
	}

	/* set the role and every role it dominates */
	for (i = ebitmap_startbit(&r->dominates); i < ebitmap_length(&r->dominates); i++) {
		if (ebitmap_get_bit(&r->dominates, i))
			ebitmap_set_bit(set, i, TRUE);
	}
	free(id);
	return 0;
}

static int define_user(void)
{
	char *id;
	user_datum_t *usrdatum;
	int ret;
#ifdef CONFIG_SECURITY_SELINUX_MLS
	mls_range_list_t *rnode;
	level_datum_t *levdatum;
	cat_datum_t *catdatum;
	int relation, l;
	char *levid;
#endif

	if (pass == 1) {
		while ((id = queue_remove(id_queue))) 
			free(id);
#ifdef CONFIG_SECURITY_SELINUX_MLS
		while ((id = queue_remove(id_queue))) { 
			free(id);
			for (l = 0; l < 2; l++) {
				while ((id = queue_remove(id_queue))) { 
					free(id);
				}
			}
		}
#endif
		return 0;
	}

	id = (char *) queue_remove(id_queue);
	if (!id) {
		yyerror("no user name for user definition?");
		return -1;
	}
	usrdatum = (user_datum_t *) hashtab_search(policydbp->p_users.table,
						   (hashtab_key_t) id);
	if (usrdatum) {
	        printf("Replacing user %s\n", id);
		ebitmap_init(&usrdatum->roles);
		usrdatum->defined = 1;
		free(id);
	} else {
	        printf("Adding user %s\n", id);
		usrdatum = (user_datum_t *) malloc(sizeof(user_datum_t));
		if (!usrdatum) {
			yyerror("out of memory");
			free(id);
			return -1;
		}
		memset(usrdatum, 0, sizeof(user_datum_t));
		usrdatum->value = ++policydbp->p_users.nprim;
		ebitmap_init(&usrdatum->roles);
		usrdatum->defined = 1;
		ret = hashtab_insert(policydbp->p_users.table,
				     (hashtab_key_t) id, (hashtab_datum_t) usrdatum);
		if (ret) {
			yyerror("hash table overflow");
			free(usrdatum);
			free(id);
			return -1;
		}
	}

	while ((id = queue_remove(id_queue))) {
		if (set_user_roles(&usrdatum->roles, id))
			continue;
	}

#ifdef CONFIG_SECURITY_SELINUX_MLS
	id = queue_remove(id_queue);
	if (!id) {
		rnode = (mls_range_list_t *) malloc(sizeof(mls_range_list_t));
		if (!rnode) {
			yyerror("out of memory");
			free(id);
			return -1;
		}
		memset(rnode, 0, sizeof(mls_range_list_t));
		levdatum = (level_datum_t *) hashtab_search(policydbp->p_levels.table,
							    (hashtab_key_t) "unclassified");
		if (!levdatum) {
			yyerror("no range for user");
			return -1;
		}
		rnode->range.level[0].sens = levdatum->level->sens;
		rnode->range.level[1].sens = levdatum->level->sens;
		rnode->next = usrdatum->ranges;
		usrdatum->ranges = rnode;
		goto skip_mls;
	} 
	do {
		rnode = (mls_range_list_t *) malloc(sizeof(mls_range_list_t));
		if (!rnode) {
			yyerror("out of memory");
			free(id);
			return -1;
		}
		memset(rnode, 0, sizeof(mls_range_list_t));

		for (l = 0; l < 2; l++) {
			levdatum = (level_datum_t *) hashtab_search(policydbp->p_levels.table,
						     (hashtab_key_t) id);
			if (!levdatum) {
				sprintf(errormsg, "unknown sensitivity %s used in user range definition", id);
				yyerror(errormsg);
				free(rnode);
				free(id);
				continue;
			}
			rnode->range.level[l].sens = levdatum->level->sens;
			ebitmap_init(&rnode->range.level[l].cat);

			levid = id;

			while ((id = queue_remove(id_queue))) {
				catdatum = (cat_datum_t *) hashtab_search(policydbp->p_cats.table,
						     (hashtab_key_t) id);
				if (!catdatum) {
					sprintf(errormsg, "unknown category %s used in user range definition", id);
					yyerror(errormsg);
					free(id);
					continue;
				}
				if (!(ebitmap_get_bit(&levdatum->level->cat, catdatum->value - 1))) {
					sprintf(errormsg, "category %s cannot be associated with level %s", id, levid);
					yyerror(errormsg);
					free(id);
					continue;
				}
				if (ebitmap_set_bit(&rnode->range.level[l].cat, catdatum->value - 1, TRUE)) {
					yyerror("out of memory");
					free(id);
					free(levid);
					ebitmap_destroy(&rnode->range.level[l].cat);
					free(rnode);
					return -1;
				}

				/*
				 * no need to keep category name
				 */
				free(id);
			}

			/*
			 * no need to keep sensitivity name
			 */
			free(levid);

			id = queue_remove(id_queue);
			if (!id)
				break;
		}

		if (l == 0) {
			rnode->range.level[1].sens = rnode->range.level[0].sens;
			if (ebitmap_cpy(&rnode->range.level[1].cat, &rnode->range.level[0].cat)) {
				yyerror("out of memory");
				free(id);
				ebitmap_destroy(&rnode->range.level[0].cat);
				free(rnode);
				return -1;
			}
		}
		relation = mls_level_relation(rnode->range.level[1], rnode->range.level[0]);
		if (!(relation & (MLS_RELATION_DOM | MLS_RELATION_EQ))) {
			/* high does not dominate low */
			yyerror("high does not dominate low");
			ebitmap_destroy(&rnode->range.level[0].cat);
			ebitmap_destroy(&rnode->range.level[1].cat);
			free(rnode);
			return -1;
		}
		rnode->next = usrdatum->ranges;
		usrdatum->ranges = rnode;
	} while ((id = queue_remove(id_queue)));
skip_mls:
#endif

	return 0;
}
