/* Authors: Karl MacMillan <kmacmillan@tresys.com>
 *          Frank Mayer <mayerf@tresys.com>
 *          David Caplan <dac@tresys.com>
 *
 * Copyright (C) 2003 - 2004 Tresys Technology, LLC
 *	This program is free software; you can redistribute it and/or modify
 *  	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, version 2.
 */

#include "include/security.h"
#include "conditional.h"


/* move all type rules to top of t/f lists to help kernel on evaluation */
static void cond_optimize(cond_av_list_t **l) {
	cond_av_list_t *top, *p,*cur;

	top = p = cur = *l;

	while(cur) {
		if ((cur->node->datum.specified & AVTAB_TYPE) && (top != cur)) {
			p->next = cur->next;
			cur->next = top;
			top = cur;
			cur = p->next;
		} else {
			p = cur;
			cur = cur->next; 
		}
	}
	*l = top;
}

/* reorder t/f lists for kernel */
void cond_optimize_lists(cond_list_t* cl) {
	cond_list_t *n;

	for (n = cl; n != NULL; n = n->next) {
		cond_optimize(&n->true_list);
		cond_optimize(&n->false_list);
	}
}


static int bool_present(int target, int bools[], int num_bools) {
	int i = 0;
	int ret = 1;

	if (num_bools > COND_MAX_BOOLS) {
		return 0;
	}
	while( i < num_bools && target != bools[i]) i++;
	if (i == num_bools) ret = 0; /* got to end w/o match */
	return ret;
}

static int same_bools(cond_node_t *a, cond_node_t *b) {
	int i, x;
	
	x = a->nbools;

	/* same number of bools? */
	if (x != b->nbools) return 0;

	/* make sure all the bools in a are also in b */
	for(i = 0; i < x; i++) if (!bool_present(a->bool_ids[i], b->bool_ids, x)) return 0;
	return 1;
}

/*
 * Determine if two conditional expressions are equal. 
 */
static int cond_expr_equal(cond_node_t *a, cond_node_t *b)
{
	cond_expr_t *cur_a, *cur_b;
		
	if (a == NULL || b == NULL)
		return 0;

	if(a->nbools != b->nbools) return 0;

	/* if exprs have <= COND_MAX_BOOLS we can check the precompute values
	 * for the expressions.
	 */
	if (a->nbools <= COND_MAX_BOOLS && b->nbools <= COND_MAX_BOOLS)	    
	{
		if(!same_bools(a,b)) return 0;
		return (a->expr_pre_comp == b->expr_pre_comp);
	}

	/* for long expressions we check for exactly the same expression */
	cur_a = a->expr;
	cur_b = b->expr;
	while (1) {
		if (cur_a == NULL && cur_b == NULL)
			return 1;
		else if (cur_a == NULL || cur_b == NULL)
			return 0;
		if (cur_a->expr_type != cur_b->expr_type)
			return 0;
		if (cur_a->expr_type == COND_BOOL) {
			if (cur_a->bool != cur_b->bool)
				return 0;
		}
		cur_a = cur_a->next;
		cur_b = cur_b->next;
	}
	return 1;
}

/* return either a pre-existing matching node or create a new node */
cond_node_t *cond_node_search(policydb_t *p, cond_node_t *cn)
{
	cond_node_t *list,*new;
	int i;

	list = p->cond_list;

	while(list) {
		if (cond_expr_equal(cn,list)) break;
		list = list->next;
	}
	if (list) return list;
	else {
		new = (cond_node_t *)malloc(sizeof (cond_node_t));
		if (!new) {
			return NULL;
		}
		new->expr = cn->expr;
		new->cur_state = cond_evaluate_expr(p, new->expr);
		new->nbools = cn->nbools;
		for (i = 0; i < cn->nbools; i++)
			new->bool_ids[i] = cn->bool_ids[i];
		new->expr_pre_comp = cn->expr_pre_comp;
		new->true_list = NULL;
		new->false_list = NULL;
		/* add conditional node to policy list */
		new->next = p->cond_list;
		p->cond_list = new;
		return new;
	}
}

/*
 * cond_evaluate_expr evaluates a conditional expr
 * in reverse polish notation. It returns true (1), false (0),
 * or undefined (-1). Undefined occurs when the expression
 * exceeds the stack depth of COND_EXPR_MAXDEPTH.
 */
int cond_evaluate_expr(policydb_t *p, cond_expr_t *expr)
{

	cond_expr_t *cur;
	int s[COND_EXPR_MAXDEPTH];
	int sp = -1;

	for (cur = expr; cur != NULL; cur = cur->next) {
		switch (cur->expr_type) {
		case COND_BOOL:
			if (sp == (COND_EXPR_MAXDEPTH - 1))
				return -1;
			sp++;
			s[sp] = p->bool_val_to_struct[cur->bool - 1]->state;
			break;
		case COND_NOT:
			if (sp < 0)
				return -1;
			s[sp] = !s[sp];
			break;
		case COND_OR:
			if (sp < 1)
				return -1;
			sp--;
			s[sp] |= s[sp + 1];
			break;
		case COND_AND:
			if (sp < 1)
				return -1;
			sp--;
			s[sp] &= s[sp + 1];
			break;
		case COND_XOR:
			if (sp < 1)
				return -1;
			sp--;
			s[sp] ^= s[sp + 1];
			break;
		case COND_EQ:
			if (sp < 1)
				return -1;
			sp--;
			s[sp] = (s[sp] == s[sp + 1]);
			break;
		case COND_NEQ:
			if (sp < 1)
				return -1;
			sp--;
			s[sp] = (s[sp] != s[sp + 1]);
			break;
		default:
			return -1;
		}
	}
	return s[0];
}

/*
 * evaluate_cond_node evaluates the conditional stored in
 * a cond_node_t and if the result is different than the
 * current state of the node it sets the rules in the true/false
 * list appropriately. If the result of the expression is undefined
 * all of the rules are disabled for safety.
 */
static int evaluate_cond_node(policydb_t *p, cond_node_t *node)
{
	int new_state;
	cond_av_list_t* cur;

	new_state = cond_evaluate_expr(p, node->expr);
	if (new_state != node->cur_state) {
		node->cur_state = new_state;
		if (new_state == -1)
			printk("expression result was undefined - disabling all rules.\n");
		/* turn the rules on or off */
		for (cur = node->true_list; cur != NULL; cur = cur->next) {
			if (new_state <= 0) {
				cur->node->datum.specified &= ~AVTAB_ENABLED;
			} else {
				cur->node->datum.specified |= AVTAB_ENABLED;
			}
		}

		for (cur = node->false_list; cur != NULL; cur = cur->next) {	
			/* -1 or 1 */
			if (new_state) {
				cur->node->datum.specified &= ~AVTAB_ENABLED;
			} else {
				cur->node->datum.specified |= AVTAB_ENABLED;
			}
		}
	}
	return 0;
}

/* precompute and simplify an expression if possible.  If left with !expression, change 
 * to expression and switch t and f. precompute expression for expressions with limited
 * number of bools.
 */
int normalize_expr(policydb_t *p, cond_node_t *cn)
{
	cond_expr_t *ne,*e;
	cond_av_list_t *tmp;
	int i, j, k, orig_value[COND_MAX_BOOLS];
	__u32 test = 0x0;
	

	cn->nbools = 0;

	memset(cn->bool_ids, 0,sizeof(cn->bool_ids) );
	cn->expr_pre_comp = 0x0;

	/* take care of !expr case */
	ne = NULL;
	e = cn->expr;
        
        /* becuase it's RPN look at last element */
	while(e->next != NULL) { 
		ne = e; e = e->next;
	}
	if (e->expr_type == COND_NOT) {
		if (ne) {
			ne->next = NULL; 
		} else { /* ne should never be NULL */
			printk("Found expr with no bools and only a ! - this should never happen.\n");
			return -1;
		}
		/* swap the true and false lists */
		tmp = cn->true_list;
		cn->true_list = cn->false_list;
		cn->false_list = tmp;
		/* free the "not" node in the list */
		free(e);
	}

	/* find all the bools in the expression */
	for(e = cn->expr; e != NULL; e = e->next) {
		switch(e->expr_type) { 
		case COND_BOOL:
			i = 0;
			/* see if we've already seen this bool */
			if (!bool_present(e->bool, cn->bool_ids, cn->nbools)) {
				/* count em all but only record up to COND_MAX_BOOLS */
				if (cn->nbools < COND_MAX_BOOLS)
					cn->bool_ids[cn->nbools++] = e->bool;
				else
					cn->nbools++;
			}
			break;
		default:
			break;
		}
	} 
 
	/* only precompute for exprs with <= COND_AX_BOOLS */
	if (cn->nbools <=  COND_MAX_BOOLS) {
		/* save the default values for the bools so we can play with them */
		for(i = 0; i < cn->nbools; i++) {
			orig_value[i] = p->bool_val_to_struct[cn->bool_ids[i] - 1]->state;
		}
		
		/* loop through all possible combinations of values for bools in expression */
		for(test = 0x0; test < (0x1 << cn->nbools) ; test++) {
			/* temporarily set the value for all the bools in the
			 * expression using the corr.  bit in test */
			for(j = 0; j < cn->nbools; j++) {
				p->bool_val_to_struct[cn->bool_ids[j] - 1]->state =
					( test & (0x1 << j) ) ? 1 : 0;
			}
			k = cond_evaluate_expr(p, cn->expr);
			if (k == -1) {
				printk("While testing expression, expression result "
				       "was undefined - this should never happen.\n");
				return -1;
			}
			/* set the bit if expression evaluates true */
			if (k) cn->expr_pre_comp |= 0x1 << test;
		}
		
		/* restore bool default values */
		for(i = 0; i < cn->nbools; i++)
			p->bool_val_to_struct[cn->bool_ids[i] - 1]->state = orig_value[i] ;
	}
	return 0;
}

int evaluate_conds(policydb_t *p)
{
	int ret;
	cond_node_t *cur;

	for (cur = p->cond_list; cur != NULL; cur = cur->next) {
		ret = evaluate_cond_node(p, cur);
		if (ret)
			return ret;
	}
	return 0;
}

int cond_policydb_init(policydb_t *p)
{
	p->bool_val_to_struct = NULL;
	p->cond_list = NULL;
	if (avtab_init(&p->te_cond_avtab))
		return -1;

	return 0;
}

static void cond_av_list_destroy(cond_av_list_t *list)
{
	cond_av_list_t *cur, *next;
	for (cur = list; cur != NULL; cur = next) {
		next = cur->next;
		/* the avtab_ptr_t node is destroy by the avtab */
		kfree(cur);
	}
}

static void cond_node_destroy(cond_node_t *node)
{
	cond_expr_t *cur_expr, *next_expr;

	if (!node)
		return;

	for (cur_expr = node->expr; cur_expr != NULL; cur_expr = next_expr) {
		next_expr = cur_expr->next;
		kfree(cur_expr);
	}
	cond_av_list_destroy(node->true_list);
	cond_av_list_destroy(node->false_list);
	kfree(node);
}

static void cond_list_destroy(cond_list_t *list)
{
	cond_node_t *next, *cur;
	
	if (list == NULL)
		return;

	for (cur = list; cur != NULL; cur = next) {
		next = cur->next;
		cond_node_destroy(cur);
	}
}

void cond_policydb_destroy(policydb_t *p)
{
	if (p->bool_val_to_struct != NULL)
		kfree(p->bool_val_to_struct);
	avtab_destroy(&p->te_cond_avtab);
	cond_list_destroy(p->cond_list);
}

int cond_init_bool_indexes(policydb_t *p)
{
	if (p->bool_val_to_struct)
		kfree(p->bool_val_to_struct);
	p->bool_val_to_struct = (cond_bool_datum_t**)
		kmalloc(p->p_bools.nprim * sizeof(cond_bool_datum_t*), GFP_KERNEL);
	if (!p->bool_val_to_struct)
		return -1;
	return 0;
}

int cond_destroy_bool(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	if (key)
		kfree(key);
	kfree(datum);
	return 0;
}

int cond_index_bool(hashtab_key_t key, hashtab_datum_t datum, void *datap)
{
	policydb_t *p;
	cond_bool_datum_t *booldatum;

	booldatum = datum;
	p = datap;
	
	if (!booldatum->value || booldatum->value > p->p_bools.nprim)
		return -EINVAL;

	p->p_bool_val_to_name[booldatum->value - 1] = key;
	p->bool_val_to_struct[booldatum->value -1] = booldatum;

	return 0;
}

int bool_isvalid(cond_bool_datum_t *b)
{
	if (!(b->state == 0 || b->state == 1))
		return 0;
	return 1;
}

int cond_read_bool(policydb_t *p, hashtab_t h, struct policy_file *fp)
{
	char *key = 0;
	cond_bool_datum_t *booldatum;
	__u32 *buf, len;

	booldatum = kmalloc(sizeof(cond_bool_datum_t), GFP_KERNEL);
	if (!booldatum)
		return -1;
	memset(booldatum, 0, sizeof(cond_bool_datum_t));

	buf = next_entry(fp, sizeof(__u32) * 3);
	if (!buf)
		goto err;

	booldatum->value = le32_to_cpu(buf[0]);
	booldatum->state = le32_to_cpu(buf[1]);
	
	if (!bool_isvalid(booldatum))
		goto err;

	len = le32_to_cpu(buf[2]);
	
	buf = next_entry(fp, len);
	if (!buf)
		goto err;
	key = kmalloc(len + 1, GFP_KERNEL);
	if (!key)
		goto err;
	memcpy(key, buf, len);
	key[len] = 0;
	if (hashtab_insert(h, key, booldatum))
		goto err;
	
	return 0;
err:
	cond_destroy_bool(key, booldatum, 0);
	return -1;
}

static int cond_read_av_list(policydb_t *p, void *fp, cond_av_list_t **ret_list, cond_av_list_t *other)
{
	cond_av_list_t *list, *last = NULL, *cur;
	avtab_key_t key;
	avtab_datum_t datum;
	avtab_ptr_t node_ptr;
	int len, i;
	__u32 *buf;
	__u8 found;

	*ret_list = NULL;

	len = 0;
	buf = next_entry(fp, sizeof(__u32));
	if (!buf)
		return -1;

	len = le32_to_cpu(buf[0]);
	if (len == 0) {
		return 0;
	}

	for (i = 0; i < len; i++) {
		if (avtab_read_item(fp, &datum, &key))
			goto err;

		/*
		 * For type rules we have to make certain there aren't any
		 * conflicting rules by searching the te_avtab and the
		 * cond_te_avtab.
		 */
		if (datum.specified & AVTAB_TYPE) {
			if (avtab_search(&p->te_avtab, &key, AVTAB_TYPE)) {
				printk("security: type rule already exists outside of a conditional.");
				goto err;
			}
			/*
			 * If we are reading the false list other will be a pointer to
			 * the true list. We can have duplicate entries if there is only
			 * 1 other entry and it is in our true list.
			 *
			 * If we are reading the true list (other == NULL) there shouldn't
			 * be any other entries.
			 */
			if (other) {
				node_ptr = avtab_search_node(&p->te_cond_avtab, &key, AVTAB_TYPE);
				if (node_ptr) {
					if (avtab_search_node_next(node_ptr, AVTAB_TYPE)) {
						printk("security: too many conflicting type rules.");
						goto err;
					}
					found = 0;
					for (cur = other; cur != NULL; cur = cur->next) {
						if (cur->node == node_ptr) {
							found = 1;
							break;
						}
					}
					if (!found) {
						printk("security: conflicting type rules.\n");
						goto err;
					}
				}
			} else {    
				if (avtab_search(&p->te_cond_avtab, &key, AVTAB_TYPE)) {
					printk("security: conflicting type rules when adding type rule for true.\n");
					goto err;
				}
			}
		}
		node_ptr = avtab_insert_nonunique(&p->te_cond_avtab, &key, &datum);
		if (!node_ptr) {
			printk("security: could not insert rule.");
			goto err;
		}

		list = kmalloc(sizeof(cond_av_list_t), GFP_KERNEL);
		if (!list)
			goto err;
		memset(list, 0, sizeof(cond_av_list_t));

		list->node = node_ptr;
		if (i == 0)
			*ret_list = list;
		else
			last->next = list;
		last = list;

	}

	return 0;
err:
	cond_av_list_destroy(*ret_list);
	*ret_list = NULL;
	return -1;
}

static int expr_isvalid(policydb_t *p, cond_expr_t *expr)
{
	if (expr->expr_type <= 0 || expr->expr_type > COND_LAST) {
		printk("security: conditional expressions uses unknown operator.\n");
		return 0;
	}
	
	if (expr->bool > p->p_bools.nprim) {
		printk("security: conditional expressions uses unknown bool.\n");
		return 0;
	}
	return 1;
}

static int cond_read_node(policydb_t *p, cond_node_t *node, void *fp)
{
	__u32 *buf;
	int len, i;
	cond_expr_t *expr = NULL, *last = NULL;

	buf = next_entry(fp, sizeof(__u32));
	if (!buf)
		return -1;

	node->cur_state = le32_to_cpu(buf[0]);
	
	len = 0;
	buf = next_entry(fp, sizeof(__u32));
	if (!buf)
		return -1;

	/* expr */
	len = le32_to_cpu(buf[0]);
	
	for (i = 0; i < len; i++ ) {
		buf = next_entry(fp, sizeof(__u32) * 2);
		if (!buf)
			goto err;

		expr = kmalloc(sizeof(cond_expr_t), GFP_KERNEL);
		if (!expr) {
			goto err;
		}
		memset(expr, 0, sizeof(cond_expr_t));

		expr->expr_type = le32_to_cpu(buf[0]);
		expr->bool = le32_to_cpu(buf[1]);
		
		if (!expr_isvalid(p, expr))
			goto err;

		if (i == 0) {
			node->expr = expr;
		} else {
			last->next = expr;
		}
		last = expr;
	}
	
	if (cond_read_av_list(p, fp, &node->true_list, NULL) != 0)
		goto err;
	if (cond_read_av_list(p, fp, &node->false_list, node->true_list) != 0)
		goto err;
	return 0;
err:
	cond_node_destroy(node);
	return -1;
}

int cond_read_list(policydb_t *p, void *fp)
{
	cond_node_t *node, *last = NULL;
	__u32 *buf;
	int i, len;

	buf = next_entry(fp, sizeof(__u32));
	if (!buf)
		return -1;

	len = le32_to_cpu(buf[0]);

	for (i = 0; i < len; i++) {
		node = kmalloc(sizeof(cond_node_t), GFP_KERNEL);
		if (!node)
			goto err;
		memset(node, 0, sizeof(cond_node_t));
		
		if (cond_read_node(p, node, fp) != 0)
			goto err;

		if (i == 0) {
			p->cond_list = node;
		} else {
			last->next = node;
		}
		last = node;			
	}
	return 0;
err:
	return -1;
}

/* Determine whether additional permissions are granted by the conditional
 * av table, and if so, add them to the result 
 */
void cond_compute_av(avtab_t *ctab, avtab_key_t *key, struct av_decision *avd)
{
	avtab_ptr_t node;
	
	if(!ctab || !key || !avd) 
		return;
		
	for(node = avtab_search_node(ctab, key, AVTAB_AV); node != NULL; 
				node = avtab_search_node_next(node, AVTAB_AV)) {
		if ( (__u32) (AVTAB_ALLOWED|AVTAB_ENABLED) ==
		     (node->datum.specified & (AVTAB_ALLOWED|AVTAB_ENABLED)))
			avd->allowed |= avtab_allowed(&node->datum);
		if ( (__u32) (AVTAB_AUDITDENY|AVTAB_ENABLED) ==
		     (node->datum.specified & (AVTAB_AUDITDENY|AVTAB_ENABLED)))
			/* Since a '0' in an auditdeny mask represents a 
			 * permission we do NOT want to audit (dontaudit), we use
			 * the '&' operand to ensure that all '0's in the mask
			 * are retained (much unlike the allow and auditallow cases).
			 */
			avd->auditdeny &= avtab_auditdeny(&node->datum);
		if ( (__u32) (AVTAB_AUDITALLOW|AVTAB_ENABLED) ==
		     (node->datum.specified & (AVTAB_AUDITALLOW|AVTAB_ENABLED)))
			avd->auditallow |= avtab_auditallow(&node->datum);
	}
	return;	
}

