/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <string.h>

#include "cdw_read_disc_info.h"
#include "cdw_disc.h"
#include "cdw_window.h"
#include "cdw_main_window.h"
#include "cdw_widgets.h"
#include "gettext.h"
#include "cdw_debug.h"





void cdw_read_and_display_disc_info(void)
{
	cdw_rv_t crv = CDW_OK;

	cdw_disc_t *disc = cdw_disc_new();
	if (disc) {
		crv = cdw_disc_get(disc);
	} else {
		crv = CDW_ERROR;
	}

	cdw_main_window_disc_info_view_display_data(disc);

	int n_lines = 20;
	int n_cols = 70;
	int begin_y = (LINES - n_lines) / 2;
	int begin_x = (COLS - n_cols) / 2;
	WINDOW *window = cdw_window_new((WINDOW *) NULL,
					n_lines, n_cols,
					begin_y, begin_x,
					CDW_COLORS_DIALOG,
					/* 2TRANS: this is a title of a window */
					_("Disc info"),
					/* 2TRANS: this is message at the bottom of a window */
					_("Press any key to close window"));
	if (!window) {
		cdw_vdm ("ERROR: can't create window\n");
		return;
	}
	WINDOW *subwindow = cdw_window_new(window,
					   n_lines - 2, n_cols - 2, 1, 1,
					   CDW_COLORS_DIALOG,
					   (char *) NULL, (char *) NULL);
	if (!subwindow) {
		cdw_vdm ("ERROR: can't create window\n");
		delwin(window);
		window = (WINDOW *) NULL;
		return;
	}

	wrefresh(window);

	int col = 2;
	if (crv != CDW_OK) {
		/* 2TRANS: this is a message in dialog window */
		mvwprintw(subwindow, 1, col, _("Can't get disc information"));
	} else {
		/* 2TRANS: this is a message/label in dialog window: "disc information"
		   meta information read from/about an optical disc */
		mvwprintw(subwindow, 1, col, "Disc information:");
		/* 2TRANS: this is a label in dialog window, "%s" is for label
		   like "CD-R" or "DVD+RW" */
		mvwprintw(subwindow, 3, col, "Disc type: %s", disc->type_label);
		if (disc->state_empty == CDW_TRUE) {
			/* 2TRANS: this is a message in dialog window */
			mvwprintw(subwindow, 4, col, "Disc is empty");
		} else {
			/* 2TRANS: this is a label in dialog window, "%s" is
			   for label like "ISO 9660" */
			mvwprintw(subwindow, 4, col, "File system: %s", disc->cdio->ofs->type_label);
			/* 2TRANS: this is a label in dialog window, "%zd" is
			   for number of tracks on optical disc */
			mvwprintw(subwindow, 5, col, "Number of tracks: %zd", disc->cdio->n_tracks);

			/* 2TRANS: this is a message in dialog window,
			   "write to the disc" = "burn data to the disc";
			   this message will be followed by a message stating
			   if you can write to disc */
			char *label = _("Can write to the disc: ");
			mvwprintw(subwindow, 6, col, "%s", label);
			int c = (int) strlen(label);
			if (disc->type_writable == CDW_TRUE) {
				if (disc->state_writable == CDW_TRUE) {
					/* 2TRANS: this is a label in dialog
					   window, displayed after "Can write
					   to the disc: " label; "yes" as in
					   "yes, you can write to the disc" */
					mvwprintw(subwindow, 6, c + 2, _("yes"));
				} else if (disc->state_writable == CDW_FALSE) {
					/* 2TRANS: this is a label in dialog
					   window, displayed after "Can write
					   to the disc: "; "no" as in "no, you
					   can't write to the disc" */
					mvwprintw(subwindow, 6, c + 2, _("no, disc is closed"));
				} else {
					/* 2TRANS: this is a label in dialog
					   window, displayed after "Can write
					   to the disc: " label; "unknown" as
					   in "I don't know if you can write
					   to this disc" */
					mvwprintw(subwindow, 6, c + 2, _("unknown, probably no"));
				}
			} else if (disc->type_writable == CDW_FALSE) {
				/* 2TRANS: this is message in dialog window,
				   displayed after "Can write to the disc: "
				   label; it means "no, you can't write to
				   this disc because it is 'read only' type
				   of disc" */
				mvwprintw(subwindow, 6, c + 2, _("no, disc type is \"read only\""));
			} else {
				/* 2TRANS: this is message in dialog window,
				   displayed after "Can write to the disc: "
				   label; it means "I don't know if you can
				   write to this disc" */
				mvwprintw(subwindow, 6, c + 2, _("unknown"));
			}


			/* 2TRANS: this is a label in dialog window,
			   it will be followed by a string describing
			   if a disc can be erased or no */
			label = _("Can erase this disc: ");
			c = (int) strlen(label);
			mvwprintw(subwindow, 7, col, "%s", label);

			if (disc->type_erasable == CDW_TRUE) {
				/* 2TRANS: this is a label in dialog window,
				   "yes" as in "yes, you can erase the disc" */
				mvwprintw(subwindow, 7, c + 2, _("yes"));
			} else if (disc->type_erasable == CDW_FALSE) {
				/* 2TRANS: this is a label in dialog window,
				   "no" as in "no, you can't erase the disc" */
				mvwprintw(subwindow, 7, c + 2, _("no"));
			} else {
				/* 2TRANS: this is a label in dialog
				   window, displayed after "Can erase this
				   disc: " label; "unknown" as in "I don't
				   know if you can erase this disc" */
				mvwprintw(subwindow, 7, c + 2, _("unknown, probably no"));
			}


			/* 2TRANS: this is a label in dialog window, it will
			   be followed by string representing volume ID
			   (a disc label) */
			mvwprintw(subwindow, 9, col, _("Volume id:"));
			mvwprintw(subwindow, 10, col, "\"%s\"", disc->cdio->ofs->volume_id);
		}
	}

	cdw_disc_delete(&disc);

	wrefresh(subwindow);
	wgetch(subwindow);

	delwin(subwindow);
	subwindow = (WINDOW *) NULL;
	delwin(window);
	window = (WINDOW *) NULL;

	return;

}
