/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2010 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <unistd.h>
#include <stdlib.h>
#include <string.h>

#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>

#include "cdw_utils.h"
#include "cdw_config.h"
#include "gettext.h"
#include "cdw_widgets.h" /* cdw_buttons_dialog() */
#include "cdw_drive.h"
#include "cdw_main_window.h"
#include "cdw_disc.h"
#include "cdw_fs.h"
#include "cdw_debug.h"
#include "cdw_processwin.h"
#include "cdw_read_disc_info.h"
#include "cdw_cdio_drives.h"
#include "cdw_string.h"



extern cdw_cdio_drive_t cdw_cdio_drives[];

/* main application configuration variable */
extern cdw_config_t global_config;

static int cdw_drive_toggle_tray2(const char *device_fullpath, bool with_processwin);
static int cdw_drive_execute_request(const char *device_fullpath, long unsigned int request);


static struct {
	CDW_DROPDOWN *drives_dropdown;
	int n_drives;
	/* cdio provides table that is indexed from zero - hence here is "ind"="index" */
	int cdio_default_drive_ind;
	/* cdw uses CDW_DRIVE_ID_CUSTOM == -1 to designate "custom", and
	   0 to N as regular indexes; because of using "-1" the name of
	   variable is "id"; for almost all cases id is a valid index;
	   except for -1; */
	int current_drive_id;
} drives;


static cdw_id_label_t drive_requests[] = {
	{ CDROMRESET,         "CDROMRESET" },
	{ CDROMEJECT,         "CDROMEJECT" },
	{ CDROMCLOSETRAY,     "CDROMCLOSETRAY" },
	{ CDROM_DRIVE_STATUS, "CDROM_DRIVE_STATUS" },
	{ 0,                  (char *) NULL}}; /* guard */




void cdw_drive_init(void)
{
	drives.drives_dropdown = (CDW_DROPDOWN *) NULL;
	drives.n_drives = cdw_cdio_drives_get_n_drives();
	if (drives.n_drives == 0) {
		drives.current_drive_id = CDW_DRIVE_ID_CUSTOM;
		return;
	}

	drives.cdio_default_drive_ind = cdw_cdio_drives_get_cdio_default_drive_ind();

	if (! strcmp(global_config.selected_drive, "default")) {
		/* use cdio default; in practice this happens only when
		   there is no cdio config file, and cdw runs with default
		   settings; there is no "select default drive" option */
		drives.current_drive_id = drives.cdio_default_drive_ind;
	} else if (! strcmp(global_config.selected_drive, "custom")) {
		/* use custom value entered by user in a field
		   in configuration window */
		drives.current_drive_id = CDW_DRIVE_ID_CUSTOM;
	} else {
		/* we need to search for device path from cdw_drive
		   on list of drives detected by cdio */
		drives.current_drive_id = -1;
		for (int i = 0; i < drives.n_drives; i++) {
			if (!strcmp(cdw_cdio_drives[i].fullpath, global_config.selected_drive)) {
				/* here index is a valid id */
				drives.current_drive_id = i;
				break;
			}
		}
		if (drives.current_drive_id == -1) {
			/* can't find a drive from configuration on list
			   of drives detected by cdio */
			cdw_vdm ("ERROR: can't find a drive from configuration on list of drives detected by cdio: \"%s\"\n", global_config.selected_drive);
			/* fall back to safe value */
			if (drives.n_drives == 0) {
				drives.current_drive_id = CDW_DRIVE_ID_CUSTOM;
			} else {
				/* let's hope that there is some default drive */
				drives.current_drive_id = drives.cdio_default_drive_ind;
			}
		}
	}

	return;
}





void cdw_drive_clean(void)
{
	if (drives.drives_dropdown != (CDW_DROPDOWN *) NULL) {
		cdw_dropdown_delete(&drives.drives_dropdown);
	}
	return;
}





/**
   \brief Check for CD drive status

   Check for CD drive status using ioctl (with parameter CDROM_DRIVE_STATUS).
   Documentation on ioctl can be found in linux source code documentation in
   Documentation/ioctl/cdrom.txt.

   This function tries to open CD device described by \p device_fullpath.

   Return codes are mapped to new names in cdrom_ctrl.h, so there is no need
   to include linux/cdrom.h again in *c files.

   Function does not inform user about errors, caller function has to do it.

   \param device_fullpath - full path to device that you want to check

   \return -1 if cannot open() cdrom file, or ioctl() failed
   \return ioctl return code: CDS_NO_INFO (information not available)
   \return ioctl return code: CDS_NO_DISC
   \return ioctl return code: CDS_TRAY_OPEN
   \return ioctl return code: CDS_DRIVE_NOT_READY
   \return ioctl return code: CDS_DISC_OK
 */
int cdw_drive_status(const char *device_fullpath)
{
	return cdw_drive_execute_request(device_fullpath, CDROM_DRIVE_STATUS);
}





/**
   \brief Eject cdrom drive tray

   \param device_fullpath - full path to '/dev' file corresponding to device

   \return CDW_OK on success
   \return CDW_SYS_ERROR if function cannot open() device or if ioctl call for this device fails
 */
cdw_rv_t cdw_drive_eject_tray(const char *device_fullpath)
{
	/* sometimes call to ioctl(..., CDROMEJECT) fails, not sure why,
	   especially for DVD discs (and for DVD-RW in particular);
	   perhaps some other process is trying to access freshly burned
	   disc as well?
	   let's give the drive some time to calm down, but it should
	   take much less time than waiting for drive after closing a tray;
	   experiments show that it may take 2-3 seconds before call to
	   cdw_drive_execute_request(..., CDROMEJECT) is successful */
	const int arbitrary_limit = 10;
	for (int i = 0; i < arbitrary_limit; i++) {
		cdw_vdm ("INFO: trying to eject drive, second %d\n", i);
		int rv = cdw_drive_execute_request(device_fullpath, CDROMEJECT);
		if (rv == 0) {
			cdw_vdm ("INFO: ejected drive after %d seconds\n", i);
			return CDW_OK;
		} else {
			sleep(1);
		}
	}
	cdw_vdm ("ERROR: failed to eject a drive\n");
	return CDW_GEN_ERROR;
}





/**
   \brief Close cdrom drive tray

   It is strongly advised to call sleep(x) (with x being e.g. 3) after
   calling this function. This gives the drive some time to recognize media.
   Otherwise any reads from disc in drive may yield incorrect results due to
   timeouts.

   \param device_fullpath - full path to '/dev' file corresponding to device

   \return CDW_OK on success
   \return CDW_SYS_ERROR if function cannot open() device or if ioctl call for this device fails
 */
cdw_rv_t cdw_drive_close_tray(const char *device_fullpath)
{
	int rv = cdw_drive_execute_request(device_fullpath, CDROMCLOSETRAY);
	if (rv == -1) {
		cdw_vdm ("ERROR: ioctl(%s, CDROMCLOSETRAY, ...) returns -1\n", device_fullpath);
		return CDW_SYS_ERROR;
	} else {
		/* give your drive some time to scan disc after closing
		   tray; the drive really may need 30 seconds; on my
		   not-so-old machine a drive with DVD+RW in it settles
		   its state in 17 seconds */
		const int arbitrary_limit = 30;
		for (int i = 0; i < arbitrary_limit; i++) {
			rv = cdw_drive_status(device_fullpath);
			if (rv == -1) {
				cdw_vdm ("ERROR: can't get drive status\n");
				return CDW_SYS_ERROR;
			} else if (rv == CDS_DRIVE_NOT_READY) {
				cdw_vdm ("INFO: waiting for drive, second %d\n", i);
				sleep(1);
			} else {
				cdw_vdm ("INFO: drive is ready after %d seconds\n", i);
				break;
			}
		}
		return CDW_OK;
	}
}





int cdw_drive_execute_request(const char *device_fullpath, long unsigned int request)
{
	/* yes, I know about sloppy casting of long unsigned int to
	   long signed int, but this is "only" debug code;
	   FIXME: fix this somehow; but how?
	   1. make int field in cdw_id_label_t of unsigned type, but
	      IDs can be negative
	   2. make two cdw_id_label_t types - for singed and unsigned int
	      field - maintenance nightmare */

	int cddev = open(device_fullpath, O_RDONLY | O_NONBLOCK);
	int e = errno;
	if (cddev == -1) {
		cdw_vdm ("ERROR: failed to open() device \"%s\", for request %s, error = \"%s\"\n",
			 device_fullpath,
			 cdw_utils_id_label_table_get_label(drive_requests, (long int) request),
			 strerror(e));
		return -1;
	}

	int rv = ioctl(cddev, request, 0);
	e = errno;
	close(cddev);
	if (rv == -1) {
		cdw_vdm ("ERROR: ioctl(%s, %s, ...) returns -1 / \"%s\"\n",
			 device_fullpath,
			 cdw_utils_id_label_table_get_label(drive_requests, (long int) request),
			 strerror(e));
	}

	return rv;
}





/**
   \brief Ensure that disc is in drive and drive tray is closed

   This function prevents program from attempting to access CD
   when there is no disc in drive.
   Uses ioctl() to check drive status and ensure that disc is in drive.
   Uses dialog box to communicate with user in case of problems.

   \return true if disc is in tray and tray is closed
   \return false if drive is empty and user decided to not put disc in drive or drive doesn't answer
 */
bool cdw_drive_is_disc_in_drive(void)
{
	const char *drive = cdw_drive_get_drive_fullpath();
	/* 2TRANS: this is title of dialog window:
	   some problem occurred while trying to access disc */
	const char *title = _("CD drive error");
	char *message = (char *) NULL;
	int buttons = 0;
	for (int i = 0; i < 10; i++) {
		/* 2TRANS: this is message displayed in process window */
		cdw_processwin_display_sub_info(_("Getting drive status"));
		cdw_processwin_wrefresh();
		int rv = cdw_drive_status(drive);
		switch (rv) {
			case CDS_DISC_OK:
				return true;
			case CDS_NO_DISC:
				/* 2TRANS: this is message in dialog window: no optical disc
				   found in drive; user can select OK or CANCEL button */
				message =  _("No disc in CD/DVD drive. Please insert disc.");
				buttons = CDW_BUTTONS_OK_CANCEL;
				break;
			case CDS_TRAY_OPEN:
				/* 2TRANS: this is message in dialog window; user
				   can select OK or CANCEL button */
				message = _("Please close CD/DVD drive tray.");
				buttons = CDW_BUTTONS_OK_CANCEL;
				break;
			case CDS_DRIVE_NOT_READY:
				sleep(2);
				/* 2TRANS: this is message in dialog window;
				   user can select OK or CANCEL button */
				message = _( "CD/DVD drive doesn't respond (timeout). Try again?");
				buttons = CDW_BUTTONS_OK_CANCEL;
				break;
			case CDS_NO_INFO:
				/* 2TRANS: this is message in dialog window;
				   cannot read some meta-data from cd; user
				   can select OK or CANCEL button */
				message = _("Can't get CD/DVD drive info. Continue?");
				buttons = CDW_BUTTONS_OK_CANCEL;
				break;
			case -1: /* cannot open device file or ioctl() failed */
				/* 2TRANS: this is message in dialog window;
				   for some reason program cannot open CD device.
				   User should check his settings */
				message = _("Can't open CD/DVD drive device. Please check your configuration.");
				buttons = CDW_BUTTONS_OK;
				break;
			default: /* all return values from ioctl() and open() covered, so this */
				/* 2TRANS: this is message in dialog window:
				   unknown error occurred */
				message = _("Can't open CD/DVD drive device. Unknown error.");
				buttons = CDW_BUTTONS_OK;
				break;
		} /* switch */

		cdw_rv_t crv = cdw_buttons_dialog(title, message,
						  buttons, CDW_COLORS_ERROR);
		cdw_main_ui_main_window_wrefresh();
		if (buttons == CDW_BUTTONS_OK) {
			break; /* break loop, go to final error message */
		} else {
			if (crv == CDW_OK) {
				; /* continue loop */
			} else {
				return false;
			}
		}

		sleep(3);
	} /* for () */

	cdw_buttons_dialog(title,
			   /* 2TRANS: this is message in dialog window: some errors
			      occurred while accessing drive */
			   _("Something went wrong.\nPlease check settings in Configuration -> Hardware and try once more."),
			   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
	cdw_main_ui_main_window_wrefresh();

	return false; /* user couldn't decide what to do or hardware settings in cdw options are invalid */
}





/**
   \brief Reload or eject disc, update disc info view

   Eject tray of cd drive. If \p final_eject is true, check value
   of config.eject to see if tray should stay open. If \p final_eject is
   false, close tray.

   Values displayed in disc info view are reset after ejecting tray and
   updated with valid values after closing tray. Valid values are
   collected by cdw_disc_current_disc_get_meta_info(true).

   This code would be used two or three times in main(), so
   I decided to extract it to separate function.

   \param device_fullpath - path to device that you want to control
   \param final_eject - should the function look at config.eject to check if tray should be left open?
 */
void cdw_drive_reload_tray_with_ui_update(const char *device_fullpath, bool final_eject)
{
	/* 2TRANS: this is message displayed in process window */
	cdw_processwin_display_sub_info(_("Ejecting tray"));
	cdw_processwin_wrefresh();
	cdw_drive_eject_tray(device_fullpath);

	cdw_disc_t *current_disc = cdw_disc_get();
	cdw_disc_reset(current_disc);
	cdw_main_ui_disc_info_view_display_data(current_disc);

	if (final_eject) {
		if (global_config.eject) { /* leave tray open */
			return;
		}
	}

	/* close tray: either user don't want to keep it opened after
	   performing writing operation (this is when config.eject == false)
	   or cdw will be doing something more on the disc (this is when
	   final_eject == false) */

	/* 2TRANS: this is message displayed in process window;
	   "tray" is an optical drive tray */
	cdw_processwin_display_sub_info(_("Closing tray, waiting for drive"));
	cdw_processwin_force_refresh();
	/* to avoid sending to drive "eject" and "close" commands
	   in very short time  */
	usleep(500000);
	cdw_drive_close_tray(device_fullpath);

	cdw_rv_t mi = cdw_disc_get_meta_info(current_disc);
	if (mi == CDW_OK) {
		/* 'current_disc' still points to one and only 'current_disc'
		   variable, which only gets updated, not reallocated, after
		   cdw_disc_get_meta_info() call */
		cdw_main_ui_disc_info_view_display_data(current_disc);
	} else {
		; /* disc info view is not updated, but user can press
		     'R' key to try to refresh it */
	}

	return;
}





int cdw_drive_toggle_tray2(const char *device_fullpath, bool with_processwin)
{
	cdw_rv_t m = cdw_fs_check_device_mounted(device_fullpath);
	if (m == CDW_SYS_ERROR) {
		; /* let's try anyway */
	} else if (m == CDW_OK) { /* device is mounted */
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Message"),
				   /* 2TRANS: this is message in dialog window */
				   _("Cannot eject tray. Please unmount your drive."),
				   CDW_BUTTONS_OK, CDW_COLORS_WARNING);
		return CDW_GEN_ERROR;
	} else { /* crv == CDW_NO, not mounted */
		;
	}

	for (int i = 0; i < 10; ) {
		int rv = cdw_drive_status(device_fullpath);
		if (rv == CDS_DISC_OK
		    || rv == CDS_NO_DISC) {
			cdw_vdm ("INFO: tray is closed (CDS_DISC_OK || CDS_NO_DISC) - ejecting\n");
			if (with_processwin) {
				/* 2TRANS: this is message displayed in process window */
				cdw_processwin_display_sub_info(_("Ejecting tray"));
			}
			cdw_drive_eject_tray(device_fullpath);
			break;

		} else if (rv == CDS_TRAY_OPEN) {
			cdw_vdm ("INFO: tray is open (CDS_TRAY_OPEN) - closing\n");
			if (with_processwin) {
				/* 2TRANS: this is message displayed in process window */
				cdw_processwin_display_sub_info(_("Closing tray and waiting for drive"));
			}
			usleep(500000);
			cdw_drive_close_tray(device_fullpath);
			break;

		} else if (rv == -1 || rv == CDS_NO_INFO) {
			cdw_vdm ("ERROR: -1 || CDS_NO_INFO\n");
			sleep(1);
			break;

		} else { /* CDS_DRIVE_NOT_READY */
			cdw_vdm ("WARNING: CDS_DRIVE_NOT_READY\n");
			sleep(1);
			i++;
		}
	}

	/* get final status: if there were no problems this will be a status
	   after toggling, and if there were problems, the error value will
	   be passed to caller */
	int retval = cdw_drive_status(device_fullpath);

	if (retval == -1 || retval == CDS_NO_INFO) {
		cdw_vdm ("ERROR: CDW_SYS_ERROR || CDS_NO_INFO\n");
		/* 2TRANS: this is title of dialog window: some error occurred
		   while trying to access disc */
		cdw_buttons_dialog(_("Drive error"),
				   /* 2TRANS: this is message in dialog window:
				      some error occurred while accessing cdrom */
				   _("Can't open device. Please check your configuration."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
	} else if (retval == CDS_DRIVE_NOT_READY) {
		cdw_vdm ("ERROR: CDS_DRIVE_NOT_READY\n");
		/* 2TRANS: this is title of dialog window: some error occurred
		   while trying to access disc */
		cdw_buttons_dialog(_("CD drive error"),
				   /* 2TRANS: this is message in dialog window */
				   _("CDROM doesn't respond (timeout)."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
	} else {
		/* some non-error value */
	}

	return retval;
}





void cdw_drive_toggle_tray_with_ui_update2(void)
{
	const char *drive = cdw_drive_get_drive_fullpath();
	if (drive == (const char *) NULL) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("Can't find any drive. Please check your hardware configuration."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return;
	}

	/* 2TRANS: this is title of process window;
	   "drive" is "optical drive" - a burner */
	cdw_processwin_create(_("Drive tray"),
			      /* 2TRANS: this is message in process window,
				  "drive" is "optical drive" - a burner */
			      _("Opening/closing drive tray"), false);

	int rv = cdw_drive_toggle_tray2(drive, true);

	cdw_disc_t *current_disc = cdw_disc_get();
	cdw_disc_reset(current_disc);
	if (rv == CDS_DISC_OK) {
		cdw_rv_t crv = cdw_read_disc_info();
		if (crv != CDW_OK) {
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Error"),
					   /* 2TRANS: this is message in dialog window */
					   _("Cannot get media info."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

		}
	} else if (rv == CDS_NO_DISC || rv == CDS_TRAY_OPEN) {
		;
	} else if (rv == CDS_DRIVE_NOT_READY) {
		cdw_vdm ("WARNING: drive still not ready\n");
	} else {
		cdw_vdm ("ERROR: failed to toggle drive tray\n");
	}

	cdw_main_ui_disc_info_view_display_data(current_disc);
	/* 2TRANS: this is message in process window,
	    "drive" is "optical drive" - a burner */
	cdw_processwin_destroy(_("Drive state changed"), false);

	return;
}





CDW_DROPDOWN *cdw_drive_make_drives_dropdown(WINDOW *window, int row, int col, int width)
{
#define LABEL_LEN 50

	drives.drives_dropdown = cdw_dropdown_new(window, row, col, width,
						  drives.n_drives + 1, CDW_COLORS_DIALOG);


	char label[LABEL_LEN + 1];
	size_t max_len = width > LABEL_LEN ? LABEL_LEN : (size_t) width;
	/* 2TRANS: this is a label in a dropdown; "custom path" is
	   "path to a device file, specified by user" */
	strncpy (label, _("Use custom path to drive"), max_len);
	label[LABEL_LEN] = '\0';
	cdw_rv_t crv = cdw_dropdown_add_item(drives.drives_dropdown, CDW_DRIVE_ID_CUSTOM, label);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to add first label \"%s\" to dropdown\n", label);
		cdw_dropdown_delete(&drives.drives_dropdown);
		return (CDW_DROPDOWN *) NULL;
	}

	for (int i = 0; i < drives.n_drives; i++) {
		snprintf(label, max_len + 1, "%s%s (%s / %s)",
			 i == drives.cdio_default_drive_ind ? "* " : "",
			 cdw_cdio_drives[i].fullpath,
			 cdw_cdio_drives[i].hw_info.psz_vendor,
			 cdw_cdio_drives[i].hw_info.psz_model);

		cdw_vdm ("INFO: adding dropdown label #%d = \"%s\"\n", i, label);
		crv = cdw_dropdown_add_item(drives.drives_dropdown, i, label);
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to add first label \"%s\" to dropdown\n", label);
			cdw_dropdown_delete(&drives.drives_dropdown);
			return (CDW_DROPDOWN *) NULL;
		}
	}
	cdw_dropdown_finalize(drives.drives_dropdown);

	crv = cdw_dropdown_set_current_item_by_id(drives.drives_dropdown, drives.current_drive_id);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set current item\n");
		/* what now? */
	}
	cdw_dropdown_display_current_item(drives.drives_dropdown);

	return drives.drives_dropdown;
}





const char *cdw_drive_get_drive_fullpath(void)
{
	if (drives.n_drives == 0) {
		/* there are zero drives detected by cdio, but there may be
		   some custom path entered in configuration window;
		   if there is no custom path, then function will return
		   empty string, which won't be an error, but will signify
		   that there are surely no drives available */
		const char *path = cdw_config_get_custom_drive();
		if (!strlen(path)) {
			path = (const char *) NULL;
		}
		cdw_vdm ("INFO: n_drives = 0, selecting custom drive path = \"%s\"\n", path);
		return path;
	}

	if (drives.current_drive_id == CDW_DRIVE_ID_CUSTOM) { /* use custom drive path */
		const char *path = cdw_config_get_custom_drive();
		cdw_vdm ("INFO: selecting custom drive path = \"%s\"\n", path);
		return path;
	} else {
		const char *path = cdw_cdio_drives[drives.current_drive_id].fullpath;
		cdw_vdm ("INFO: selecting cdio drive path #%d = \"%s\"\n", drives.current_drive_id, path);
		return path;
	}
}





cdw_rv_t cdw_drive_print_help_message(WINDOW *window)
{
#if 0
	/* 2TRANS: this is message displayed as a small help in configuration
	   window. Please keep line breaks, they are used for making message
	   layout more readable, not to preserve certain lengths of lines.
	   Please keep "/dev/xxx" unchanged. Remember to escape quotes. */
	cdw_textarea_print_message(window, _("INFO:\n\n\"CD Reader/Writer device\" entry should have form \'/dev/xxx\' and is obligatory - it cannot be left empty. Don't put ending slash in the path.\n\n\"SCSI device\" entry should have \'X,Y,Z\' or \'ATAPI:X,Y,Z\' form and is optional. It should be set and used only if cdrecord has troubles using \'/dev/xxx\' device - otherwise leave empty. Consult cdrecord manual and/or output of 'cdrecord dev=help' command for details of \"SCSI device\" entry."),
				   CDW_ALIGN_LEFT);
#else
	char *message = cdw_string_concat
		/* 2TRANS: this is a help message */
		(drives.n_drives == 0 ? _("cdw can't detect any drives, you may want to enter and use custom value.\n") : "",
		 /* here -1 is only an initial value */
		 drives.n_drives != 0 && drives.cdio_default_drive_ind != -1 ?
		 /* 2TRANS: this is a help message, describing content
		    of a dropdown */
		 _("'*' in dropdown denotes default drive.\n") : "",
		 /* 2TRANS: this is a help message, keep \"/dev/xxx\" unchanged */
		 _("Custom path to device should have form \"/dev/xxx\", without ending slash.\n\"SCSI device\" field is for experts only, and if filled, cdw may not work correctly.\n"),
		 (char *) NULL);

	cdw_textarea_print_message(window, message, CDW_ALIGN_LEFT);
	free(message);
	message = (char *) NULL;
#endif
	return CDW_OK;
}





void cdw_drive_save_config(cdw_config_t *config)
{
	int id = cdw_dropdown_get_current_item_id(drives.drives_dropdown);
	drives.current_drive_id = id;
	if (id == CDW_DRIVE_ID_CUSTOM) {
		/* current selection is "use custom drive path" */
		strncpy(config->selected_drive, "custom", OPTION_FIELD_LEN_MAX);
		config->selected_drive[OPTION_FIELD_LEN_MAX] = '\0';
		cdw_assert (!strcmp(config->selected_drive, "custom"), "ERROR: failed to correctly save \"custom\" value\n");
	} else {
		/* current selection is one of drives detected by cdio */
		strncpy(config->selected_drive, cdw_cdio_drives[id].fullpath, OPTION_FIELD_LEN_MAX);
		config->selected_drive[OPTION_FIELD_LEN_MAX] = '\0';
		cdw_assert (!strcmp(config->selected_drive, cdw_cdio_drives[id].fullpath), "ERROR: failed to correctly save non-custom value \"%s\"\n", cdw_cdio_drives[id].fullpath);
	}

	cdw_vdm ("INFO: cdw drive saved as \"%s\"\n", config->selected_drive);

	return;
}




int cdw_drive_get_current_drive_id(void)
{
	return drives.current_drive_id;
}

