/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * A few small modifications (C) 2007 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * Content of file:
 * - main()
 * - cdw_main_loop(): main events/keys handling loop
 * - clean_before_cdw_exit(): clean up resources before exiting cdw
 */

#include <stdio.h>
#include <stdlib.h>

#include <ncursesw/ncurses.h>
#include <ncursesw/menu.h>

#include <fcntl.h>
#include <libintl.h>
#include <locale.h>
#include <mcheck.h>
#include <string.h>
#include <unistd.h>

#include "help.h"
#include "options.h"
#include "log.h"
#include "config.h"
#include "gettext.h"
#include "commands.h"
#include "cdda2wav.h"
#include "cdw_widgets.h" /* dialogbox() */
#include "main.h"
#include "utils.h"
#include "cdw_tmp_files.h"
#include "thread.h"
#include "cdrom_ctrl.h"
#include "cdw_ui.h" /* delete_files () */

#ifdef GTK_GUI
#include "gtkgui.h"
#endif

#define COLOR_GRAY



void clean_before_cdw_exit(void);
int cdw_main_loop(void);


/* global across files */
int num;
extern struct conf config; /* main cdw configuration variable */
extern WINDOW *mainwin;    /* main application window */
extern MENU *menu;
extern MENU *cdmenu;



int DEBUGMSG = 0;



int main(int argc, char *argv[])
{
	int rv; /* generic return value variable */

	mtrace();

	cdw_locale_init();

	if (argc > 1) {
		process_commandline_args(argc, argv);
	}

	/* initialize ncurses - do this as soon as possible,
	 * some error situations may require use of dialog windows */
	rv = cdw_curses_init();
	if (rv == -1) {
		clean_before_cdw_exit();
		printf(_("\nCannot start cdw: needed min 80x25 terminal!\n"));
		exit(-1);
	}

	/* setup of cdw options module */
	rv = options_init();

	/* creating new processes in threads code may create some zombie processes */
	zombie_handler_init();

	/* list of files to be written to image/disc */
	filelist_init();

	/* cdw uses some temporary files */
	tmpfiles_init();

	/* application name string for main app window - yes, it's trivial */
	cdw_title_init();

	/* main app window */
	mainform_init();

	select_window(TRUE);

	/*
	 * initially cursor is on first left-hand menu item, but tooltips are
	 * displayed only after cursor movement; show tooltip for initial
	 * cursor position
	 */
	display_tooltip(0);

	/* main program loop - top level keys handler */
	cdw_main_loop();

	/*
	 * now let's unwind (almost) all setup code
	 * called before cdw_main_loop(); we will do this in
	 * one nice function: clean_before_cdw_exit() - wrapper function user
	 * in many other places
	 */
	clean_before_cdw_exit();

	return 0;
}




/**
 * Deallocate all previously allocated resources, terminate ncurses
 *
 * Prepare application for closing:close all files, deallocate memory,
 * destroy custom and ncurses widgets. Exit ncurses. Try to do all this
 * in reverse order to the one found in main().
 */
void clean_before_cdw_exit(void)
{
	title_clean();

	filelist_clean(); /* list of files to be written to image/disc */

	mainform_clean(); /* main application window */

	options_clean(); /* cdw options module */

	endwin(); /* exit ncurses */

	return;
}




/**
 * Main ncurses events loop - main keys handling loop
 *
 * Function that handles keys when cdw is started and user sits in front of
 * main cdw window: menu on the left, list of files in main area and
 * information area at the bottom. User can select items from letf-hand menu
 * or use hotkeys.
 *
 * \returns 0
 */
int cdw_main_loop(void)
{
	char command[555];
	int c;
	int rv = 0;

	while (c = wgetch(mainwin)) {
		switch (c) {

			/* -------- HOT KEYS -------- */
#ifdef HAVE_LIBMYSQLCLIENT
		case 'c':
		case 'C':
			/* CDDB */
			cddb_window();
			wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
			select_window(FALSE);
			break;
#endif
#ifdef HAVE_LIBSQLITE
		case 'c':
		case 'C':
			/* CDDB */
			cddb_window();
			wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
			select_window(FALSE);
			break;
#endif
		case 'e':
		case 'E':
			/* eject tray */
			rv = cdrom_status();
			if (rv == CDS_TRAY_OPEN) {
				close_tray(config.cdrwdevice);
			} else {
				eject_tray(config.cdrwdevice);
			}
			break;
#if 0 // not in this release
		case 'g':
		case 'G':
			/* cdda2wav */
			cdda2wav(mainwin, &config);
			wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
			select_window(TRUE);
			break;
#endif
		case 'q':
		case 'Q':
			/* this should be the same code as for menu item 'Quit' below */
			rv = dialogbox(_("Please confirm"), _("Do you really want to quit cdw?"), DIALOG_OK_CANCEL);
			if (rv == BUTTON_OK) {
				clean_before_cdw_exit();
				exit(EXIT_SUCCESS);
			}
			break;
		case KEY_F(1):
			/*  Show readme */
			/* sprintf(command, "%s/README.gz", DOCDIR);
			 * log_window(_("Help"), command, 1); */
			show_help_main();
			wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
			select_window(TRUE);
			break;
		case KEY_F(2):
			/* Edit volume ID */
			input_box(_("Volume ID"), _("Enter volume label:"), config.volumeid, 0);
			wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
			select_window(TRUE);
			break;
		case KEY_F(3):
			/* Edit other parameters */
			input_box(_("CD record parameters"), _("Enter parameter:"), config.other, 0);
			wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
			select_window(TRUE);
			break;
		case KEY_F(4):
			/* CD size select */
			size_selector();
			wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
			select_window(TRUE);
			break;
		case 'L':
		case 'l':
		case KEY_F(5):
			/* Show last log */
			log_window(_("Last log"), config.logfile, 0);
			wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
			select_window(TRUE);
			break;
		case KEY_F(6):
			/* Show CD info */
			sprintf(command, "%s dev=%s -atip", CDRECORD, config.scsi);
			run_command(command, _("Load CD info..."), 0, 0, 0, "");
			log_window(_("CD info"), config.logfile, 0);
			wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
			select_window(TRUE);
			break;
/*	    case KEY_F(9):
		dialogbox("Dialog box", "This is a sample message. Show in a simple\ndialog box\n\nOk?", 3);
		wrefresh(derwin(mainwin,LINES-1,COLS-1,0,0));
		select_window(TRUE);
		break;*/

		case KEY_F(10):
			/* Show GPL */
			sprintf(command, "%s/LICENSE", DOCDIR);
			log_window(_("License"), command, 1);
			wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
			select_window(TRUE);
			break;
#ifdef HAVE_LIBMYSQLCLIENT
		case 'A':
		case 'a':
			/* Add disk to dic database */
			add_to_dic();
			wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
			select_window(TRUE);
			break;
#endif
#ifdef HAVE_LIBSQLITE
		case 'A':
		case 'a':
			/* Add disk to dic database */
			add_to_dic();
			wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
			select_window(TRUE);
			break;
#endif

		/* ******************************
	         * ******  LEFT-HAND MENU  ******
		 * ******************************/

		case KEY_HOME:
			menu_driver(menu, REQ_FIRST_ITEM);
			display_tooltip(item_index(current_item(menu)));
			break;
		case KEY_END:
			menu_driver(menu, REQ_LAST_ITEM);
			display_tooltip(item_index(current_item(menu)));
			break;
		case KEY_DOWN:
			menu_driver(menu, REQ_DOWN_ITEM);
			display_tooltip(item_index(current_item(menu)));
			break;
		case KEY_UP:
			menu_driver(menu, REQ_UP_ITEM);
			display_tooltip(item_index(current_item(menu)));
			break;
		case CDW_ENTER:
			// FIXME - replace 0, 1, 2 etc. with symbolic names
			/* Add files to list of files to be written to cd/image */
			if (item_index(current_item(menu)) == 0) {
				put_addwin(getmaxy(mainwin) - 10, getmaxx(mainwin) - 10, (LINES - (getmaxy(mainwin) - 10)) / 2, (COLS - (getmaxx(mainwin) - 10)) / 2);
				wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
				select_window(TRUE);
			}
			/* Delete files from list of files to be written to cd/image */
			if (item_index(current_item(menu)) == 1) {
				/* if cd files list not empty */
				if (num > 0) {
					set_menu_fore(cdmenu, COLOR_PAIR(1) | A_REVERSE);
					set_menu_fore(menu, COLOR_PAIR(4));
					wrefresh(mainwin);
					delete_files();
					set_menu_fore(cdmenu, COLOR_PAIR(1));
					set_menu_fore(menu, COLOR_PAIR(4) | A_REVERSE);
				} else {
					dialogbox(_("No files to delete"), _("Nothing to delete - no files selected."), DIALOG_OK);
				}
			}

			/* Create iso image */
			if ((item_index(current_item(menu)) == 2)) {
				if (num > 0) { /* there are some files on 'selected files' list */
					rv = conditional_volume_label_dialog();
					if (rv != CDW_ENTER) {
						dialogbox(_("Operation canceled"), _("You pressed escape - writing cancelled."), DIALOG_OK);
						break;
					}
					rv = run_command_create_image();
					after_event("\"Write to image\" log", 1);
					if (rv != 0) {
						break; // FIXME - why break?
					}
				} else {
					dialogbox(_("No files to write"), _("No files selected. Please use 'Add files'"), DIALOG_OK);
				}
			}

#if 0 // not in this release
			/* Copy image */
			if ((item_index(current_item(menu)) == 3)) {
				int exist = 0;
				FILE *isofile;

				if ((isofile = fopen(config.tempdir, "r")) != NULL) {
					exist = dialogbox(_("Image exist"), _("copy: Image file exist!\nDo you wish to overwrite it?"), 3);
					fclose(isofile);
				}

				// erase 'Image exists' dialog box window
				wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
				select_window(FALSE);

				if (disc_in_drive()) {
					if (exist == 0) {
						copy_image(_("Copy CD image"));
					}
					wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
					select_window(FALSE);
					wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
					select_window(FALSE);
				}
				wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
				select_window(FALSE);
			}
#endif
			/* Write selected files to optical disk */
			if (item_index(current_item(menu)) == 3) {
				if (!disc_in_drive()) {
					break;
				}

				if (strcmp(config.fromimage, "1") == 0) {
					if (access(config.tempdir, R_OK) == 0) {
						/* Write from image */
						run_command_write_from_image();
						eject_tray(config.cdrom);
						close_tray(config.cdrom);
						after_event("\"Write image\" log", 1);
					} else {
						dialogbox(_("No image file"), _("No image file. Please check your options."), DIALOG_OK);
					}
				} else if ((strcmp(config.fromimage, "0") == 0)) {
					rv = conditional_volume_label_dialog();
					if (rv == CDW_ESCAPE) {
						dialogbox(_("Operation canceled"), _("You pressed escape - writing cancelled."), DIALOG_OK);
						break;
					}
					if ( ! filelist_empty() ){
						/* Write directly from filesystem, based on list of selected files */
						run_command_write_direct();
						eject_tray(config.cdrom);
						close_tray(config.cdrom);
						after_event("\"Write direct\" log", 1);
					} else {
						dialogbox(_("No files to write"), _("No files selected. Please use 'Add files'"), DIALOG_OK);
					}
				} else { /* is it really necessary? */
					;
				}
			}
#if 0 // not in this release
			/* Write Audio */
			if ((item_index(current_item(menu)) == 5)) {
				run_command_write_audio();
			}
			/* Copy data CD */
			if ((item_index(current_item(menu)) == 6) && (config.cdrom != config.cdrwdevice)) {
				run_command_copy_disk();
			}
#endif
			/* Blank CD-RW */
			if (item_index(current_item(menu)) == 4) {
				if (!disc_in_drive()) {
					break;
				}

				/* prompt user for blank method */
				rv =  blank_method_selector();
				wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
				select_window(FALSE);

				if (rv == CDW_ESCAPE) { /* selection of blanking method was cancelled by user */
					break;
				}

				/* check if disc is erasable;
				 * possible answers:
				 * yes (rv = 0), don't know (rv = 0), no (rv != 0) */
				int rv = run_command_check_erasable();
				if (rv == 0) {
					rv = run_command_blank_cd();
					wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
					select_window(FALSE);
					if (rv == 0) { // blanking successful
						eject_tray(config.cdrom);
						close_tray(config.cdrom);
					}
				} else {
					dialogbox(_("Media not erasable"), _("Disc in drive is not erasable. I cannot blank it"), DIALOG_OK);
				}

			}

			/* Edit cdw options */
			if (item_index(current_item(menu)) == 5) {
				/* options_window() reads current configuration
				 * (saved in disk file), shows user a window
				 * that allows him modify the options, save
				 * options to disk file and to global variable
				 * struct conf config */
				options_window();
				wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
				select_window(FALSE);
				fill_info();
			}

			/* Quit - Exit cdw */
			if (item_index(current_item(menu)) == 6) {
				/* this should be the same code as for q/Q hotkey above */
				rv = dialogbox(_("Please confirm"), _("Do you really want to quit cdw?"), DIALOG_OK_CANCEL);
				if (rv == BUTTON_OK) {
					clean_before_cdw_exit();
					exit(EXIT_SUCCESS);
				}
			}

			break;
		} /* switch(c) */
		wrefresh(mainwin);
	} /* while (c = wgetch(mainwin)) */

	return 0;
}


