/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <thread>

#ifdef _OPENMP
#include <omp.h>
#endif

#include <stdio.h>
#include <sys/stat.h> /* stat */

#include <map>

#include <cdi.h>
#include "pipe.h"
#include "pstream.h"
#include "timer.h"
#include "cdo_output.h"
#include "cdo_options.h"
#include "commandline.h"
#include "assert.h"
#include "compare.h"
#include "array.h"
#include "cdo_options.h"
#include "cdo_history.h"
#include "cdo_defaultValues.h"
#include "cdo_options.h"
#include "cdi_lockedIO.h"

#ifdef HAVE_LIBPTHREAD
#include <pthread.h>
#include "pthread_debug.h"
#endif

static int processNum = 0;
void
setProcessNum(int p_num)
{
  processNum = p_num;
}

void
closeAllStreams()
{
}

Pstream::Pstream(std::string p_filename) : m_name(p_filename){ispipe = false;}

Pstream::Pstream(int p_processID)
{
  init();
  ispipe = true;
  pipe = std::make_shared<pipe_t>();
  pipe->pipeSetName(p_processID, m_cdoStreamID);
  m_name = pipe->name;
}

CdoStreamID
pstreamToPointer(int idx)
{
    return nullptr;
}

void
Pstream::init()
{
  isopen = false;
  m_fileID = -1;
  m_mode = 0;
#ifdef HAVE_LIBPTHREAD
  rthreadID = 0;
  wthreadID = 0;
#endif
}

bool
Pstream::isPipe()
{
  return ispipe;
}

int
Pstream::openReadPipe()
{
#ifdef HAVE_LIBPTHREAD
  rthreadID = pthread_self();

  Debug(PSTREAM, "pipe %s", pipe->name);

  isopen = true;

  return m_cdoStreamID;
#else
  cdoAbort("Cannot use pipes, pthread support not compiled in!");
  return -1;
#endif
}

void
Pstream::openReadFile()
{

  Debug(PSTREAM, "Opening (r) file: %s ", m_name);

  openLock();

  CdiStreamID fileID = streamOpenRead(m_name.c_str());
  if (fileID < 0) cdiOpenError(fileID, "Open failed on >%s<", m_name.c_str());

  if (Options::numStreamWorker > 0) streamDefNumWorker(fileID, Options::numStreamWorker);

  isopen = true;

  if (CdoDefault::FileType == CDI_UNDEFID) CdoDefault::FileType = streamInqFiletype(fileID);

  cdoInqHistory(fileID);

  openUnlock();

  m_mode = 'r';
  m_fileID = fileID;
}

int
Pstream::openWritePipe(int filetype)
{
#ifdef HAVE_LIBPTHREAD
  Debug(PSTREAM, "pipe %s", pipe->name);

  wthreadID = pthread_self();
  m_filetype = filetype;
  isopen = true;

  return m_cdoStreamID;
#else
  return -1;
#endif
}

int
Pstream::openWriteFile(int filetype)
{
  if (Options::cdoInteractive)
    {
      struct stat stbuf;
      const int rstatus = stat(m_name.c_str(), &stbuf);
      /* If permanent file already exists, query user whether to overwrite or exit */
      if (rstatus != -1) query_user_exit(m_name.c_str());
    }

  Debug(PSTREAM, "Opening (w) file %s", m_name);

  if (filetype == CDI_UNDEFID) filetype = CDI_FILETYPE_GRB;

  if (processNum == 1 && Threading::ompNumThreads == 1) timer_start(timer_write);

  openLock();
  int fileID = streamOpenWrite(m_name.c_str(), filetype);
  openUnlock();

  if (processNum == 1 && Threading::ompNumThreads == 1) timer_stop(timer_write);
  if (fileID < 0) cdiOpenError(fileID, "Open failed on >%s<", m_name.c_str());
  isopen = true;

  cdoDefHistory(fileID, commandLine());

  if (CdoDefault::Byteorder != CDI_UNDEFID) streamDefByteorder(fileID, CdoDefault::Byteorder);

  set_comp(fileID, filetype);

  m_mode = 'w';
  m_fileID = fileID;
  m_filetype = filetype;

  return m_cdoStreamID;
}

int
Pstream::openWrite(int p_filetype)
{
  if (ispipe)
    {
      openWritePipe(p_filetype);
    }
  else
    {
      Debug(PROCESS, "Trying to open: %s", m_name);
      openWriteFile(p_filetype);
    }
  return 1;
}

int
Pstream::openRead()
{
  if (ispipe)
    {
      Debug(PROCESS, "Trying to open pipe: %s", pipe->name);
      openReadPipe();
    }
  else
    {
      Debug(PROCESS, "Trying to open file: %s", m_name);
      openReadFile();
    }
  return 1;
}

int
Pstream::openAppend()
{
  if (ispipe)
    {
      Debug(PROCESS, "pipe %s", pipe->name);
      cdoWarning("This operator doesn't work with pipes!");
    }
  else
    {
      if (processNum == 1 && Threading::ompNumThreads == 1) timer_start(timer_write);
      openLock();
      int fileID = streamOpenAppend(m_name.c_str());
      openUnlock();
      if (processNum == 1 && Threading::ompNumThreads == 1) timer_stop(timer_write);
      if (fileID > 0)
        {
          isopen = true;
          int filetype = streamInqFiletype(fileID);
          set_comp(fileID, filetype);

          m_mode = 'a';
          m_fileID = fileID;
        }
    }
  return m_fileID;
}

void
Pstream::closePipe()
{
  // pipe->close();
  //  pthread_cond_signal(pipe->recInq);

  // pthread_mutex_lock(pipe->m_mutex);
  isopen = false;
  pipe->close();
  // pthread_mutex_unlock(pipe->m_mutex);
  // pthread_cond_signal(pipe->isclosed);

  Debug(PSTREAM,"Joining with write thread: %ld", wthreadID);
  pthread_join(wthreadID, nullptr);
}

size_t
Pstream::getNvals()
{
  return m_nvals;
}

void
Pstream::waitForPipe()
{
  pipe->close();
  std::unique_lock<std::mutex> locked_mutex(pipe->m_mutex);
  Debug(PSTREAM,"Waiting for pipe");
  while (isopen)
    {
      Debug(PSTREAM, "wait of read close");
      pthread_cond_wait(pipe->isclosed, locked_mutex);
    }
  locked_mutex.unlock();
}

void
Pstream::close()
{
  if (ispipe)
    {
#ifdef HAVE_LIBPTHREAD
      pthread_t threadID = pthread_self();

      Debug(PSTREAM, "thID: %d rthID: %d wthID: %d", threadID, rthreadID, wthreadID);

      if (pthread_equal(threadID, rthreadID))
        {
          closePipe();
        }
      else if (pthread_equal(threadID, wthreadID))
        {
          waitForPipe();
        }
      else
        {
          cdoAbort("Internal problem! Close pipe %s", m_name);
        }
#else
      cdoAbort("Cannot use pipes, pthread support not compiled in!");
#endif
    }
  else
    {
      Debug(PSTREAM, "%s fileID %d", m_name, m_fileID);

      streamCloseLocked(m_fileID);
      isopen = false;

      if (m_varlist.size())
        {
          m_varlist.clear();
          m_varlist.shrink_to_fit();
        }
    }
}

int
Pstream::inqVlist()
{
  int vlistID = -1;

#ifdef HAVE_LIBPTHREAD
  // read from pipe
  if (ispipe)
    {
      vlistID = pipe->pipeInqVlist(m_vlistID);
      if (vlistID == -1) cdoAbort("Couldn't read data from input stream %s!", m_name.c_str());
    }
  // read from file through cdi streamInqVlist
  else
#endif
    {
      if (processNum == 1 && Threading::ompNumThreads == 1) timer_start(timer_read);
      vlistID = streamInqVlistLocked(m_fileID);
      if (processNum == 1 && Threading::ompNumThreads == 1) timer_stop(timer_read);
      if (vlistID == -1)
        {
          cdoAbort("Couldn't read data from input fileID %d!", m_fileID);
        }

      const int nsubtypes = vlistNsubtypes(vlistID);
      if (nsubtypes > 1) cdoWarning("Subtypes are unsupported, the processing results are possibly wrong!");

      if (CdoDefault::TaxisType != CDI_UNDEFID) taxisDefType(vlistInqTaxis(vlistID), CdoDefault::TaxisType);

      m_vlistID = vlistID;
    }

  return vlistID;
}

void
Pstream::defVlist(int p_vlistID)
{
#ifdef HAVE_LIBPTHREAD
  if (ispipe)
    {
      Debug(PSTREAM, "%s pstreamID %d", pipe->name, m_cdoStreamID);
      int vlistIDcp = vlistDuplicate(p_vlistID);
      pipe->pipeDefVlist(m_vlistID, vlistIDcp);
    }
  else
#endif
    {
      if (CdoDefault::DataType != CDI_UNDEFID)
        {
          int varID, nvars = vlistNvars(p_vlistID);

          for (varID = 0; varID < nvars; ++varID) vlistDefVarDatatype(p_vlistID, varID, CdoDefault::DataType);

          if (CdoDefault::DataType == CDI_DATATYPE_FLT64 || CdoDefault::DataType == CDI_DATATYPE_FLT32)
            {
              for (varID = 0; varID < nvars; varID++)
                {
                  vlistDefVarAddoffset(p_vlistID, varID, 0.0);
                  vlistDefVarScalefactor(p_vlistID, varID, 1.0);
                }
            }
        }

      if (Options::cdoChunkType != CDI_UNDEFID)
        {
          const int nvars = vlistNvars(p_vlistID);
          for (int varID = 0; varID < nvars; ++varID) vlistDefVarChunkType(p_vlistID, varID, Options::cdoChunkType);
        }

      if (Options::CMOR_Mode)
        {
          cdo_def_tracking_id(p_vlistID, "tracking_id");
          cdo_def_creation_date(p_vlistID);
        }

      if (Options::VersionInfo) cdiDefAttTxt(p_vlistID, CDI_GLOBAL, "CDO", (int) strlen(cdoComment()), cdoComment());

#ifdef _OPENMP
      if (Threading::ompNumThreads > 1)
        cdiDefAttInt(p_vlistID, CDI_GLOBAL, "cdo_openmp_thread_number", CDI_DATATYPE_INT32, 1, &Threading::ompNumThreads);
#endif
      defVarList(p_vlistID);

      if (processNum == 1 && Threading::ompNumThreads == 1) timer_start(timer_write);
      streamDefVlistLocked(m_fileID, p_vlistID);
      if (processNum == 1 && Threading::ompNumThreads == 1) timer_stop(timer_write);
    }
}

void
Pstream::inqRecord(int *varID, int *levelID)
{
#ifdef HAVE_LIBPTHREAD
  if (ispipe)
    {
      Debug(PSTREAM, "%s pstreamID %d", pipe->name, m_cdoStreamID);
      pipe->pipeInqRecord(varID, levelID);
    }

  else
#endif
    {
      if (processNum == 1 && Threading::ompNumThreads == 1) timer_start(timer_read);
      streamInqRecLocked(m_fileID, varID, levelID);
      if (processNum == 1 && Threading::ompNumThreads == 1) timer_stop(timer_read);
    }
  m_varID = *varID;
}

void
Pstream::defRecord(int varID, int levelID)
{
  m_varID = varID;

#ifdef HAVE_LIBPTHREAD
  if (ispipe)
    {
      Debug(PSTREAM, "%s pstreamID %d", pipe->name, m_cdoStreamID);

      pipe->pipeDefRecord(varID, levelID);
    }
  else
#endif
    {
      if (processNum == 1 && Threading::ompNumThreads == 1) timer_start(timer_write);
      streamDefRecLocked(m_fileID, varID, levelID);
      if (processNum == 1 && Threading::ompNumThreads == 1) timer_stop(timer_write);
    }
}

void
Pstream::readRecord(float *data, size_t *nmiss)
{
#ifdef HAVE_LIBPTHREAD

  if (ispipe)
    {
      cdoAbort("pipeReadRecord not implemented for memtype float!");
      // pipeReadRecord(pstreamptr, data, nmiss);
    }
  else
#endif
    {
      if (processNum == 1 && Threading::ompNumThreads == 1) timer_start(timer_read);
      streamReadrecordFloatLocked(m_fileID, data, nmiss);
      if (processNum == 1 && Threading::ompNumThreads == 1) timer_stop(timer_read);
    }
}

void
Pstream::readRecord(double *data, size_t *nmiss)
{
#ifdef HAVE_LIBPTHREAD
  if (ispipe)
    {
      Debug(PSTREAM, "%s pstreamID %d", pipe->name, m_cdoStreamID);

      m_nvals += pipe->pipeReadRecord(m_vlistID, data, nmiss);
    }
  else
#endif
    {
      if (processNum == 1 && Threading::ompNumThreads == 1) timer_start(timer_read);
      streamReadrecordDoubleLocked(m_fileID, data, nmiss);
      m_nvals += gridInqSize(vlistInqVarGrid(m_vlistID, m_varID));
      if (processNum == 1 && Threading::ompNumThreads == 1) timer_stop(timer_read);
    }
}

void
Pstream::writeRecord(double *p_data, size_t p_nmiss)
{

#ifdef HAVE_LIBPTHREAD
  if (ispipe)
    {
      Debug(PSTREAM, "%s pstreamID %d", pipe->name, m_cdoStreamID);

      pipe->pipeWriteRecord(p_data, p_nmiss);
    }
  else
#endif
    {
      int varID = m_varID;
      if (processNum == 1 && Threading::ompNumThreads == 1) timer_start(timer_write);

      if (varID < (int) m_varlist.size())
        if (m_varlist[varID].check_datarange) m_varlist[varID].checkDatarange(p_data, p_nmiss);

      streamWriteRecordDoubleLocked(m_fileID, p_data, p_nmiss);

      if (processNum == 1 && Threading::ompNumThreads == 1) timer_stop(timer_write);
    }
}

void
Pstream::writeRecord(float *p_data, size_t p_nmiss)
{
#ifdef HAVE_LIBPTHREAD
  if (ispipe)
    {
      Debug(PSTREAM, "%s pstreamID %d", pipe->name, m_cdoStreamID);

      cdoAbort("pipeWriteRecord not implemented for memtype float!");
    }
  else
#endif
    {
      if (processNum == 1 && Threading::ompNumThreads == 1) timer_start(timer_write);

      streamWriteRecordFloatLocked(m_fileID, p_data, p_nmiss);

      if (processNum == 1 && Threading::ompNumThreads == 1) timer_stop(timer_write);
    }
}

// prepares and sets new timestep for further processing
int
Pstream::inqTimestep(int p_tsID)
{
  int nrecs;
  if (ispipe)
    {
      nrecs = pipe->pipeInqTimestep(p_tsID);
    }
  else
    {
      if (processNum == 1 && Threading::ompNumThreads == 1) timer_start(timer_read);
      nrecs = streamInqTimeStepLocked(m_fileID, p_tsID);
      if (processNum == 1 && Threading::ompNumThreads == 1) timer_stop(timer_read);

      if (p_tsID == 0 && CdoDefault::TaxisType != CDI_UNDEFID)
        {
          taxisDefType(vlistInqTaxis(m_vlistID), CdoDefault::TaxisType);
        }
      // updating current timestep
      if (nrecs && p_tsID != m_tsID)
        {
          m_tsID = p_tsID;
        }
    }
  return nrecs;
}

void
Pstream::defTimestep(int p_tsID)
{
#ifdef HAVE_LIBPTHREAD
  if (ispipe)
    {
      Debug(PSTREAM, "%s pstreamID %d", pipe->name, m_cdoStreamID);

      pipe->pipeDefTimestep(m_vlistID, p_tsID);
    }
  else
#endif
    {
      if (processNum == 1 && Threading::ompNumThreads == 1) timer_start(timer_write);
      // don't use sync -> very slow on GPFS
      //  if ( p_tsID > 0 ) streamSync(fileID);

      streamDefTimeStepLocked(m_fileID, p_tsID);

      if (processNum == 1 && Threading::ompNumThreads == 1) timer_stop(timer_write);
    }
}

void
Pstream::copyRecord(CdoStreamID src)
{
  if (ispipe || src->ispipe) cdoAbort("This operator can't be combined with other operators!");
  streamCopyRecordLocked(m_fileID, src->m_fileID);
}

int
Pstream::inqFileType()
{
  return
#ifdef HAVE_LIBPTHREAD
      ispipe ? m_filetype :
#endif
             streamInqFiletype(m_fileID);
}

int
Pstream::inqByteorder()
{
  return
#ifdef HAVE_LIBPTHREAD
      ispipe ? m_filetype :
#endif
             streamInqByteorder(m_fileID);
}

int
Pstream::getFileID()
{
  return m_fileID;
}


//- - - - - - - - - - - - - - - - - - - - - - - - has to be moved
void
pstreamDebug(int debug)
{
  PSTREAM = debug;
}
