/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/
#include <cdi.h>
#include "cdo_options.h"
#include <vector>

#include "cdo_output.h"
#include "dmemory.h"
#include "compare.h"
#include "cdi_lockedIO.h"

bool *
cdo_read_timestepmask(const char *maskfile, int *n)
{
  *n = 0;

  int streamID = streamOpenReadLocked(maskfile);
  int vlistID = streamInqVlist(streamID);

  int nvars = vlistNvars(vlistID);
  if (nvars > 1) cdoAbort("timestepmask %s contains more than one variable!", maskfile);

  size_t gridsize = gridInqSize(vlistInqVarGrid(vlistID, 0));
  if (gridsize > 1) cdoAbort("timestepmask %s has more than one gridpoint!", maskfile);

  int nlev = zaxisInqSize(vlistInqVarZaxis(vlistID, 0));
  if (nlev > 1) cdoAbort("timestepmask %s has more than one level!", maskfile);

  int nts = vlistNtsteps(vlistID);
  if (nts == -1)
    {
      nts = 0;
      while (streamInqTimestep(streamID, nts)) nts++;

      if (Options::cdoVerbose) cdoPrint("%s: counted %i timeSteps in %s", __func__, nts, maskfile);

      streamClose(streamID);
      streamID = streamOpenReadLocked(maskfile);
    }
  else if (Options::cdoVerbose)
    cdoPrint("%s: found %i timeSteps in %s", __func__, nts, maskfile);

  *n = nts;
  bool *imask = (bool *) Malloc(nts * sizeof(bool));

  int nrecs;
  int tsID = 0;
  while ((nrecs = streamInqTimestep(streamID, tsID)))
    {
      if (nrecs != 1) cdoAbort("Internal error; unexprected number of records!");

      int varID, levelID;
      size_t nmiss;
      double value;
      streamInqRecord(streamID, &varID, &levelID);
      streamReadRecord(streamID, &value, &nmiss);

      imask[tsID] = !(nmiss || IS_EQUAL(value, 0));

      tsID++;
    }

  streamClose(streamID);

  return imask;
}

bool *
cdo_read_mask(const char *maskfile, size_t *n)
{
  *n = 0;

  int streamID = streamOpenReadLocked(maskfile);
  int vlistID = streamInqVlist(streamID);

  int nvars = vlistNvars(vlistID);
  if (nvars > 1) cdoAbort("Mask %s contains more than one variable!", maskfile);

  size_t gridsize = gridInqSize(vlistInqVarGrid(vlistID, 0));

  int nlev = zaxisInqSize(vlistInqVarZaxis(vlistID, 0));
  if (nlev > 1) cdoAbort("Mask %s has more than one level!", maskfile);

  *n = gridsize;
  bool *imask = (bool *) Malloc(gridsize * sizeof(bool));
  std::vector<double> array(gridsize);

  int nrecs = streamInqTimestep(streamID, 0);
  if (nrecs != 1) cdoAbort("Internal error; unexprected number of records!");

  int varID, levelID;
  size_t nmiss;
  streamInqRecord(streamID, &varID, &levelID);
  streamReadRecord(streamID, array.data(), &nmiss);
  streamClose(streamID);

  for (size_t i = 0; i < gridsize; ++i) imask[i] = IS_NOT_EQUAL(array[i], 0);

  return imask;
}

int *
cdo_read_index(const char *indexfile, size_t *n)
{
  *n = 0;

  int streamID = streamOpenReadLocked(indexfile);
  int vlistID = streamInqVlist(streamID);

  int nvars = vlistNvars(vlistID);
  if (nvars > 1) cdoAbort("Index file %s contains more than one variable!", indexfile);

  size_t gridsize = gridInqSize(vlistInqVarGrid(vlistID, 0));

  int nlev = zaxisInqSize(vlistInqVarZaxis(vlistID, 0));
  if (nlev > 1) cdoAbort("Index file %s has more than one level!", indexfile);

  *n = gridsize;
  int *index = (int *) Malloc(gridsize * sizeof(int));
  std::vector<double> array(gridsize);

  int nrecs = streamInqTimestep(streamID, 0);
  if (nrecs != 1) cdoAbort("Internal error; unexprected number of records!");

  int varID, levelID;
  size_t nmiss;
  streamInqRecord(streamID, &varID, &levelID);
  streamReadRecord(streamID, array.data(), &nmiss);
  streamClose(streamID);

  for (size_t i = 0; i < gridsize; ++i) index[i] = (int) lround(array[i]) - 1;

  return index;
}
