/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2006 Brockmann Consult
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Ydrunpctl    ydrunpctl         Multi-year daily running percentiles
*/

#include <cdi.h>

#include "cdo_int.h"
#include "param_conversion.h"
#include "calendar.h"

#include "percentiles_hist.h"
#include "percentiles.h"
#include "datetime.h"

#define NDAY 373

int
getmonthday(int64_t date)
{
  int year, month, day;
  cdiDecodeDate(date, &year, &month, &day);
  return month * 100 + day;
}

void *
Ydrunpctl(void *process)
{
  int varID;
  int gridID;
  int nrecs;
  int levelID;
  int inp, its;
  size_t nmiss;
  int nlevels;
  int year, month, day;
  int64_t vdates1[NDAY], vdates2[NDAY];
  int vtimes1[NDAY] /*, vtimes2[NDAY]*/;
  int nsets[NDAY];
  FieldVector2D vars2[NDAY];
  HISTOGRAM_SET *hsets[NDAY];

  cdoInitialize(process);
  cdoOperatorAdd("ydrunpctl", func_pctl, 0, nullptr);

  operatorInputArg("percentile number, number of timesteps");
  operatorCheckArgc(2);
  const double pn = parameter2double(operatorArgv()[0]);
  percentile_check_number(pn);
  const int ndates = parameter2int(operatorArgv()[1]);

  for (int dayoy = 0; dayoy < NDAY; dayoy++) nsets[dayoy] = 0;

  CdoStreamID streamID1 = cdoOpenRead(0);
  CdoStreamID streamID2 = cdoOpenRead(1);
  CdoStreamID streamID3 = cdoOpenRead(2);

  const int vlistID1 = cdoStreamInqVlist(streamID1);
  const int vlistID2 = cdoStreamInqVlist(streamID2);
  const int vlistID3 = cdoStreamInqVlist(streamID3);
  const int vlistID4 = vlistDuplicate(vlistID1);

  vlistCompare(vlistID1, vlistID2, CMP_ALL);
  vlistCompare(vlistID1, vlistID3, CMP_ALL);

  const int taxisID1 = vlistInqTaxis(vlistID1);
  const int taxisID2 = vlistInqTaxis(vlistID2);
  const int taxisID3 = vlistInqTaxis(vlistID3);
  /* TODO - check that time axes 2 and 3 are equal */

  const int taxisID4 = taxisDuplicate(taxisID1);
  if (taxisHasBounds(taxisID4)) taxisDeleteBounds(taxisID4);
  vlistDefTaxis(vlistID4, taxisID4);

  const int calendar = taxisInqCalendar(taxisID1);
  const int dpy = calendar_dpy(calendar);

  CdoStreamID streamID4 = cdoOpenWrite(3);
  cdoDefVlist(streamID4, vlistID4);

  const int nvars = vlistNvars(vlistID1);

  const int maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recList(maxrecs);

  const size_t gridsizemax = vlistGridsizeMax(vlistID1);

  Field field;
  field.resize(gridsizemax);

  std::vector<CdoDateTime> datetime(ndates + 1);

  FieldVector3D vars1(ndates + 1);
  for (its = 0; its < ndates; its++) fieldsFromVlist(vlistID1, vars1[its], FIELD_VEC);

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID2, tsID)))
    {
      if (nrecs != cdoStreamInqTimestep(streamID3, tsID))
        cdoAbort("Number of records at time step %d of %s and %s differ!", tsID + 1, cdoGetStreamName(1), cdoGetStreamName(2));

      const int64_t vdate = taxisInqVdate(taxisID2);
      const int vtime = taxisInqVtime(taxisID2);

      if (vdate != taxisInqVdate(taxisID3))
        cdoAbort("Verification dates at time step %d of %s and %s differ!", tsID + 1, cdoGetStreamName(1), cdoGetStreamName(2));

      if (Options::cdoVerbose) cdoPrint("process timestep: %d %d %d", tsID + 1, vdate, vtime);

      cdiDecodeDate(vdate, &year, &month, &day);

      const int dayoy = (month >= 1 && month <= 12) ? (month - 1) * 31 + day : 0;
      if (dayoy < 0 || dayoy >= NDAY) cdoAbort("Day %d out of range!", dayoy);

      vdates2[dayoy] = vdate;
      // vtimes2[dayoy] = vtime;

      if (!vars2[dayoy].size())
        {
          fieldsFromVlist(vlistID2, vars2[dayoy], FIELD_VEC);
          hsets[dayoy] = hsetCreate(nvars);

          for (varID = 0; varID < nvars; varID++)
            {
              gridID = vlistInqVarGrid(vlistID2, varID);
              nlevels = zaxisInqSize(vlistInqVarZaxis(vlistID2, varID));

              hsetCreateVarLevels(hsets[dayoy], varID, nlevels, gridID);
            }
        }

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID2, &varID, &levelID);
          cdoReadRecord(streamID2, vars2[dayoy][varID][levelID].vec.data(), &nmiss);
          vars2[dayoy][varID][levelID].nmiss = nmiss;
        }

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID3, &varID, &levelID);
          cdoReadRecord(streamID3, field.vec.data(), &field.nmiss);
          field.grid = vars2[dayoy][varID][levelID].grid;
          field.missval = vars2[dayoy][varID][levelID].missval;

          hsetDefVarLevelBounds(hsets[dayoy], varID, levelID, &vars2[dayoy][varID][levelID], &field);
        }

      tsID++;
    }

  for (tsID = 0; tsID < ndates; tsID++)
    {
      nrecs = cdoStreamInqTimestep(streamID1, tsID);
      if (nrecs == 0) cdoAbort("File has less then %d timesteps!", ndates);

      datetime[tsID].date = taxisInqVdate(taxisID1);
      datetime[tsID].time = taxisInqVtime(taxisID1);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          if (tsID == 0)
            {
              recList[recID].varID = varID;
              recList[recID].levelID = levelID;
              recList[recID].lconst = vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT;
            }

          cdoReadRecord(streamID1, vars1[tsID][varID][levelID].vec.data(), &nmiss);
          vars1[tsID][varID][levelID].nmiss = nmiss;
        }
    }

  while (true)
    {
      datetime_avg(dpy, ndates, datetime.data());

      const int64_t vdate = datetime[ndates].date;
      const int vtime = datetime[ndates].time;

      cdiDecodeDate(vdate, &year, &month, &day);

      const int dayoy = (month >= 1 && month <= 12) ? (month - 1) * 31 + day : 0;
      if (dayoy < 0 || dayoy >= NDAY) cdoAbort("Day %d out of range!", dayoy);

      vdates1[dayoy] = vdate;
      vtimes1[dayoy] = vtime;

      if (!vars2[dayoy].size()) cdoAbort("No data for day %d in %s and %s", dayoy, cdoGetStreamName(1), cdoGetStreamName(2));

      for (varID = 0; varID < nvars; varID++)
        {
          if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT) continue;
          nlevels = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));

          for (levelID = 0; levelID < nlevels; levelID++)
            for (inp = 0; inp < ndates; inp++) hsetAddVarLevelValues(hsets[dayoy], varID, levelID, &vars1[inp][varID][levelID]);
        }

      datetime[ndates] = datetime[0];
      vars1[ndates] = vars1[0];

      for (inp = 0; inp < ndates; inp++)
        {
          datetime[inp] = datetime[inp + 1];
          vars1[inp] = vars1[inp + 1];
        }

      nrecs = cdoStreamInqTimestep(streamID1, tsID);
      if (nrecs == 0) break;

      datetime[ndates - 1].date = taxisInqVdate(taxisID1);
      datetime[ndates - 1].time = taxisInqVtime(taxisID1);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          cdoReadRecord(streamID1, vars1[ndates - 1][varID][levelID].vec.data(), &nmiss);
          vars1[ndates - 1][varID][levelID].nmiss = nmiss;
        }

      nsets[dayoy] += ndates;
      tsID++;
    }
  /*
  int outyear = 1e9;
  for ( dayoy = 0; dayoy < NDAY; dayoy++ )
    if ( nsets[dayoy] )
      {
        int year, month, day;
        cdiDecodeDate(vdates1[dayoy], &year, &month, &day);
        if ( year < outyear ) outyear = year;
      }

  for ( dayoy = 0; dayoy < NDAY; dayoy++ )
    if ( nsets[dayoy] )
      {
        int year, month, day;
        cdiDecodeDate(vdates1[dayoy], &year, &month, &day);
        vdates1[dayoy] = cdiEncodeDate(outyear, month, day);
      }
  */
  int otsID = 0;
  for (int dayoy = 0; dayoy < NDAY; dayoy++)
    if (nsets[dayoy])
      {
        if (getmonthday(vdates1[dayoy]) != getmonthday(vdates2[dayoy]))
          cdoAbort("Verification dates for day %d of %s, %s and %s are different!", dayoy, cdoGetStreamName(0),
                   cdoGetStreamName(1));

        for (varID = 0; varID < nvars; varID++)
          {
            if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT) continue;
            nlevels = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));

            for (levelID = 0; levelID < nlevels; levelID++)
              hsetGetVarLevelPercentiles(&vars2[dayoy][varID][levelID], hsets[dayoy], varID, levelID, pn);
          }

        taxisDefVdate(taxisID4, vdates1[dayoy]);
        taxisDefVtime(taxisID4, vtimes1[dayoy]);
        cdoDefTimestep(streamID4, otsID);

        for (int recID = 0; recID < maxrecs; recID++)
          {
            if (otsID && recList[recID].lconst) continue;

            const int varID = recList[recID].varID;
            const int levelID = recList[recID].levelID;
            cdoDefRecord(streamID4, varID, levelID);
            cdoWriteRecord(streamID4, vars2[dayoy][varID][levelID].vec.data(), vars2[dayoy][varID][levelID].nmiss);
          }

        otsID++;
      }

  for (int dayoy = 0; dayoy < NDAY; dayoy++)
    {
      if (vars2[dayoy].size())
        {
          hsetDestroy(hsets[dayoy]);
        }
    }


  cdoStreamClose(streamID4);
  cdoStreamClose(streamID3);
  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
