/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2006 Brockmann Consult
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Ydaypctl   ydaypctl        Multi-year daily percentiles
*/

#include <cdi.h>

#include "cdo_int.h"
#include "param_conversion.h"

#include "percentiles_hist.h"
#include "percentiles.h"

#define NDAY 373

int getmonthday(int64_t date);

void *
Ydaypctl(void *process)
{
  int varID;
  int gridID;
  int year, month, day;
  int nrecs;
  int levelID;
  size_t nmiss;
  int nlevels;
  int64_t vdates1[NDAY], vdates2[NDAY];
  int vtimes1[NDAY];
  long nsets[NDAY];
  FieldVector2D vars1[NDAY];
  HISTOGRAM_SET *hsets[NDAY];

  cdoInitialize(process);
  cdoOperatorAdd("ydaypctl", func_pctl, 0, nullptr);

  operatorInputArg("percentile number");
  const double pn = parameter2double(operatorArgv()[0]);
  percentile_check_number(pn);

  for (int dayoy = 0; dayoy < NDAY; dayoy++)
    {
      hsets[dayoy] = nullptr;
      nsets[dayoy] = 0;
    }

  CdoStreamID streamID1 = cdoOpenRead(0);
  CdoStreamID streamID2 = cdoOpenRead(1);
  CdoStreamID streamID3 = cdoOpenRead(2);

  const int vlistID1 = cdoStreamInqVlist(streamID1);
  const int vlistID2 = cdoStreamInqVlist(streamID2);
  const int vlistID3 = cdoStreamInqVlist(streamID3);
  const int vlistID4 = vlistDuplicate(vlistID1);

  vlistCompare(vlistID1, vlistID2, CMP_ALL);
  vlistCompare(vlistID1, vlistID3, CMP_ALL);

  const int taxisID1 = vlistInqTaxis(vlistID1);
  const int taxisID2 = vlistInqTaxis(vlistID2);
  const int taxisID3 = vlistInqTaxis(vlistID3);
  /* TODO - check that time axes 2 and 3 are equal */

  const int taxisID4 = taxisDuplicate(taxisID1);
  if (taxisHasBounds(taxisID4)) taxisDeleteBounds(taxisID4);
  vlistDefTaxis(vlistID4, taxisID4);

  CdoStreamID streamID4 = cdoOpenWrite(3);
  cdoDefVlist(streamID4, vlistID4);

  const int nvars = vlistNvars(vlistID1);

  const int maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recList(maxrecs);

  const size_t gridsizemax = vlistGridsizeMax(vlistID1);

  Field field;
  field.resize(gridsizemax);

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID2, tsID)))
    {
      if (nrecs != cdoStreamInqTimestep(streamID3, tsID))
        cdoAbort("Number of records at time step %d of %s and %s differ!", tsID + 1, cdoGetStreamName(1), cdoGetStreamName(2));

      const int64_t vdate = taxisInqVdate(taxisID2);
      const int vtime = taxisInqVtime(taxisID2);

      if (vdate != taxisInqVdate(taxisID3))
        cdoAbort("Verification dates at time step %d of %s and %s differ!", tsID + 1, cdoGetStreamName(1), cdoGetStreamName(2));

      if (Options::cdoVerbose) cdoPrint("process timestep: %d %d %d", tsID + 1, vdate, vtime);

      cdiDecodeDate(vdate, &year, &month, &day);

      const int dayoy = (month >= 1 && month <= 12) ? (month - 1) * 31 + day : 0;
      if (dayoy < 0 || dayoy >= NDAY) cdoAbort("Day %d out of range!", dayoy);

      vdates2[dayoy] = vdate;

      if (!vars1[dayoy].size())
        {
          fieldsFromVlist(vlistID1, vars1[dayoy], FIELD_VEC);
          hsets[dayoy] = hsetCreate(nvars);

          for (varID = 0; varID < nvars; varID++)
            {
              gridID = vlistInqVarGrid(vlistID1, varID);
              nlevels = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));

              hsetCreateVarLevels(hsets[dayoy], varID, nlevels, gridID);
            }
        }

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID2, &varID, &levelID);
          cdoReadRecord(streamID2, vars1[dayoy][varID][levelID].vec.data(), &nmiss);
          vars1[dayoy][varID][levelID].nmiss = nmiss;
        }

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID3, &varID, &levelID);
          cdoReadRecord(streamID3, field.vec.data(), &field.nmiss);
          field.grid = vars1[dayoy][varID][levelID].grid;
          field.missval = vars1[dayoy][varID][levelID].missval;

          hsetDefVarLevelBounds(hsets[dayoy], varID, levelID, &vars1[dayoy][varID][levelID], &field);
        }

      tsID++;
    }

  tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      const int64_t vdate = taxisInqVdate(taxisID1);
      const int vtime = taxisInqVtime(taxisID1);

      if (Options::cdoVerbose) cdoPrint("process timestep: %d %d %d", tsID + 1, vdate, vtime);

      cdiDecodeDate(vdate, &year, &month, &day);

      const int dayoy = (month >= 1 && month <= 12) ? (month - 1) * 31 + day : 0;
      if (dayoy < 0 || dayoy >= NDAY) cdoAbort("Day %d out of range!", dayoy);

      vdates1[dayoy] = vdate;
      vtimes1[dayoy] = vtime;

      if (!vars1[dayoy].size()) cdoAbort("No data for day %d in %s and %s", dayoy, cdoGetStreamName(1), cdoGetStreamName(2));

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          if (tsID == 0)
            {
              recList[recID].varID = varID;
              recList[recID].levelID = levelID;
              recList[recID].lconst = vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT;
            }

          cdoReadRecord(streamID1, vars1[dayoy][varID][levelID].vec.data(), &nmiss);
          vars1[dayoy][varID][levelID].nmiss = nmiss;

          hsetAddVarLevelValues(hsets[dayoy], varID, levelID, &vars1[dayoy][varID][levelID]);
        }

      nsets[dayoy]++;
      tsID++;
    }

  int otsID = 0;
  for (int dayoy = 0; dayoy < NDAY; dayoy++)
    if (nsets[dayoy])
      {
        if (getmonthday(vdates1[dayoy]) != getmonthday(vdates2[dayoy]))
          cdoAbort("Verification dates for the day %d of %s and %s are different!", dayoy, cdoGetStreamName(0),
                   cdoGetStreamName(1));

        for (varID = 0; varID < nvars; varID++)
          {
            if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT) continue;
            nlevels = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));

            for (levelID = 0; levelID < nlevels; levelID++)
              hsetGetVarLevelPercentiles(&vars1[dayoy][varID][levelID], hsets[dayoy], varID, levelID, pn);
          }

        taxisDefVdate(taxisID4, vdates1[dayoy]);
        taxisDefVtime(taxisID4, vtimes1[dayoy]);
        cdoDefTimestep(streamID4, otsID);

        for (int recID = 0; recID < maxrecs; recID++)
          {
            if (otsID && recList[recID].lconst) continue;

            const int varID = recList[recID].varID;
            const int levelID = recList[recID].levelID;
            cdoDefRecord(streamID4, varID, levelID);
            cdoWriteRecord(streamID4, vars1[dayoy][varID][levelID].vec.data(), vars1[dayoy][varID][levelID].nmiss);
          }

        otsID++;
      }

  for (int dayoy = 0; dayoy < NDAY; dayoy++)
    {
      if (vars1[dayoy].size())
        {
          hsetDestroy(hsets[dayoy]);
        }
    }

  cdoStreamClose(streamID4);
  cdoStreamClose(streamID3);
  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
