/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Vertcum    vertcum         Vertical cumulative
      Vertcum    vertcumhl       Vertical cumulative on hybrid sigma half level
*/

#include <cdi.h>

#include "cdo_int.h"

#define IS_SURFACE_LEVEL(zaxisID) (zaxisInqType(zaxisID) == ZAXIS_SURFACE && zaxisInqSize(zaxisID) == 1)

static void
add_vars_mv(size_t gridsize, double missval, const std::vector<double> &var1, const std::vector<double> &var2, std::vector<double> &var3)
{
  const double missval1 = missval;
  const double missval2 = missval;
  /*
  for ( size_t i = 0; i < gridsize; ++i )
    var3[i] = ADDMN(var2[i], var1[i]);
  */
  for (size_t i = 0; i < gridsize; ++i)
    {
      var3[i] = var2[i];
      if (!DBL_IS_EQUAL(var1[i], missval1))
        {
          if (!DBL_IS_EQUAL(var2[i], missval2))
            var3[i] += var1[i];
          else
            var3[i] = var1[i];
        }
    }
}

void *
Vertcum(void *process)
{
  int nrecs;
  int nlevshl = 0;
  int varID, levelID;
  size_t nmiss;

  cdoInitialize(process);

  // clang-format off
                  cdoOperatorAdd("vertcum",    0,  0, nullptr);
  int VERTCUMHL = cdoOperatorAdd("vertcumhl",  0,  0, nullptr);
  // clang-format on

  const int operatorID = cdoOperatorID();

  CdoStreamID streamID1 = cdoOpenRead(0);

  const int vlistID1 = cdoStreamInqVlist(streamID1);
  const int vlistID2 = vlistDuplicate(vlistID1);

  int zaxisIDhl = -1;

  if (operatorID == VERTCUMHL)
    {
      double *vct = nullptr;
      bool lhybrid = false;
      const int nzaxis = vlistNzaxis(vlistID1);
      for (int i = 0; i < nzaxis; ++i)
        {
          const int zaxisID = vlistZaxis(vlistID1, i);
          const int nlevs = zaxisInqSize(zaxisID);

          if (zaxisInqType(zaxisID) == ZAXIS_HYBRID && nlevs > 1)
            {
              int nvct = zaxisInqVctSize(zaxisID);
              if (nlevs == (nvct / 2 - 1))
                {
                  if (lhybrid == false)
                    {
                      lhybrid = true;
                      nlevshl = nlevs + 1;

                      double *vct = (double *) Malloc(nvct * sizeof(double));
                      zaxisInqVct(zaxisID, vct);

                      zaxisIDhl = zaxisCreate(ZAXIS_HYBRID_HALF, nlevshl);
                      double *levels = (double *) Malloc(nlevshl * sizeof(double));
                      for (levelID = 0; levelID < nlevshl; ++levelID) levels[levelID] = levelID + 1;
                      zaxisDefLevels(zaxisIDhl, levels);
                      Free(levels);
                      zaxisDefVct(zaxisIDhl, nvct, vct);
                      vlistChangeZaxisIndex(vlistID2, i, zaxisIDhl);
                    }
                  else if (vct)
                    {
                      if (memcmp(vct, zaxisInqVctPtr(zaxisID), nvct * sizeof(double)) == 0)
                        vlistChangeZaxisIndex(vlistID2, i, zaxisIDhl);
                    }
                }
            }
        }
      if (vct) Free(vct);
    }

  VarList varList1, varList2;
  varListInit(varList1, vlistID1);
  varListInit(varList2, vlistID2);

  const int nvars = vlistNvars(vlistID1);
  std::vector<std::vector<size_t>> varnmiss(nvars);
  std::vector<std::vector<std::vector<double>>> vardata1(nvars);
  std::vector<std::vector<std::vector<double>>> vardata2(nvars);

  for (varID = 0; varID < nvars; varID++)
    {
      const size_t gridsize = varList1[varID].gridsize;
      const int nlevs = varList1[varID].nlevels;
      const int nlevs2 = varList2[varID].nlevels;

      varnmiss[varID].resize(nlevs);
      vardata1[varID].resize(nlevs);
      vardata2[varID].resize(nlevs2);
      for (levelID = 0; levelID < nlevs; ++levelID) vardata1[varID][levelID].resize(gridsize);
      for (levelID = 0; levelID < nlevs2; ++levelID) vardata2[varID][levelID].resize(gridsize);
    }

  const int taxisID1 = vlistInqTaxis(vlistID1);
  const int taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  CdoStreamID streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          cdoReadRecord(streamID1, &vardata1[varID][levelID][0], &varnmiss[varID][levelID]);
        }

      for (varID = 0; varID < nvars; ++varID)
        {
          const double missval = varList2[varID].missval;
          const size_t gridsize = varList2[varID].gridsize;
          const int nlevs2 = varList2[varID].nlevels;

          if (operatorID == VERTCUMHL && nlevs2 == nlevshl)
            {
              for (size_t i = 0; i < gridsize; ++i) vardata2[varID][0][i] = 0;
            }
          else
            {
              for (size_t i = 0; i < gridsize; ++i) vardata2[varID][0][i] = vardata1[varID][0][i];
            }

          for (levelID = 1; levelID < nlevs2; ++levelID)
            {
              if (operatorID == VERTCUMHL && nlevs2 == nlevshl)
                add_vars_mv(gridsize, missval, vardata1[varID][levelID - 1], vardata2[varID][levelID - 1],
                            vardata2[varID][levelID]);
              else
                add_vars_mv(gridsize, missval, vardata1[varID][levelID], vardata2[varID][levelID - 1], vardata2[varID][levelID]);
            }

          if (operatorID == VERTCUMHL && nlevs2 == nlevshl)
            {
              std::vector<double> &var1 = vardata2[varID][nlevs2 - 1];
              for (levelID = 0; levelID < nlevs2; ++levelID)
                {
                  std::vector<double> &var2 = vardata2[varID][levelID];
                  for (size_t i = 0; i < gridsize; ++i)
                    {
                      if (IS_NOT_EQUAL(var1[i], 0))
                        var2[i] /= var1[i];
                      else
                        var2[i] = 0;
                    }
                }
            }
        }

      for (varID = 0; varID < nvars; ++varID)
        {
          const double missval = varList2[varID].missval;
          const size_t gridsize = varList2[varID].gridsize;
          const int nlevs2 = varList2[varID].nlevels;
          for (levelID = 0; levelID < nlevs2; ++levelID)
            {
              double *single = vardata2[varID][levelID].data();
              nmiss = arrayNumMV(gridsize, single, missval);
              cdoDefRecord(streamID2, varID, levelID);
              cdoWriteRecord(streamID2, single, nmiss);
            }
        }

      tsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  vlistDestroy(vlistID2);

  cdoFinish();

  return nullptr;
}
