/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Ensstat    ensrange        Ensemble range
      Ensstat    ensmin          Ensemble minimum
      Ensstat    ensmax          Ensemble maximum
      Ensstat    enssum          Ensemble sum
      Ensstat    ensmean         Ensemble mean
      Ensstat    ensavg          Ensemble average
      Ensstat    ensstd          Ensemble standard deviation
      Ensstat    ensstd1         Ensemble standard deviation
      Ensstat    ensvar          Ensemble variance
      Ensstat    ensvar1         Ensemble variance
      Ensstat    enspctl         Ensemble percentiles
*/

#include <cdi.h>

#include "cdo_int.h"
#include "param_conversion.h"
#include "cdo_task.h"

#include "percentiles.h"
#include "cdo_options.h"
#include "util_files.h"
#include "cimdOmp.h"

struct ens_file_t
{
  CdoStreamID streamID;
  int vlistID;
  size_t nmiss[2];
  double missval[2];
  double *array[2];
};

struct ensstat_arg_t
{
  int t;
  int varID[2];
  int levelID[2];
  int vlistID1;
  CdoStreamID streamID2;
  int nfiles;
  ens_file_t *efData;
  double *array2Data;
  double *count2Data;
  Field *fieldsData;
  int operfunc;
  double pn;
  bool lpctl;
  bool withCountData;
  int nvars;
};

static void *
ensstat_func(void *ensarg)
{
  if (CDO_task) cdo_omp_set_num_threads(Threading::ompNumThreads);

  ensstat_arg_t *arg = (ensstat_arg_t *) ensarg;
  const int t = arg->t;
  const int nfiles = arg->nfiles;
  ens_file_t *ef = arg->efData;
  Field *fields = arg->fieldsData;
  double *array2 = arg->array2Data;
  double *count2 = arg->count2Data;
  const bool withCountData = arg->withCountData;

  bool lmiss = false;
  for (int fileID = 0; fileID < nfiles; fileID++)
    if (ef[fileID].nmiss[t] > 0) lmiss = true;

  const int gridID = vlistInqVarGrid(arg->vlistID1, arg->varID[t]);
  const size_t gridsize = gridInqSize(gridID);
  const double missval = vlistInqVarMissval(arg->vlistID1, arg->varID[t]);

  size_t nmiss = 0;
#ifdef HAVE_OPENMP4
#pragma omp parallel for default(shared) reduction(+ : nmiss)
#endif
  for (size_t i = 0; i < gridsize; ++i)
    {
      const int ompthID = cdo_omp_get_thread_num();

      fields[ompthID].missval = missval;
      fields[ompthID].nmiss = 0;
      for (int fileID = 0; fileID < nfiles; fileID++)
        {
          fields[ompthID].vec[fileID] = ef[fileID].array[t][i];
          if (lmiss && DBL_IS_EQUAL(fields[ompthID].vec[fileID], ef[fileID].missval[t]))
            {
              fields[ompthID].vec[fileID] = missval;
              fields[ompthID].nmiss++;
            }
        }

      array2[i] = arg->lpctl ? vfldpctl(fields[ompthID], arg->pn) : vfldfun(fields[ompthID], arg->operfunc);

      if (DBL_IS_EQUAL(array2[i], fields[ompthID].missval)) nmiss++;

      if (withCountData) count2[i] = nfiles - fields[ompthID].nmiss;
    }

  cdoDefRecord(arg->streamID2, arg->varID[t], arg->levelID[t]);
  cdoWriteRecord(arg->streamID2, array2, nmiss);

  if (withCountData)
    {
      cdoDefRecord(arg->streamID2, arg->varID[t] + arg->nvars, arg->levelID[t]);
      cdoWriteRecord(arg->streamID2, count2, 0);
    }

  return nullptr;
}

static void
addOperators(void)
{
  // clang-format off
  cdoOperatorAdd("ensrange", func_range, 0, nullptr);
  cdoOperatorAdd("ensmin",   func_min,   0, nullptr);
  cdoOperatorAdd("ensmax",   func_max,   0, nullptr);
  cdoOperatorAdd("enssum",   func_sum,   0, nullptr);
  cdoOperatorAdd("ensmean",  func_mean,  0, nullptr);
  cdoOperatorAdd("ensavg",   func_avg,   0, nullptr);
  cdoOperatorAdd("ensstd",   func_std,   0, nullptr);
  cdoOperatorAdd("ensstd1",  func_std1,  0, nullptr);
  cdoOperatorAdd("ensvar",   func_var,   0, nullptr);
  cdoOperatorAdd("ensvar1",  func_var1,  0, nullptr);
  cdoOperatorAdd("enspctl",  func_pctl,  0, nullptr);
  cdoOperatorAdd("ensskew",  func_skew,  0, nullptr);
  cdoOperatorAdd("enskurt",  func_kurt,  0, nullptr);
  // clang-format on
}

void *
Ensstat(void *process)
{
  void *task = CDO_task ? cdo_task_new() : nullptr;
  ensstat_arg_t ensstat_arg;
  int nrecs0;

  cdoInitialize(process);

  addOperators();

  const int operatorID = cdoOperatorID();
  const int operfunc = cdoOperatorF1(operatorID);

  const bool lpctl = operfunc == func_pctl;

  int argc = operatorArgc();
  const int nargc = argc;

  double pn = 0;
  if (operfunc == func_pctl)
    {
      operatorInputArg("percentile number");
      pn = parameter2double(operatorArgv()[0]);
      percentile_check_number(pn);
      argc--;
    }

  bool withCountData = false;
  if (argc == 1)
    {
      if (strcmp("count", operatorArgv()[nargc - 1]) == 0)
        withCountData = true;
      else
        cdoAbort("Unknown parameter: >%s<", operatorArgv()[nargc - 1]);
    }

  const int nfiles = cdoStreamCnt() - 1;

  if (Options::cdoVerbose) cdoPrint("Ensemble over %d files.", nfiles);

  const char *ofilename = cdoGetStreamName(nfiles);

  if (!Options::cdoOverwriteMode && fileExists(ofilename) && !userFileOverwrite(ofilename))
    cdoAbort("Outputfile %s already exists!", ofilename);

  std::vector<ens_file_t> ef(nfiles);

  FieldVector fields(Threading::ompNumThreads);
  for (int i = 0; i < Threading::ompNumThreads; i++) fields[i].resize(nfiles);

  for (int fileID = 0; fileID < nfiles; fileID++)
    {
      ef[fileID].streamID = cdoOpenRead((fileID));
      ef[fileID].vlistID = cdoStreamInqVlist(ef[fileID].streamID);
    }

  // check that the contents is always the same
  for (int fileID = 1; fileID < nfiles; fileID++) vlistCompare(ef[0].vlistID, ef[fileID].vlistID, CMP_ALL);

  const int vlistID1 = ef[0].vlistID;
  const int vlistID2 = vlistDuplicate(vlistID1);
  const int taxisID1 = vlistInqTaxis(vlistID1);
  const int taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const size_t gridsizemax = vlistGridsizeMax(vlistID1);

  for (int fileID = 0; fileID < nfiles; fileID++)
    {
      ef[fileID].array[0] = (double *) Malloc(gridsizemax * sizeof(double));
      ef[fileID].array[1] = CDO_task ? (double *) Malloc(gridsizemax * sizeof(double)) : nullptr;
    }

  std::vector<double> array2(gridsizemax);

  const int nvars = vlistNvars(vlistID2);
  std::vector<double> count2;
  if (withCountData)
    {
      count2.resize(gridsizemax);
      for (int varID = 0; varID < nvars; ++varID)
        {
          char name[CDI_MAX_NAME];
          vlistInqVarName(vlistID2, varID, name);
          strcat(name, "_count");
          const int gridID = vlistInqVarGrid(vlistID2, varID);
          const int zaxisID = vlistInqVarZaxis(vlistID2, varID);
          const int timetype = vlistInqVarTimetype(vlistID2, varID);
          const int cvarID = vlistDefVar(vlistID2, gridID, zaxisID, timetype);
          vlistDefVarName(vlistID2, cvarID, name);
          vlistDefVarDatatype(vlistID2, cvarID, CDI_DATATYPE_INT16);
          if (cvarID != (varID + nvars)) cdoAbort("Internal error, varIDs do not match!");
        }
    }

  CdoStreamID streamID2 = cdoOpenWrite((nfiles));
  cdoDefVlist(streamID2, vlistID2);

  ensstat_arg.vlistID1 = vlistID1;
  ensstat_arg.streamID2 = streamID2;
  ensstat_arg.nfiles = nfiles;
  ensstat_arg.array2Data = array2.data();
  ensstat_arg.count2Data = count2.data();
  ensstat_arg.fieldsData = fields.data();
  ensstat_arg.operfunc = operfunc;
  ensstat_arg.pn = pn;
  ensstat_arg.lpctl = lpctl;
  ensstat_arg.withCountData = withCountData;
  ensstat_arg.nvars = nvars;
  ensstat_arg.t = 0;

  bool lwarning = false;
  bool lerror = false;
  int t = 0;
  int tsID = 0;
  do
    {
      nrecs0 = cdoStreamInqTimestep(ef[0].streamID, tsID);
      for (int fileID = 1; fileID < nfiles; fileID++)
        {
          CdoStreamID streamID = ef[fileID].streamID;
          const int nrecs = cdoStreamInqTimestep(streamID, tsID);
          if (nrecs != nrecs0)
            {
              if (nrecs == 0)
                {
                  lwarning = true;
                  cdoWarning("Inconsistent ensemble file, too few time steps in %s!", cdoGetStreamName(fileID));
                }
              else if (nrecs0 == 0)
                {
                  lwarning = true;
                  cdoWarning("Inconsistent ensemble file, too few time steps in %s!", cdoGetStreamName(0));
                }
              else
                {
                  lerror = true;
                  cdoWarning("Inconsistent ensemble file, number of records at time step %d of %s and %s differ!", tsID + 1,
                             cdoGetStreamName(0), cdoGetStreamName(fileID));
                }
              goto CLEANUP;
            }
        }

      if (nrecs0 > 0)
        {
          taxisCopyTimestep(taxisID2, taxisID1);
          cdoDefTimestep(streamID2, tsID);
        }

      for (int recID = 0; recID < nrecs0; recID++)
        {
          int varID = -1, levelID = -1;

          for (int fileID = 0; fileID < nfiles; fileID++)
            {
              cdoInqRecord(ef[fileID].streamID, &varID, &levelID);
              ef[fileID].missval[t] = vlistInqVarMissval(ef[fileID].vlistID, varID);
            }
          //#pragma omp parallel for default(none) shared(ef, t)
          for (int fileID = 0; fileID < nfiles; ++fileID)
            {
              cdoReadRecord(ef[fileID].streamID, ef[fileID].array[t], &ef[fileID].nmiss[t]);
            }

          ensstat_arg.efData = ef.data();
          ensstat_arg.varID[t] = varID;
          ensstat_arg.levelID[t] = levelID;
          if (CDO_task)
            {
              cdo_task_start(task, ensstat_func, &ensstat_arg);
              cdo_task_wait(task);
              // t = !t;
            }
          else
            {
              ensstat_func(&ensstat_arg);
            }
        }

      tsID++;
    }
  while (nrecs0 > 0);

CLEANUP:

  if (lwarning) cdoWarning("Inconsistent ensemble, processed only the first %d timesteps!", tsID);
  if (lerror) cdoAbort("Inconsistent ensemble, processed only the first %d timesteps!", tsID);

  for (int fileID = 0; fileID < nfiles; fileID++) cdoStreamClose(ef[fileID].streamID);

  cdoStreamClose(streamID2);

  for (int fileID = 0; fileID < nfiles; fileID++)
    {
      if (ef[fileID].array[0]) Free(ef[fileID].array[0]);
      if (ef[fileID].array[1]) Free(ef[fileID].array[1]);
    }

  if (task) cdo_task_delete(task);

  cdoFinish();

  return nullptr;
}
