/*
 * main.c
 *
 * Copyright (C) 2003 Bastian Blank <waldi@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $LastChangedBy: bastian $
 * $LastChangedDate: 2008-01-13 16:24:45 +0000 (So, 13 Jan 2008) $
 * $LastChangedRevision: 1463 $
 */

#include <config.h>

#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/mount.h>
#include <unistd.h>

#include "download.h"
#include "execute.h"
#include "frontend.h"
#include "install.h"
#include "log.h"
#include "packages.h"
#include "prepare.h"
#include "suite.h"

char install_root[PATH_MAX];
static const char *helperdir;

static int count, count_done;
static di_list mounts;

#define BUFSIZE_command 4096

static void install_progress_set (int n)
{
  frontend_progress_set (((float) n/count) * 500 + 500);
}

static int install_execute_io_handler (const char *buf, size_t n __attribute__ ((unused)), void *user_data __attribute__ ((unused)))
{
  log_text (DI_LOG_LEVEL_OUTPUT, "%s", buf);
  return 0;
}

static void install_execute_progress_update (di_packages *packages, char *package, di_package_status status)
{
  di_package *p = di_packages_get_package (packages, package, 0);
  if (p && p->status < status)
  {
    log_text (DI_LOG_LEVEL_DEBUG, "Updating %s to status %d", p->key.string, status);
    p->status = status;
  }
}

static int install_execute_progress_io_handler (const char *buf, size_t n __attribute__ ((unused)), void *user_data)
{
  di_packages *packages = user_data;
  char buf_package[128];

  install_execute_io_handler (buf, n, NULL);

  if (sscanf (buf, "Unpacking replacement %128s", buf_package) == 1 ||
      sscanf (buf, "Unpacking %128s", buf_package) == 1)
  {
    log_message (LOG_MESSAGE_INFO_INSTALL_PACKAGE_UNPACK, buf_package);
    install_progress_set (++count_done);
    install_execute_progress_update (packages, buf_package, di_package_status_unpacked);
  }
  else if (sscanf (buf, "Setting up %128s", buf_package) == 1)
  {
    log_message (LOG_MESSAGE_INFO_INSTALL_PACKAGE_CONFIGURE, buf_package);
    install_progress_set (++count_done);
    install_execute_progress_update (packages, buf_package, di_package_status_installed);
  }

  return 0;
}

static int install_execute_target_progress (const char *command, di_packages *packages)
{
  return execute_target_full (command, install_execute_progress_io_handler, install_execute_io_handler, packages);
}

di_slist *install_list (di_packages *packages, di_packages_allocator *allocator, di_slist *install, di_package_priority priority, di_package_status status)
{
  di_slist *list1, *list2;
  di_slist_node *node;

  list1 = di_slist_alloc ();

  for (node = install->head; node; node = node->next)
  {
    di_package *p = node->data;
    if (p->priority >= priority && p->status < status)
      di_slist_append (list1, p);
  }

  list2 = di_packages_resolve_dependencies (packages, list1, allocator);

  di_slist_free (list1);

  list1 = di_slist_alloc ();

  for (node = list2->head; node; node = node->next)
  {
    di_package *p = node->data;
    if (p->status < status)
      di_slist_append (list1, p);
  }

  di_slist_free (list2);

  return list1;
}

di_slist *install_list_package (di_packages *packages, di_packages_allocator *allocator, char *package, di_package_status status)
{
  di_slist *list1, *list2;
  di_slist_node *node;
  di_package *p;

  list1 = di_slist_alloc ();

  p = di_packages_get_package (packages, package, 0);
  if (!p || p->status >= status)
    return list1;

  di_slist_append (list1, p);

  list2 = di_packages_resolve_dependencies (packages, list1, allocator);

  di_slist_free (list1);

  list1 = di_slist_alloc ();

  for (node = list2->head; node; node = node->next)
  {
    di_package *p = node->data;
    if (p->status < status)
      di_slist_append (list1, p);
  }

  di_slist_free (list2);

  return list1;
}

di_slist *install_list_package_only (di_packages *packages, char *package, di_package_status status)
{
  di_slist *list;
  di_package *p;

  list = di_slist_alloc ();

  p = di_packages_get_package (packages, package, 0);

  if (p && p->status < status)
    di_slist_append (list, p);

  return list;
}

int install_apt_install (di_packages *packages, di_slist *install)
{
  char buf[BUFSIZE_command];
  size_t len;
  di_slist_node *node;
  int count = 0;

  strcpy (buf, "apt-get install --yes -o APT::Get::AllowUnauthenticated=true -o APT::Install-Recommends=false");
  len = strlen (buf);

  for (node = install->head; node; node = node->next)
  {
    count++;
    di_package *p = node->data;
    len += 1 + p->key.size;
    if (len >= sizeof (buf))
      log_text (DI_LOG_LEVEL_ERROR, "buffer overflow");
    strcat (buf, " ");
    strcat (buf, p->key.string);
  }

  if (count && install_execute_target_progress (buf, packages))
    return 1;

  return 0;
}

int install_dpkg_configure (di_packages *packages, int force)
{
  char buf[BUFSIZE_command];

  strcpy (buf, "dpkg --configure -a");
  if (force)
    strcat (buf, " --force-all");

  if (install_execute_target_progress (buf, packages))
    return 1;

  return 0;
}

static int install_dpkg_all (char *buf, size_t bufsize, di_packages *packages, di_slist *install)
{
  char buf1[256];
  size_t len;
  di_slist_node *node;

  len = strlen (buf);

  for (node = install->head; node; node = node->next)
  {
    di_package *p = node->data;
    build_target_deb_root (buf1, sizeof (buf1), package_get_local_filename (p));
    len += 1 + strlen (buf1);
    if (len >= bufsize)
      log_text (DI_LOG_LEVEL_ERROR, "buffer overflow");
    strcat (buf, " ");
    strcat (buf, buf1);
  }

  if (install_execute_target_progress (buf, packages))
    return 1;

  return 0;
}

int install_dpkg_install (di_packages *packages, di_slist *install, int force)
{
  char buf[BUFSIZE_command];

  strcpy (buf, "dpkg -i");
  if (force)
    strcat (buf, " --force-all");

  return install_dpkg_all (buf, sizeof (buf), packages, install);
}

int install_dpkg_unpack (di_packages *packages, di_slist *install)
{
  char buf[BUFSIZE_command];

  strcpy (buf, "dpkg --unpack --force-all");

  return install_dpkg_all (buf, sizeof (buf), packages, install);
}

int install_extract (di_slist *install)
{
  int count = 0, count_done = 0;
  struct di_slist_node *node;

  for (node = install->head; node; node = node->next)
    count++;

  for (node = install->head; node; node = node->next)
  {
    di_package *p = node->data;
    log_message (LOG_MESSAGE_INFO_INSTALL_PACKAGE_EXTRACT, p->package);
    package_extract (p);
    count_done++;
    frontend_progress_set (((float) count_done/count) * 50 + 450);
  }

  return 0;
}

int install (di_packages *packages, di_packages_allocator *allocator, di_slist *install)
{
  di_slist_node *node;

  for (node = install->head; node; node = node->next)
    count += 2;

  if (suite_install (packages, allocator, install))
    log_text (DI_LOG_LEVEL_ERROR, "Couldn't install system due to errors!");

  return 0;
}

int install_target_check (const char *target)
{
  struct stat statbuf;

  umask (022);

  if (!stat (target, &statbuf))
  {
    if (!(S_ISDIR (statbuf.st_mode)))
      return 1;
  }
  else
  {
    if (mkdir (target, 0755))
      return 1;
  }

  if (!realpath (target, install_root))
    return 1;

  if (install_root[0] != '/')
  {
    char tmp[PATH_MAX];
    if (getcwd (tmp, sizeof (tmp) - strlen (install_root) - 1) == NULL)
      return 1;
    strcat (tmp, "/");
    strcat (tmp, install_root);
    if (!realpath (tmp, install_root))
      return 1;
  }
  return 0;
}

int install_init (const char *_helperdir)
{
  helperdir = _helperdir;

  if (prepare_install ())
    return 1;
  log_open ();
  return 0;
}

int install_mount (const char *what)
{
  char buf[PATH_MAX];
  int ret;
  snprintf (buf, sizeof buf, "%s/%s", install_root, what);
  if (!strcmp (what, "dev"))
    ret = mount ("/dev", buf, NULL, MS_BIND | MS_RDONLY, 0);
  else if (!strcmp (what, "proc"))
    ret = mount ("proc", buf, "proc", 0, 0);
  else
    return 1;
  if (ret)
    log_text (DI_LOG_LEVEL_ERROR, "Unable to mount %s.", what);
  else
    di_list_append (&mounts, strdup (what));
  return ret;
}

void install_umount (void)
{
  char buf[PATH_MAX];
  di_list_node *node;
  for (node = mounts.head; node; node = node->next)
  {
    snprintf (buf, sizeof (buf), "%s/%s", install_root, (char *) node->data);
    umount (buf);
  }
}

int install_helper_install (const char *name)
{
  char file_source[4096];
  char file_dest[4096];
  char buf[4096];
  int ret;
  struct stat s;

  log_message (LOG_MESSAGE_INFO_INSTALL_HELPER_INSTALL, name);

  snprintf (file_source, sizeof (file_source), "%s/%s.deb", helperdir, name);
  snprintf (file_dest, sizeof (file_source), "%s/%s.deb", install_root, name);

  if (stat (file_source, &s) < 0)
    log_text (DI_LOG_LEVEL_ERROR, "Helper package %s not found.", name);

  snprintf (buf, sizeof (buf), "cp %s %s", file_source, file_dest);
  ret = execute (buf);
  if (ret)
    return ret;

  snprintf (buf, sizeof (buf), "dpkg -i /%s.deb", name);
  ret = execute_target (buf);
  unlink (file_dest);

  return ret;
}

int install_helper_remove (const char *name)
{
  char buf[128];

  log_message (LOG_MESSAGE_INFO_INSTALL_HELPER_REMOVE, name);

  snprintf (buf, sizeof (buf), "dpkg -P %s", name);
  execute_target (buf);

  return 0;
}

