/* the cantus project.
 * (c)2002 by Samuel Abels (sam@manicsadness.com)
 * This project's homepage is: http://software.manicsadness.com/cantus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <ctype.h>
#include "support.h"
#include "configfile.h"
#include "rules.h"
#include "gui_rules.h"


/***************************************************************************************
 * BELOW FOLLOW THE STATICS
 ***************************************************************************************/

// this will kill spaces at the start/end/both of source. spaces are:
// single space, tab, vertical tab, form feed, carriage return, or newline
// loc: 0-start 1-end 2-both
static void
wipe_spaces(gchar *source, gint loc)
{
	gchar *psource = NULL;
	gchar *destination = malloc(2048);
	gchar *pdestination = destination;

	if(source != NULL)
	{
		if((loc==0) || (loc==2))
		{
			strncpy(destination, source, 2047);
			while(*pdestination == ' ')
				pdestination++;
			strncpy(source, pdestination, 2047);
		}
		if((loc==1) || (loc==2))
		{
			for(psource = source+strlen(source)-1; *psource != '\0'; psource--)
			{
				if(isspace(*psource)) *psource='\0';
				else break;
			}
		}
	}
	
	free (destination);
	return;
}

/***************************************************************************************
 * END OF STATICS
 ***************************************************************************************/


/*
 * Get the requested option out of the requested category
 * out of the config variable. store its value in "value".
 * returns TRUE if successful, FALSE if not.
 */
gboolean
configfile_option_get(gchar *config, gchar *category, gchar *option, gchar *value)
{
	gchar *pconfig = config;
	GList *file = NULL;
	GList *fileitem = NULL;
	
	gchar *data = NULL;
	gchar rowdata[2048] = "\0";
	gchar category_src[2048] = "\0";
	gchar option_src[2048] = "\0";
	gchar value_src[2048] = "\0";
	gint i = -1;
	gboolean success = FALSE;

	*value = '\0';
	
// get the complete config to a glist (so the rows are separated)
	while(*pconfig != '\0')
	{
		i = -1;
		rowdata[0] = '\0';
		while( *pconfig != '\n'
			&& *pconfig != '\0' )
		{
			rowdata[++i] = *pconfig;
			rowdata[i+1] = '\0';
			++pconfig;
		}
		file = g_list_append (file, strdup(rowdata));

		if(*pconfig == '\0')
			break;
		++pconfig;
	}
	
// If the list is empty, return.
	if( file == NULL )
		return FALSE;
	
// parse through the glist, looking for the correct category
	fileitem = g_list_first(file);
	while(fileitem)
	{
		data = (gchar *)fileitem->data;
// if the line begins with a "[", we expect an category
		if( *data == '['
			&& *(data + 1) != ']'
			&& *(data + 1) != '\n'
			&& *(data + 1) != '\0' )
		{
			strncpy(category_src, data + 1, 2047);
			if(strchr(category_src, ']'))
				*strchr(category_src, ']') = '\0';
			
			wipe_spaces(category_src, 2);
		}
// if not, and if its not an empty line, and its not an comment ("#"),
// and the line contains an "=" we expect an option.
		if( *data != '['
			&& *data != '#'
			&& *data != '\n'
			&& *data != '\0'
			&& strchr(data, '=') != NULL )
		{
			strcpy(option_src, data);
			*strchr(option_src, '=') = '\0';
			
			if(*(strchr(data, '=') + 1) != '\0')
				strcpy(value_src, strchr(data, '=') + 1);
			
			wipe_spaces(option_src, 2);
			wipe_spaces(value_src, 2);
		}
// found match?
		if( option_src[0] != '\0'
			&& strcmp(category, category_src) == 0
			&& strcmp(option, option_src) == 0 )
		{
// if the value is written in "", then remove them
			if(value_src[0] == '"')
			{
				strncpy(value, value_src+1, 2047);
				if(*(value_src+strlen(value_src)-1) == '"')
					*(value+strlen(value)-1) = '\0';
			}
			else
				strncpy(value, value_src, 2047);
			success = TRUE;
		}
		
		option_src[0] = '\0';
		value_src[0] = '\0';

		g_free (data);
		fileitem = fileitem->next;
	}
	
	g_list_free (file);

	return success;
}



/*
 * Adds a option to the config variable. If it is present, it is overwritten.
 */
gboolean
configfile_option_add(gchar *config, gchar *category, gchar *option, gchar *value)
{
	gchar *pconfig = config;
	GList *file = NULL;
	GList *fileitem = NULL;
	
	gchar *data = NULL;
	gchar rowdata[2048];
	gchar category_src[2048];
	gchar option_src[2048];
	gchar value_src[2048];
	gint i = -1;
	gboolean success = FALSE;

	if(!category || !option || !value)
		return FALSE;
	
// get the complete config to a glist (so the rows are separated)
	while(*pconfig != '\0')
	{
		i = -1;
		rowdata[0] = '\0';
		while( *pconfig != '\n'
			&& *pconfig != '\0' )
		{
			rowdata[++i] = *pconfig;
			rowdata[i+1] = '\0';
			++pconfig;
		}
		file = g_list_append(file, g_strdup(rowdata));
		
		if(*pconfig == '\0')
			break;
		++pconfig;
	}
	
// If the list is empty, don't look for existing options
	if(file)
		fileitem = g_list_first(file);
	else
		fileitem = NULL;

	category_src[0] = '\0';
// parse through the glist, looking for the correct category
	while(fileitem)
	{
		data = (gchar *)fileitem->data;
// if the line begins with a "[", we expect an category
		if( *data == '['
			&& *(data + 1) != ']'
			&& *(data + 1) != '\n'
			&& *(data + 1) != '\0' )
		{
			strncpy(category_src, data + 1, 2047);
			if(strchr(category_src, ']'))
				*strchr(category_src, ']') = '\0';
			
			wipe_spaces(category_src, 2);
		}
// if not, and if its not an empty line, and its not a comment ("#"),
// and the line contains an "=" we expect an option.
		if( *data != '['
			&& *data != '#'
			&& *data != '\n'
			&& *data != '\0'
			&& strchr(data, '=') )
		{
			strncpy(option_src, data, 2047);
			*strchr(option_src, '=') = '\0';
			
			if(*(strchr(data, '=') + 1) != '\0')
				strncpy(value_src, strchr(data, '=') + 1, 2047);
			
// cleanup a bit
			wipe_spaces(option_src, 2);
			wipe_spaces(value_src, 2);
		}
// if the option is already in the given category, replace it by the one with the given value.
		if( option_src[0] != '\0'
			&& category_src[0] != '\0'
			&& strcmp(category, category_src) == 0
			&& strcmp(option, option_src) == 0 )
		{
			free(data);
			(gchar *)fileitem->data = malloc(2048);
			data = (gchar *)fileitem->data;
			snprintf(data, 2047, "%s = \"%s\"", option, value);
			success = TRUE;
		}
// reset variables.
		option_src[0] = '\0';
		value_src[0] = '\0';

		fileitem = fileitem->next;
	}
	
// if were done, were done :)
	if(success)
		goto successful;
	success = FALSE;

// so the option is not yet present, we parse through the glist, looking for the correct category.
// If the list is empty, don't look for existing categories
	if(file)
		fileitem = g_list_first(file);
	else
		fileitem = NULL;

	i = -1;
	category_src[0] = '\0';
	while(fileitem)
	{
		data = (gchar *)fileitem->data;
		++i;
// if the line begins with a "[", we expect a category
		if( *data == '['
			&& *(data + 1) != ']'
			&& *(data + 1) != '\n'
			&& *(data + 1) != '\0' )
		{
			strncpy(category_src, data + 1, 2047);
			if(strchr(category_src, ']'))
				*strchr(category_src, ']') = '\0';
			
			wipe_spaces(category_src, 2);
		}
// if we found it, we insert the new option below
		if(strcmp(category, category_src) == 0)
		{
			snprintf(rowdata, 2047, "%s = \"%s\"", option, value);
			success = TRUE;
// if there are more rows to come, just insert, otherwise append.
			if( fileitem->next
			    && (fileitem->next)->data )
			{
				file = g_list_insert(file, strdup(rowdata), i+1);
			}
			else
			{
				file = g_list_append(file, strdup(rowdata));
				file = g_list_append(file, strdup("\0"));
			}
			break;
		}
		
		fileitem = fileitem->next;
	}

// if were done, were done :)
	if(success)
		goto successful;
	success = FALSE;

// so the category was not found, we append it...
	snprintf(rowdata, 2047, "[%s]", category);
	file = g_list_append(file, strdup("\0"));
	file = g_list_append(file, strdup(rowdata));
// ...and the option
	snprintf(rowdata, 2047, "%s = \"%s\"", option, value);
	file = g_list_append(file, strdup(rowdata));

successful:
// copy the product to "config" and empty out the alloced variables
	*config = '\0';
	fileitem = g_list_first(file);
	while(fileitem)
	{
		data = (gchar *)fileitem->data;
		sprintf(config, "%s%s\n", config, data);
		free(data);
		fileitem = fileitem->next;
	}
	g_list_free (file);
	
	return TRUE;
}


/*
 * Reads the configfile into memory
 * returns 0 if success, or >0 if an error.
 */
gint
config_read(gchar *config, gint maxsize)
{
	FILE *configfile = NULL;
	gchar *fullfilename = NULL;
	
// Open configfile
	fullfilename = g_strconcat(getenv("HOME"), "/.cantusrc", '\0');
	configfile = fopen(fullfilename, "rb");
	if(!configfile)
	{
		free(fullfilename);
		return 1;
	}
	
	memset(config, 0, maxsize);
// Read file into memory
	if(fread(config, 1, maxsize-1, configfile) < 3)
	{
		fclose(configfile);
		free(fullfilename);
		return 2;
	}
	
	fclose(configfile);
	free(fullfilename);
	return 0;
}


/*
 * Writes the configfile from memory to disk
 * returns 0 if success, or >0 if an error.
 */
gint
config_write(gchar *config)
{
	FILE *configfile = NULL;
	gchar *fullfilename = NULL;
	
// Open configfile
	fullfilename = g_strconcat(getenv("HOME"), "/.cantusrc", '\0');
	configfile = fopen(fullfilename, "wb");
	free(fullfilename);
	if(!configfile)
		return 1;

// Read file into memory
	if(!fwrite(config, 1, strlen(config), configfile))
	{
		fclose(configfile);
		return 2;
	}
	
	fclose(configfile);
	return 0;
}







/*
 * Removes a category from the config variable.
 */
gboolean
configfile_category_remove(gchar *config, gchar *category)
{
	gchar *pconfig = config;
	GList *file = NULL;
	GList *fileitem = NULL;
	
	gchar *data = NULL, *foo = NULL;
	gchar rowdata[2048];
	gchar category_src[2048];
	gint i = -1;
	gboolean success = FALSE;
	
// get the complete config to a glist (so the rows are separated)
	while(*pconfig != '\0')
	{
		while( *pconfig != '\n'
			&& *pconfig != '\0' )
		{
			rowdata[++i] = *pconfig;
			rowdata[i+1] = '\0';
			++pconfig;
		}
		foo = malloc(2047);
		strncpy(foo, rowdata, 2047);
		file = g_list_append(file, foo);
		rowdata[0] = '\0';
		
		if(*pconfig == '\0')
			break;
		++pconfig;
		i = -1;
	}
	
// If the list is empty, don't look for existing categories
	if(file)
		fileitem = g_list_first(file);
	else
		fileitem = NULL;

// parse through the glist, looking for the correct category
	while(fileitem)
	{
		data = (gchar *)fileitem->data;
// if the line begins with a "[", we expect an category
		if( *data == '['
			&& *(data + 1) != ']'
			&& *(data + 1) != '\n'
			&& *(data + 1) != '\0' )
		{
			strncpy(category_src, data + 1, 2047);
			if(strchr(category_src, ']'))
				*strchr(category_src, ']') = '\0';
			
			wipe_spaces(category_src, 2);
			
			if(strcmp(category, category_src)==0)
			{
// if the category was found, remove it. If there is an empty line before, remove it.
				success = TRUE;
				if( fileitem->prev
					&& *((gchar *)((fileitem->prev)->data)) == '\0' )
				{
					data = (gchar *)((fileitem->prev)->data);
					file = g_list_remove(file, data);
					free(data);
				}
				
				data = (gchar *)fileitem->data;
				fileitem = fileitem->next;
				file = g_list_remove(file, data);
				free(data);

// and remove all of its subentries.
				while(fileitem)
				{
					data = (gchar *)fileitem->data;
					
					if( *data == '['
						|| (*data == '\0' && *(gchar *)(fileitem->next)->data == '[') )
						break;
					
					fileitem = fileitem->next;
					file = g_list_remove(file, data);
					free(data);
				}
			}
		}
		if(fileitem)
	    	    fileitem = fileitem->next;
	}
	
// copy the product to "config" and empty out the alloced variables
	*config = '\0';
	fileitem = g_list_first(file);
	while(fileitem)
	{
		data = (gchar *)fileitem->data;
		sprintf(config, "%s%s\n", config, data);
		free(data);
		fileitem = fileitem->next;
	}
	return success;
}





/*
 * This function checks if a configfile exists. If it does, it will check for validity,
 * otherwise it will try to create a new one.
 */
gboolean
configfile_check(gchar *configfilename, gchar *mustbeversion)
{
	gchar config[10000];
	gchar isversion[2048];
	
// if we have no configfile, create a new one.
	if( access(configfilename, R_OK) != 0 )
		return configfile_set_default();
	
// If we cannot access the configfile, quit.
	if( access(configfilename, W_OK) != 0 )
	{
		printf("Unable to access configfile! (%s)\n", configfilename);
		return FALSE;
	}
	
// Read configuration
	config_read(config, 10000);
	
// Read configfile version;
	configfile_option_get(config, "program internal", "cfgversion", isversion);
	
// Wrong version? Then rename the file or quit.
	if( strcmp(isversion, mustbeversion) != 0 )
	{
		gchar backupcfgfile[2048];
		
		g_error("Configfile is incompatible... probably out of date. Current version is: 3\nCreating new configfile...\n");
		
// Create a backup of the old file.
		snprintf(backupcfgfile, 2047, "%s.old", configfilename);
		if( rename(configfilename, backupcfgfile) != 0 )
		{
			g_error("Couldn't backup old configfile! (New name: %s)\nExiting.\n", backupcfgfile);
			return FALSE;
		}
		else
			printf("Created backup configfile: %s\n", backupcfgfile);
		
		if( !configfile_set_default() )
		{
			printf("Unable to create new configfile! (%s)\n", configfilename);
			return FALSE;
		}
	}
	return TRUE;
}



// write a default configfile
gboolean
configfile_set_default(void)
{
	FILE *configfile = NULL;
	gchar *foo = NULL;
	gchar config[10000] = {
		"[program internal]\n"
		"cfgversion = \"3\"\n"
		"\n"
		"[program general]\n"
		"hide splash = \"FALSE\"\n"
		"show hidden files = \"TRUE\"\n"
		"toggle queue background = \"TRUE\"\n"
		"\n"
		"[queue columns]\n"
#ifdef HAVE_OGG_H
		"column23#1 = \"Ogg-Genre\"\n"
		"column22#1 = \"Ogg-Comment\"\n"
		"column21#1 = \"Ogg-Year\"\n"
		"column20#1 = \"Ogg-Album\"\n"
		"column19#1 = \"Ogg-Song\"\n"
		"column18#1 = \"Ogg-Track\"\n"
		"column17#1 = \"Ogg-Artist\"\n"
#endif
		"column16#1 = \"V1-Comment\"\n"
		"column15#1 = \"V1-Genre\"\n"
		"column14#1 = \"V1-Year\"\n"
		"column13#1 = \"V1-Album\"\n"
		"column12#1 = \"V1-Song\"\n"
		"column11#1 = \"V1-Track\"\n"
		"column10#1 = \"V1-Artist\"\n"
		"column9#1 = \"V2-Comment\"\n"
		"column8#1 = \"V2-Genre\"\n"
		"column7#1 = \"V2-Year\"\n"
		"column6#1 = \"V2-Album\"\n"
		"column5#1 = \"V2-Song\"\n"
		"column4#1 = \"V2-Track\"\n"
		"column3#1 = \"V2-Artist\"\n"
		"column2#1 = \"Destinationname\"\n"
		"column1#1 = \"Status\"\n"
		"column0#1 = \"Filename\"\n"
		"\n"
		"[freedb]\n"
		"cgi-path = \"\"\n"
		"proxy port = \"\"\n"
		"proxy address = \"\"\n"
		"enable proxy = \"FALSE\"\n"
		"\n"
		"[freedb servers]\n"
		"server0#1 = \"us.freedb.org\"\n"
		"server0#2 = \"80\"\n"
		"server0#3 = \"HTTP\"\n"
		"server0#4 = \"/~cddb/cddb.cgi\"\n"
		"\n"
		"[rules preset default]\n"
		"strip double spaces = \"FALSE\"\n"
		"insert space before uppercase = \"FALSE\"\n"
		"first number twodigit = \"FALSE\"\n"
		"strip chars before digit = \"FALSE\"\n"
		"wipe spaces at = \"start and end\"\n"
		"wipe spaces = \"FALSE\"\n"
		"increase first digit by = \"1\"\n"
		"increase first digit = \"FALSE\"\n"
		"delete char to = \"1\"\n"
		"delete char from = \"1\"\n"
		"delete char = \"FALSE\"\n"
		"forcecc case = \"only first letter to upper\"\n"
		"forcecc = \"FALSE\"\n"
		"remove parentheses which = \"leading\"\n"
		"remove parentheses = \"FALSE\"\n"
		"format to = \"%b - %c\"\n"
		"format from = \"(%a) %b - %c\"\n"
		"format = \"FALSE\"\n"
		"replace by = \" - \"\n"
		"replace from = \"-\"\n"
		"replace = \"FALSE\"\n"
		"set perms rwx = \"read\"\n"
		"set perms mode = \"permit (all)\"\n"
		"set perms = \"FALSE\"\n"
		"twodigit as track to tag = \"FALSE\"\n"
		"delete v2 = \"FALSE\"\n"
		"delete v1 = \"FALSE\"\n"
		"copy v1 to v2 = \"FALSE\"\n"
		"copy v2 to v1 = \"FALSE\"\n"
		"cleanup v1tag = \"FALSE\"\n"
		"v2 to filename format = \"%a - %s\"\n"
		"v2 to filename = \"FALSE\"\n"
		"v1 to filename format = \"%a - %s\"\n"
		"v1 to filename = \"FALSE\"\n"
		"\n"
		"[rules list preset default]\n"
		"rule12#1 = \"- Cleanup buggy ID3V1 tags.\"\n"
		"rule12#2 = \"tagcleanup\"\n"
		"rule11#1 = \"- Force-set characters case to all words begin uppercase.\"\n"
		"rule11#2 = \"forcecc\"\n"
		"rule11#3 = \"all words begin uppercase\"\n"
		"rule10#1 = \"- Force-set characters case to every character lowercase.\"\n"
		"rule10#2 = \"forcecc\"\n"
		"rule10#3 = \"every character lowercase\"\n"
		"rule9#1 = \"- Strip multiple spaces.\"\n"
		"rule9#2 = \"stripspace\"\n"
		"rule8#1 = \"- Kill spaces at start and end of the filename.\"\n"
		"rule8#2 = \"killspace\"\n"
		"rule8#3 = \"start and end\"\n"
		"rule7#1 = \"- Insert a space before every uppercase letter.\"\n"
		"rule7#2 = \"insertspace\"\n"
		"rule6#1 = \"- Replace \"-\" by \" - \".\"\n"
		"rule6#2 = \"replace\"\n"
		"rule6#3 = \"-\"\n"
		"rule6#4 = \" - \"\n"
		"rule5#1 = \"- Replace \"%20\" by \" \".\"\n"
		"rule5#2 = \"replace\"\n"
		"rule5#3 = \"%20\"\n"
		"rule5#4 = \" \"\n"
		"rule4#1 = \"- Replace \"_\" by \" \".\"\n"
		"rule4#2 = \"replace\"\n"
		"rule4#3 = \"_\"\n"
		"rule4#4 = \" \"\n"
		"rule3#1 = \"- Set permissions (permit (all) -> read).\"\n"
		"rule3#2 = \"setperms\"\n"
		"rule3#3 = \"permit (all)\"\n"
		"rule3#4 = \"read\"\n"
		"rule2#1 = \"- Set permissions (deny (others) -> write).\"\n"
		"rule2#2 = \"setperms\"\n"
		"rule2#3 = \"deny (others)\"\n"
		"rule2#4 = \"write\"\n"
		"rule1#1 = \"- Set permissions (permit (all) -> write).\"\n"
		"rule1#2 = \"setperms\"\n"
		"rule1#3 = \"permit (all)\"\n"
		"rule1#4 = \"write\"\n"
		"rule0#1 = \"- Set permissions (deny (all) -> execute).\"\n"
		"rule0#2 = \"setperms\"\n"
		"rule0#3 = \"deny (all)\"\n"
		"rule0#4 = \"execute\"\n\0" };
	
// Open configfile
	configfile = fopen(foo = g_strconcat(getenv("HOME"), "/.cantusrc", '\0'), "wb");
	if(!configfile)
	{
		free(foo);
		return FALSE;
	}
	
	fprintf(configfile, "%s", config);
	
	fclose(configfile);
	free(foo);
	return TRUE;
}




// Read the configfile and put the options to the global struct.	
void
get_config_to_struct(Cfg *cfg)
{
	gchar config[10000];
	gchar value[4096];
	gchar option[2048];
	gint curcol = 0;
	
// Read configuration
	config_read(config, 10000);
	
// Show splash?
	configfile_option_get(config, "program general", "hide splash", value);
	if( strncmp(value, "TRUE", 4) == 0 )
		cfg->hidesplash = TRUE;
	else
		cfg->hidesplash = FALSE;
	
// Toggle queue background?
	configfile_option_get(config, "program general", "toggle queue background", value);
	if( strncmp(value, "TRUE", 4) == 0 )
		cfg->togglequeuebg = TRUE;
	else
		cfg->togglequeuebg = FALSE;
	
// Determine defaulttag for the tageditor
	cfg->defaulttag = 3;
	configfile_option_get(config, "program general", "default tag v1", value);
	if( strcmp(value, "TRUE") == 0 )
		cfg->defaulttag = 1;
	configfile_option_get(config, "program general", "default tag v2", value);
	if( strcmp(value, "TRUE") == 0 )
		cfg->defaulttag = 2;

// set homedir and if specified, the default directory.
	strncpy(cfg->homedir, getenv("HOME"), 4095);
	configfile_option_get(config, "program general", "default directory", cfg->defaultdir);
	if( *cfg->defaultdir == '\0' )
		strncpy(cfg->defaultdir, getenv("HOME"), 4095);
	
// determine the correct order for the queue columns
// get correct order from configfile
	curcol = 1;
	strncpy(option, "column0#1", 2047);
	while(configfile_option_get(config, "queue columns", option, value))
	{
		cfg->qcols = g_list_append(cfg->qcols, strdup(value));
		snprintf(option, 2047, "column%i#1", curcol++);
	}
}



void
destroy_config (Cfg *cfg)
{
	GList *item = NULL;
	
	item = g_list_first(cfg->qcols);
	while(item)
	{
		free(item->data);
		item = item->next;
	}
	g_list_free(cfg->qcols);
	cfg->qcols = NULL;
}
