/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
#include "ActionTransition.h"
#include "ActionStateViewer.h"

#include <Application.h>
#include <Component.h>
using namespace camitk;

// --------------- constructor --------------
ActionTransition::ActionTransition(QObject * sender, const char * signal, QState * sourceState, QString buttonText, bool applyPreviousAction, QTextStream * logStream) : QSignalTransition(sender, signal, sourceState) {
    this->buttonText = buttonText;
    this->applyPreviousAction = applyPreviousAction;
    this->logStream = logStream;
}

// --------------- onTransition --------------
void ActionTransition::onTransition(QEvent *e) {
    if (logStream != NULL) {
        startTime = new QTime();
        startTime->start();
        (*logStream)  << "\t<transition>" << endl;
        (*logStream) << "\t\t<startTime>" << startTime->toString("hh:mm:ss:zzz") << "</startTime>" << endl;
    }

    Action::ApplyStatus status(Action::TRIGGERED);

    // Apply Previous action (if necessary)
    QAbstractState * previousState = this->sourceState();
    ActionState * previousActionState = dynamic_cast<ActionState *> (previousState);

    if ((previousState) && (applyPreviousAction)) {
        status = previousActionState->applyAction();
    }

    // Close needed components
    if (! componentsToClose.isEmpty()) {
        if (logStream != NULL) {
            (*logStream) << "\t\t<closing>" << endl;
        }

        ComponentList compsToClose;
        // Find the components to close (from their names and types)
        ComponentList allComps = Application::getAllComponents();
        ComponentList::const_iterator compIt;
        QMap<QString, QString>::const_iterator namesIt;

        for (namesIt = componentsToClose.begin(); namesIt != componentsToClose.end(); namesIt++) {
            QString compName = namesIt.key();
            QString compType = namesIt.value();

            // Look for the corresponding component into the list of all components
            compIt = allComps.begin();

            while ( (compIt != allComps.end()) &&
                    ( ((*compIt)->getName() != compName) ||
                      (!(*compIt)->isInstanceOf(compType)) )
                  )
                compIt++;

            if (compIt != allComps.end()) {
                compsToClose.append((*compIt));
            }
        }

        // Close them !
        for (compIt = compsToClose.begin(); compIt != compsToClose.end(); compIt++) {
            bool forceClose = componentsToForceClose.contains((*compIt)->getName());
            if (logStream != NULL) {
                (*logStream) << "\t\t\t<component name='" << (*compIt)->getName();
                if (forceClose)
                    (*logStream) << "' force='true";
                (*logStream) << "'/>" << endl;
            }
            if (forceClose)
                (*compIt)->setModified(false);

            Application::close((*compIt));
        }

        if (logStream != NULL) {
            (*logStream) << "\t\t</closing>" << endl;
        }
    }

    // Go to next State
    QAbstractState * nextState = this->targetState();
    ActionState * nextActionState = dynamic_cast<ActionState *> (nextState);

    if (nextActionState) {
        nextActionState->setPreviousActionStatus(status);
        ActionStateViewer::getActionStateViewer()->setState(nextActionState);
    }

    if (logStream != NULL) {
        QTime endTime = QTime::currentTime();

        (*logStream) << "\t\t<endTime>" << endTime.toString("hh:mm:ss:zzz") << "</endTime>" << endl;
        (*logStream) << "\t\t<timeEnlapsed unit='ms'>" << (startTime->elapsed()) << "</timeEnlapsed>" << endl;
        (*logStream)  << "\t</transition>" << endl;
    }
}

// --------------- setButtonText --------------
void ActionTransition::setButtonText(QString buttonText) {
    this->buttonText = buttonText;
}

// --------------- getButtonText --------------
QString ActionTransition::getButtonText() {
    return this->buttonText;
}

// --------------- addComponentToClose --------------
void ActionTransition::addComponentToClose(QString compName, QString compType, bool force) {
    this->componentsToClose.insert(compName, compType);
    if (force)
        componentsToForceClose.append(compName);
}
