/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

//- Action stuffs
#include "MeshClipping.h"

//- Core stuffs
#include "ImageComponent.h"
#include "Application.h"
#include "InteractiveViewer.h"
#include "RendererWidget.h"
#include "ActionWidget.h"
using namespace camitk;

// Vtk stuffs (Clipping functions...)
#include <vtkProperty.h>
#include <vtkClipDataSet.h>
#include <vtkClipPolyData.h>
#include <vtkExtractGeometry.h>
#include <vtkExtractPolyDataGeometry.h>
#include <vtkRenderWindow.h>

//- Qt stuffs (Action Viewer)
#include <QMessageBox>
#include <QBoxLayout>
#include <qdebug.h>


// --------------- constructor -------------------
MeshClipping::MeshClipping(ActionExtension *extension): Action(extension) {
    //-- Initializing action parameters
    this->setName("Mesh Clipping");
    this->setDescription("Interactive Mesh Clipping in the 3D Viewer");
    this->setComponent("MeshComponent");    // nothing needed
    this->setFamily("Mesh Processing");

    this->addTag("Mesh");
    this->addTag("Clipping");
    this->addTag("3D Interaction");

    // please always use lazy instanciation for GUI
    // (i.e., no GUI related code in action constructors)
    this->widget = NULL; 
    this->smoothing = false;
    this->allComponents = false;
    this->planeVisibility = true;
    this->save = false;

}

// --------------- destructor -------------------
MeshClipping::~MeshClipping() {
    delete(this->widget);
}

// --------------- getWidget -------------------
QWidget *MeshClipping::getWidget() {
    //-- Customizing the action widget (specific UI, specific buttons...)
    if (!widget) {
        widget = new PlaneWidget(this);
        customizeActionLayout();
    }

    //-- Creating the clipping plane widget (the first time getWidget() is called)
    if (!widget->planeWidget) {
        vtkRenderWindowInteractor *iren =
            InteractiveViewer::get3DViewer()->
            getRendererWidget()->GetRenderWindow()->GetInteractor();

        //-- Configure the plane widget including callbacks
        widget->planeWidget =
            vtkSmartPointer < vtkImplicitPlaneWidget >::New();
        widget->planeWidget->SetInteractor(iren);

        //-- Bounding box definition and placement
        widget->planeWidget->SetPlaceFactor(1.5);

        InteractiveViewer::get3DViewer()->getBounds(limBounds);
        widget->planeWidget->PlaceWidget(limBounds);
        QVector3D origin((limBounds[1] + limBounds[0]) / 2,
                         (limBounds[3] + limBounds[2]) / 2,
                         (limBounds[5] + limBounds[4]) / 2);
        QVector3D normal(0, 0, (limBounds[4] - limBounds[5]));
        widget->planeWidget->SetNormal(normal.x(), normal.y(), normal.z());
        widget->planeWidget->SetOrigin(origin.x(), origin.y(), origin.z());
        widget->planeWidget->GetOutlineProperty()->SetColor(0, 0, 0);
        widget->planeWidget->GetOutlineProperty()->SetLineWidth(1);
        widget->planeWidget->GetPlaneProperty()->SetOpacity(0);

        widget->planeWidget->UpdatePlacement();

        //-- Widget Action Parameters
        widget->planeWidget->OutlineTranslationOff();
        widget->planeWidget->ScaleEnabledOff();
        widget->planeWidget->AddObserver(vtkCommand::EndInteractionEvent, this->widget);
    }

    if(planeVisibility) {
        widget->planeWidget->EnabledOn();
    }
    else {
        widget->planeWidget->EnabledOff();
    }

    return Action::getWidget();
}

// --------------- apply -------------------
Action::ApplyStatus MeshClipping::apply() {
    //-- Plane creation
    vtkSmartPointer < vtkPlane > plane = vtkSmartPointer < vtkPlane >::New();
    widget->planeWidget->GetPlane(plane);

    list = Application::getAllComponents();

    //-- Clipping the targeted components
    if(list.size() != 0) {
        //-- Clipping
        Component *candidate;
        foreach(candidate, list) {
            if(candidate->isInstanceOf("MeshComponent")) {
                if(allComponents || candidate->isSelected()) {
                    clipMesh(dynamic_cast < MeshComponent * >(candidate), plane);
                }
            }
        }
    }
    else {
        //--No targets : disabling the widget
        widget->planeWidget->EnabledOff();
    }

    save = false;
    Application::refresh();
    QApplication::restoreOverrideCursor();
    return SUCCESS;
}

// --------------- customizeActionLayout -------------------
void MeshClipping::customizeActionLayout() {
    QLayout *informationFrameLayout = Action::getWidget()->layout();

    //-- Hidding apply/revert buttons
    ActionWidget *actionWidget = dynamic_cast<ActionWidget * >(this->actionWidget);
    actionWidget->setButtonVisibility(false);

    //-- Global/Selected component clipping Button
    QPushButton *allComponentsButton = new QPushButton("Clip All Components");
    informationFrameLayout->addWidget(allComponentsButton);
    QObject::connect(allComponentsButton, SIGNAL(released()), SLOT(changeAllComponents()));

    //-- Smoothing button
    QPushButton *smoothButton = new QPushButton("Smooth Clipping");
    informationFrameLayout->addWidget(smoothButton);
    QObject::connect(smoothButton, SIGNAL(released()), SLOT(changeSmoothing()));

    //-- Visibility button
    QPushButton *visibilityButton = new QPushButton("Hide clipping widget");
    informationFrameLayout->addWidget(visibilityButton);
    QObject::connect(visibilityButton, SIGNAL(released()), SLOT(changeVisibility()));

    //-- Restore unclipped meshes
    QPushButton *restoreButton = new QPushButton("Restore unclipped meshes");
    informationFrameLayout->addWidget(restoreButton);
    QObject::connect(restoreButton, SIGNAL(released()), SLOT(restoreMeshes()));

    //-- Update button (when component added?)
    QPushButton *updateButton = new QPushButton("Update");
    informationFrameLayout->addWidget(updateButton);
    QObject::connect(updateButton, SIGNAL(released()), this, SLOT(updateBox()));

    //-- Save button (when component added?)
    QPushButton *saveButton = new QPushButton("Save Clipped Component(s)");
    informationFrameLayout->addWidget(saveButton);
    QObject::connect(saveButton, SIGNAL(released()), this, SLOT(saveClippedMeshes()));

    //-- set the layout for the action widget
    Action::getWidget()->setLayout(informationFrameLayout);
}

// --------------- updateBox -------------------
void MeshClipping::updateBox() {
    //-- Saving previous values for the plane normal and origin
    double origin[3];
    widget->planeWidget->GetOrigin(origin);
    double normal[3];
    widget->planeWidget->GetNormal(normal);

    //-- Looking for new 3D viewer bounds
    double currentBounds[6];
    ComponentList list = Application::getTopLevelComponents();
    (*list.begin())->getBounds(limBounds);
    Component *currentComponent;
    foreach(currentComponent, list) {
        currentComponent->getBounds(currentBounds);
        if(currentBounds[0] < limBounds[0])
            limBounds[0] = currentBounds[0];
        if(currentBounds[1] > limBounds[1])
            limBounds[1] = currentBounds[1];
        if(currentBounds[2] < limBounds[2])
            limBounds[2] = currentBounds[2];
        if(currentBounds[3] > limBounds[3])
            limBounds[3] = currentBounds[3];
        if(currentBounds[4] < limBounds[4])
            limBounds[4] = currentBounds[4];
        if(currentBounds[5] > limBounds[5])
            limBounds[5] = currentBounds[5];
    }

    widget->planeWidget->PlaceWidget(limBounds);

    //-- Replacing the plane in the newly updated box
    widget->planeWidget->SetNormal(normal);
    widget->planeWidget->SetOrigin(origin);
    widget->planeWidget->UpdatePlacement();

    Application::refresh();
}

// --------------- clipMesh -------------------
void MeshClipping::clipMesh(MeshComponent *mesh, vtkSmartPointer < vtkPlane > plane) {
    switch(mesh->getPointSet()->GetDataObjectType()) {
    case 0:
        //-- mesh is a vtkPolyData
        if(smoothing) {
            //-- smoothed clip
            vtkSmartPointer < vtkClipPolyData > filter =
                vtkSmartPointer < vtkClipPolyData >::New();
            filter->SetInputConnection(mesh->getDataPort());
            filter->InsideOutOn();
            filter->SetClipFunction(plane);
            filter->Update();
            mesh->setDataConnection(filter->GetOutputPort());
            if (save) {
                new MeshComponent(filter->GetOutput(), mesh->getName() + " clipped");
            }
        } else {
            //-- chiselled clip
            vtkSmartPointer < vtkExtractPolyDataGeometry > filter =
                vtkSmartPointer < vtkExtractPolyDataGeometry >::New();
            filter->SetInputConnection(mesh->getDataPort());
            filter->ExtractInsideOn();
            filter->SetImplicitFunction(plane);
            filter->Update();
            mesh->setDataConnection(filter->GetOutputPort());
            if (save) {
                new MeshComponent(filter->GetOutput(), mesh->getName() + " clipped");
            }
        }
        break;

    case 2:
    case 4:
        //-- mesh is a vtkUnstructuredGrid or vtkStructuredGrid
        if(smoothing) {
            //-- smoothed clip
            vtkSmartPointer < vtkClipDataSet > filter =
                vtkSmartPointer < vtkClipDataSet >::New();
            filter->SetInputConnection(mesh->getDataPort());
            filter->InsideOutOn();
            filter->SetClipFunction(plane);
            filter->Update();
            mesh->setDataConnection(filter->GetOutputPort());
            if (save) {
                new MeshComponent((vtkPointSet*)filter->GetOutput(), mesh->getName() + " clipped");
            }
        } else {
            //-- chiselled clip
            vtkSmartPointer < vtkExtractGeometry > filter =
                vtkSmartPointer < vtkExtractGeometry >::New();
            filter->SetInputConnection(mesh->getDataPort());
            filter->ExtractInsideOn();
            filter->SetImplicitFunction(plane);
            filter->Update();
            mesh->setDataConnection(filter->GetOutputPort());
            if (save) {
                new MeshComponent((vtkPointSet*)filter->GetOutput(), mesh->getName() + " clipped");
            }
        }
        break;
    default:
        //-- Not implemented
        QString msg;
        QTextStream in(&msg);
        in << "Mesh_Clipping::clipMesh(...) called on " << mesh->getName()  << endl;
        in << "Mesh clipping not implemented for this file format : " << mesh->getFileName() << endl;

        QMessageBox::warning(NULL, "Action Called", msg);
        break;
    }
}

// --------------- restoreMeshes -------------------
void MeshClipping::restoreMeshes() {
    Component *candidate;
    foreach(candidate, list) {
        candidate->setDataConnection(candidate->getDataPort());
    }
    list.clear();
    Application::refresh();
}

// --------------- changeVisibility -------------------
void MeshClipping::changeVisibility() {
    planeVisibility = !planeVisibility;

    //- Updating button text
    QPushButton *button = dynamic_cast< QPushButton * >(sender());
    if(planeVisibility)
        button->setText("Hide clipping widget");
    else
        button->setText("Show clipping widget");
    Application::refresh();
}

// --------------- changeSmoothing -------------------
void MeshClipping::changeSmoothing() {
    smoothing = !smoothing;

    //- Updating button text
    QPushButton *button = dynamic_cast< QPushButton * >(sender());
    if(smoothing)
        button->setText("Raw clipping");
    else
        button->setText("Smooth clipping");

    apply();

}

// --------------- changeAllComponents -------------------
void MeshClipping::changeAllComponents() {
    allComponents = !allComponents;

    //- Updating button text
    QPushButton *button = dynamic_cast< QPushButton * >(sender());
    if(allComponents) {
        button->setText("Clip Selected Components only");
    } else {
        button->setText("Clip All Components");
        Component *candidate;
        foreach(candidate, list) {
            candidate->setDataConnection(candidate->getDataPort());
        }
    }
    apply();
}

// --------------- saveClippedMeshes -------------------
void MeshClipping::saveClippedMeshes() {
    save = true;
    apply();
}


