/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
#include "DicomImageComponent.h"
#include "gdcmImageReader.h"
#include "vtkGDCMImageReader.h"
#include "gdcmAttribute.h"

// Qt includes
#include <QDir>

// VTK includes
#include <vtkStringArray.h>
#include <vtkPointData.h>


using namespace camitk;
using namespace gdcm;

// --------------- Constructor -------------------
DicomImageComponent::DicomImageComponent(const QString & path) throw(AbortException) : ImageComponent(path)
{
    // Use GDCM libraries to handle dicom volume image files
    vtkGDCMImageReader *imageReader = vtkGDCMImageReader::New(); // reader to construct the image as a vtk image data
    gdcm::Reader *reader = new gdcm::Reader(); // reader to get header with tags from the DICOM images and fix image data according

    // read the single .dcm file
    imageReader->SetFileName(path.toStdString().c_str());
    reader->SetFileName(path.toStdString().c_str());

    // read the bunch of files using gdcm
    imageReader->Update();
    reader->Read();
    
    // Get the data spacing. 
    // The ZSpacing, corresponding to the Slice thickness is wrong, as it is not read from its DICOM tag.
    double *spacing = imageReader->GetDataSpacing();
    
    // Get slice thickness information from the header of the DICOM image
    File file = reader->GetFile();
    gdcm::DataSet ds = file.GetDataSet();
    Attribute<0x0018, 0x0050> at; // Tag 0018 0050 => Slice Thickness
    at.Set(ds);
    spacing[2] = at.GetValue(); // spacing[2] => z-spacing.
    
    // vtkGDCMImageReader class doesn't consider correctly slice-thickness
    // Thus, we need to use this class to preserve the z-spacing in the volumic reconstructed image
    // For more information, see official example : http://gdcm.sourcearchive.com/documentation/2.0.14-4/TestvtkGDCMImageReader2_8cxx-source.html
    vtkImageChangeInformation *imageInfoChanger = vtkImageChangeInformation::New();
    imageInfoChanger->SetInput(imageReader->GetOutput());
    imageInfoChanger->SetOutputSpacing(spacing[0], spacing[1], spacing[2]);
    imageInfoChanger->Update();

    vtkSmartPointer<vtkImageData> image = imageInfoChanger->GetOutput();

    // load the image into CamiTK
    setImageData(image, false);

    // use gdcm reader to provide good information for the lut
    double range[2] = {0.0, 0.0};
    imageReader->GetOutput()->GetScalarRange(range);
    this->getLut()->SetRange(range);  // we need to set up range and table values
    this->getLut()->SetNumberOfTableValues(abs(range[0]) + abs(range[1]));

}

// --------------- destructor -------------------
DicomImageComponent::~DicomImageComponent() {

}

