#include <module.h>
inherit "module";
inherit "caudiumlib";
#include <camas/screens.h>	// For screennames
#include <camas/msg.h>		// MSG() Language macros
#include <camas/globals.h>	// Global definitions
#include <camas/pmods.h>        // Local or normal camas pike modules
#include <camas/addressbook.h> // Address book defines and translations

inherit camas_tags;

object parser = camas_tags->cache_parse_html();

class MailindexArgs {
  array mails_contents;
  int sizeof_mails;
};

// ============================================================================
//  MAILINDEX screen container handler
// ============================================================================
string screen(mapping args, string contents, object id)
{
  string out = "";
 
  CDEBUG("Mailindex SCREEN\n");
  CSESSION->screen = "mailindex";	// Admin screen name
  object o = MailindexArgs();
  o->sizeof_mails = sizeof(CSESSION->mails);
  o->mails_contents = CSESSION->mails;
 
  mapping tags =
  ([
    "camas_checknewmail"      : tag_camas_mailindex,
    "camas_deletemarked"      : tag_camas_mailindex,
    "camas_deletealltrash"    : tag_camas_mailindex,
    "camas_movetotrash"       : tag_camas_mailindex,
    "camas_movemarkedbutton"  : tag_camas_mailindex,
    "camas_movemarkedlist"    : tag_camas_mailindex,
    "camas_archivemarked"     : tag_camas_mailindex,
    "camas_selectall"         : tag_camas_mailindex,
    "camas_showallaftersearch": tag_camas_mailindex,
    "camas_replymarked"       : tag_camas_mailindex,
    "camas_replymovemarked"   : tag_camas_mailindex,
    "camas_replytoallmarked"  : tag_camas_mailindex,
    "camas_replytoallmovemarked": tag_camas_mailindex,
    "camas_sendmarkeddraftsmove": tag_camas_mailindex,
    "camas_sendmarkeddrafts": tag_camas_mailindex,
  ]);

  mapping containers =
  ([
    "camas_mailindex"         : container_camas_mailindex,
  ]);
 
  contents = parser->run(contents, tags, containers, CSESSION->layout, __LINE__ + CAMAS.Tools.get_hash(contents), id, o);
  CSESSION->mails = o->mails_contents;

  if(!id->misc->_xml_parser && QUERY(ent_parse))
    contents = parse_scopes(contents,cb_scopes,id);

  args->method = "post";
  args->name = "camasmailindexform";

  args = CAMAS.Tools.set_target(id, args);
 
  out += CAMAS.Tools.make_container("form", args, contents);

  return out;
}

//! container: camas_mailindex
//!  Container for listing mails
//! childcontainer : mail
//! childcontainer : nomail
//! note: screen: mailindex
string container_camas_mailindex(string tag_name, int pos, mapping args, string contents,object id, object o)
{
  string out = "<!-- Begining of camas_mailindex -->";   // HTML to output

  id->conf->get_provider("navbar")->set_nb_elements(id, o->sizeof_mails);
  id->conf->get_provider("navbar")->set_nb_elements_per_page(id, (int)CSESSION->visiblemail);

  int firstvisiblemail = id->conf->get_provider("navbar")->get_min_element(id);
 
  int lastvisiblemail = id->conf->get_provider("navbar")->get_max_element(id);

  mapping containers =
  ([
    "camas_mail"     : container_camas_mailindex_mail,
    "camas_nomail"   : container_camas_mailindex_nomail,
  ]);
  mapping tags =
  ([
    "camas_checknewmail"      : tag_camas_mailindex,
    "camas_deletemarked"      : tag_camas_mailindex,
    "camas_deletealltrash"    : tag_camas_mailindex,
    "camas_movetotrash"       : tag_camas_mailindex,
    "camas_movemarkedbutton"  : tag_camas_mailindex,
    "camas_movemarkedlist"    : tag_camas_mailindex,
    "camas_archivemarked"     : tag_camas_mailindex,
    "camas_selectall"         : tag_camas_mailindex,
    "camas_showallaftersearch": tag_camas_mailindex,
    "camas_replymarked"       : tag_camas_mailindex,
    "camas_replymovemarked"   : tag_camas_mailindex,
    "camas_replytoallmarked"  : tag_camas_mailindex,
    "camas_replytoallmovemarked": tag_camas_mailindex,
    "camas_sendmarkeddraftsmove": tag_camas_mailindex,
    "camas_sendmarkeddrafts": tag_camas_mailindex,
  ]);
  
  contents = parser->run(contents, tags, containers, CSESSION->layout, __LINE__  + "/" + pos, id, o);
  contents = CAMAS.Parse.parse_html(contents,
              ([
	        "camas_forward"           : tag_camas_readmail,
                "camas_forwardmove"       : tag_camas_readmail
              ]), ([ ]), id, o);

#if 1
  out += contents;
#else
  int firstdisplayedid = firstvisiblemail;
  int lastdisplayedid  = lastvisiblemail;
  int firstid	       = 1;
  int lastid	       = o->sizeof_mails;

  int firstdisplayeduid = 0, 
    lastdisplayeduid = 0, 
    firstuid = 0, 
    lastuid = 0;

  if (o->sizeof_mails > 0)
  {
    firstdisplayeduid= CSESSION->mails[firstdisplayedid-1]->imap->UID;
    lastdisplayeduid = CSESSION->mails[lastdisplayedid-1]->imap->UID;
    firstuid	     = CSESSION->mails[firstid-1]->imap->UID;
    lastuid	     = CSESSION->mails[lastid-1]->imap->UID;
  }

  array outlet =
  ({
    ([
      "firstdisplayedid" : firstdisplayedid,
      "lastdisplayedid"  : lastdisplayedid,
      "firstdisplayeduid": firstdisplayeduid,
      "lastdisplayeduid" : lastdisplayeduid,
      "firstid"          : firstid,
      "lastid"           : lastid,
      "firstuid"         : firstuid,
      "lastuid"          : lastuid,

    ])
  });
  
  out += do_output_tag(args, outlet, contents, id);
#endif

  out += "<!-- End of camas_mailindex -->";
  return out;
}

//! subcontainer : camas_mail
//! parentcontainer : camas_mailindex
//! childcontainer : camas_title
//! childcontainer : camas_contents
//! childcontainer : camas_searchresult
//!  &lt;camas_mail&gt;contents&lt;/camas_mail&gt; is displayed if there are mails to display
//! note: screen: mailindex
string container_camas_mailindex_mail(string tag_name, int pos, mapping args, string contents, object id, object o)
{
  string out = "<!-- Begining of mail -->";

  if(o->sizeof_mails && CSESSION->nothingfound==0)
  {

    // sessionsortorder : current sort order
    //        sortorder : sort order defined in the user preferences
    if(!CSESSION->sessionsortorder)
      CSESSION->sessionsortorder = CSESSION->sortorder;

    mapping containers =
    ([ 
      "camas_title"          : container_camas_mailindex_mail_title,
      "camas_contents"       : container_camas_mailindex_mail_contents,
      "camas_searchresult"   : container_camas_mailindex_mail_searchresult,
      "camas_nosearchresult" : container_camas_mailindex_mail_nosearchresult,
    ]);

    mapping tags =
    ([
      "camas_checknewmail"      : tag_camas_mailindex,
      "camas_deletemarked"      : tag_camas_mailindex,
      "camas_deletealltrash"    : tag_camas_mailindex,
      "camas_movetotrash"       : tag_camas_mailindex,
      "camas_movemarkedbutton"  : tag_camas_mailindex,
      "camas_movemarkedlist"    : tag_camas_mailindex,
      "camas_archivemarked"     : tag_camas_mailindex,
      "camas_selectall"         : tag_camas_mailindex,
      "camas_showallaftersearch": tag_camas_mailindex,
      "camas_replymarked"       : tag_camas_mailindex,
      "camas_replymovemarked"   : tag_camas_mailindex,
      "camas_replytoallmarked"  : tag_camas_mailindex,
      "camas_replytoallmovemarked": tag_camas_mailindex,
      "camas_sendmarkeddraftsmove": tag_camas_mailindex,
      "camas_sendmarkeddrafts": tag_camas_mailindex,
    ]);
    
    out += parser->run(contents, tags, containers, CSESSION->layout, __LINE__  + "/" + pos, id, o);
  }

  out += "<!-- End of mail -->";
  return out;
}

//! subcontainer : camas_nomail
//! parentcontainer : camas_mailindex
//!  &lt;camas_nomail&gt;contents&lt;/camas_nomail&gt; is displayed is there are no mails to display
//! childcontainer : camas_searchresult
//! childcontainer : camas_nosearchresult
//! note: screen: mailindex
string container_camas_mailindex_nomail(string tag_name, int pos, mapping args, string contents, object id, object o)
{
  string out = "<!-- Begining of nomail -->";
  string searchresult;

  if(!o->sizeof_mails || CSESSION->nothingfound==1)
  {
    mapping containers =
    ([
      "camas_searchresult"   : container_camas_mailindex_nomail_searchresult,
      "camas_nosearchresult" : container_camas_mailindex_nomail_nosearchresult,
    ]);
    
    mapping tags =
    ([
      "camas_checknewmail"      : tag_camas_mailindex,
      "camas_deletealltrash"    : tag_camas_mailindex,
      "camas_movetotrash"       : tag_camas_mailindex,
      "camas_showallaftersearch": tag_camas_mailindex,
    ]);

    out += parser->run(contents, tags, containers, CSESSION->layout, __LINE__  + "/" + pos, id, o);
  }

  out += "<!-- End of nomail -->";
  return out;
}

//! subcontainer : camas_title
//! parentcontainer : camas_mail
//!  &lt;camas_title&lt;contents&gt;/camas_title&gt; is displayed one time
//! screen : mailindex
string container_camas_mailindex_mail_title(string tag_name, int pos, mapping args, string contents, object id, object o)
{
  mapping containers =
  ([
    "camas_href"        : container_camas_mailindex_mail_title_href, 
  ]);
  
  string out = parser->run(contents, ([ ]), containers, CSESSION->layout, __LINE__  + "/" + pos, id, o);

  return out;
}

//! subcontainer : camas_href
//! parentcontainer : camas_title
//!  Generates links for sorting mails
//! note: screen: mailindex
string container_camas_mailindex_mail_title_href(string tag_name, int pos, mapping args, string contents, object id, object o)
{
  string out = "";

  // default sort to num
  args->action = zero_type(args->action) ? "sortnum" : args->action;

  string column = args->action - "sort";
  CSESSION->currentcolumn = column;

  mapping vars = ([ ]);

  if (CSESSION->sortcolumn == column)
    vars += ([ "actiontogglesortorder" : "1" ]);
  else
    vars += ([
              "actionchangesort" : "1",
              "col"              : column,
    ]);
  
  args->href = CAMAS.Tools->make_get_url(id, args, vars);

  out += CAMAS.Tools.make_container("a", args, contents);

  out = CAMAS.Parse.parse_html(out,
                   ([
                    "camas_img_arrow"        : tag_camas_mailindex_mail_title_href_arrow,
                    ]),
                   ([ ]),
                   id);

  return out;
}

//! subtag : camas_img_arrow
//!  Draws arrows for sorting mails
//! attribute : src
//! attribute : arrowup
//! attribute : arrownone
//! attribute : arrowdown
//! parentcontainer : camas_href
//! note: screen: mailindex
string tag_camas_mailindex_mail_title_href_arrow(string tag_name, mapping args, object id)
{
  if(CSESSION->currentcolumn == CSESSION->sortcolumn)
  {
    if(CSESSION->sessionsortorder=="forward")
    {
      if(args->arrowup)
      {
        args->src = args->arrowup;
	args->alt = "/\\";
        m_delete(args, "arrowup");
	m_delete(args, "arrowdown");
	m_delete(args, "arrownone");
      	return CAMAS.Tools.make_tag("img", args);
      }
      else
        return "/\\";
    }
    else if(CSESSION->sessionsortorder=="backward")
    {
      if(args->arrowdown)
      {
        args->src = args->arrowdown;
	args->alt = "\\/";
	m_delete(args, "arrowdown");
	m_delete(args, "arrowup");
	m_delete(args, "arrownone");
        return CAMAS.Tools.make_tag("img", args);
      }
      else
        return "\\/";
    }
  }
  else
  {
    if(args->arrownone)
    {
      args->src = args->arrownone;
      args->alt = "-";
      m_delete(args, "arrownone");
      m_delete(args, "arrowdown"); 
      m_delete(args, "arrowup");
      return CAMAS.Tools.make_tag("img", args);
    }
    else
      return "-";
  }

  return "";
}

//! subcontainer : camas_contents
//! parentcontainer : camas_mail
//!  &lt;camas_contents&gt;contents&lt;/camas_contents&gt; is displayed as many times as there are visible mails
//! variable : #num#
//!  The number of the email in the list.
//! variable : #uid#
//!  The UID of the mail (IMAP UID)
//! variable : #unread#
//!  Wether the mail has been read or not.
//!  Returns "unread" if unread, "" otherwise.
//!  Data comes from the \Seen IMAP flag
//! variable : #answered#
//!  Wether the mail has been answered to or not.
//!  Returns "answered" is answered, "" otherwise.
//!  Data comes from the \Answered IMAP flag
//! variable : #flagged#
//!  Wether the mail has been flagged or not.
//!  Returns "flagged" if flagged, "" otherwise.
//!  Data come from the \Flagged IMAP flag
//! variable : #deleted#
//!  Wheter the mail has been marked as deleted or not.
//!  Returns "deleted" if deleted, "" otherwise.
//!  Data comes from the \Deleted IMAP flag
//! variable : #recent#
//!  Wheter the mail is recent or not. A recent mail is a mail which appear in the list for the first time.
//!  Returns "recent" if recent, "" otherwise.
//!  Data comes from the \Recent IMAP flag
//! variable : #draft#
//!  Wheter the mail is a draft or not.
//!  Returns "draft" if draft, "" otherwise
//! variable : #date#
//!  The date the mail was sent.
//!  Data come from the "Date" mail header (date of the first computer sending the mail).
//! variable: #received#
//!  The date the mail was received by the last SMTP server
//!  This date is given by the IMAP server.
//! variable : #delay#
//!  Since how many time the mail has been sent.
//! variable: #delaystep#
//!  The number of time slices since the mail has been sent.
//! variable : #unixtime#
//!  The date the mail was sent in seconds since 01/01/1970
//! variable : #from_name#
//!  The name of the sender.
//!  Reverts to the sender email address if name can't be determined.
//! variable : #from_addr#
//!  The email address of the sender
//! variable : #to_name#
//!  The name of the recipient(s)
//!  Reverts to the recipient email address if name can't be determined.
//! variable : #to_addr#
//!  The address of the recipient(s)
//! variable: #to#
//!  The name and address of the recipient(s)
//! variable : #subject#
//!  The subject of the message.
//!  Data comes from the "Subject" header.
//! variable: #message-id#
//!  The message-id of the message.
//!  Data comes from the "Message-Id" header.
//! variable : #size#
//!  The size of the message.
//! variable : #attachment#
//!  Wheter the mail has attachements or not.
//!  Returns "attachment" is there are attachments, "" otherwise
//!  Attachements only concern multipart/related or multipart/mixed but not multipart/alternative
//! childcontainer : camas_href
//! childcontainer : camas_delay
//! childcontainer : camas_date
//! childcontainer : camas_input_selectmail
//! note: screen: mailindex
string container_camas_mailindex_mail_contents(string tag_name, int pos, mapping args, string contents, object id, object o)
{
  object camas_main = CAMAS_MODULE;
  int time1, time2, time3, time4;
  if (!objectp(camas_main))
  {
    CDEBUG("module camas_main is not present");
    return "foobar";
  }
  string out = "<!-- Begining of contents -->";
  int currentmail = 0;

  // Displaying messages
  int frommail, tomail, deltamail;

  /* Sorting stuff */
  // change sorting if :
  // the sorting column has changed and
  // we didn't change the sort order (to need to compute all in that case, just use reverse()
  // and or the sort column is not the same as the previous one
  //     or the sort column is not the default sortcolumn
  if (CSESSION->mailssortcolumn &&
      (CSESSION->mailssortcolumn != CSESSION->sortcolumn)
      )
  {
    mapping colvals = ([ ]);
	
    for (int i = 0; i < o->sizeof_mails; i++)
    {
      mapping mail = o->mails_contents[i];
      // use the folder + IMAP UID of the given mail as the primary key
      string key = " " + mail->imap->MAILBOX + sprintf("%08d", mail->imap->UID);
     
      switch (CSESSION->sortcolumn)
      {
      case "date":
        colvals[CAMAS.DateTools.sortable_date(mail->imap->ENVELOPE[DATE_IDX], mail->imap->INTERNALDATE)
                + key] = i;
        break;

      case "received":
        colvals[CAMAS.DateTools.sortable_date(mail->imap->INTERNALDATE, mail->imap->ENVELOPE[DATE_IDX])
                + key] = i;
        break;

      case "from":
        string from_name, from_addr;
        if (mail->imap->ENVELOPE[FROM_IDX]) 
	{
          from_name = mail->imap->ENVELOPE[FROM_IDX][0][0];
          from_addr = mail->imap->ENVELOPE[FROM_IDX][0][2] + "@" + mail->imap->ENVELOPE[FROM_IDX][0][3];
          from_addr = replace (from_addr, ({ "@.MISSING-HOST-NAME.", "@0" }), ({ "", "" }));
        }
        else
          from_name = "-";
        from_name = CAMAS.Tools.fix_header (from_name ? from_name : from_addr); 
        colvals[lower_case (from_name) + key] = i;
        break;

      case "to":
        string to;
	// try to sort recipients smartly:
	// sort on recipients names if available
	// sort on recipients address if names are not available
	if (mail->imap->ENVELOPE[TO_IDX])
        {
          array(string) tos = ({ });
	  int i = 0;
          foreach(mail->imap->ENVELOPE[TO_IDX], array recipients)
          {
	    tos += ({ "" });
            if(recipients[0] && recipients[0] != "?")
              tos[i] = recipients[0];
	    if(recipients[2] && recipients[3])
              tos[i] += replace(recipients[2] + "@" + recipients[3],
 			({ "@.MISSING-HOST-NAME.", "@0" }), ({ "", "" }));
	    i++;
          }
          to = CAMAS.Tools.fix_header(tos * ",");
        }
        else
        {
          to = "-";
        }
        to = CAMAS.Tools.fix_header (to);
        colvals[lower_case (to) + key] = i;
        break;

      case "subject":
        string nsub, subj;
        subj = CAMAS.Tools.fix_header (mail->imap->ENVELOPE[SUBJECT_IDX]) || "?";
        while (sscanf (lower_case (subj), "re:%*[ \t]%s", nsub) >= 1)
          subj = nsub;
        colvals[subj + key] = i;
        break;

      case "size":
        colvals[sprintf ("%08d", ((int)mail->imap["RFC822.SIZE"])) + key] = i;
        break;
      
      case "num":
      default:
        colvals[key] = i;
        break;
      }  // switch
    } // for

    CSESSION->mailssortcolumn = CSESSION->sortcolumn;
    if(sizeof(colvals) != o->sizeof_mails)
      write("sizeof(colvals)=%d != sizeof_mails=%d\n",
        sizeof(colvals), o->sizeof_mails);
    if(sizeof(colvals) != sizeof(o->mails_contents))
      write("sizeof(colvals)=%d != sizeof(o->mails_contents)=%d\n",
        sizeof(colvals), sizeof(o->mails_contents));
    array order = sort (indices (colvals)); 
    array newmails = allocate (sizeof(order));
    for (int t = 0; t < sizeof(newmails); t++)
    {
      if(colvals[order[t]] > sizeof(o->mails_contents))
        write("colvals[order[t]] > sizeof(o->mails_contents) (%d > %d)\n", 
	  colvals[order[t]], sizeof(o->mails_contents));
      newmails[t] = o->mails_contents[colvals[order[t]]]; 
    }
    o->mails_contents = newmails;
  } // if
  
  if(CSESSION->sessionsortorder == "backward")
    o->mails_contents = reverse(o->mails_contents);
  // update the mail number since in very few cases 
  // sizeof(colvals) < sizeof(CSESSION->mail)
  id->conf->get_provider("navbar")->set_nb_elements(id, sizeof(o->mails_contents));

  frommail = id->conf->get_provider("navbar")->get_min_element(id);
 
  tomail = id->conf->get_provider("navbar")->get_max_element(id);

  for (currentmail = frommail; currentmail <= tomail; currentmail++)
  {
    object mail;
    // IMAP flags
    string 
      unread_state, 
      answered_state,
      flagged_state,
      deleted_state,
      draft_state,
      recent_state,
      // headers
      from_name = "",
      from_name_field,
      from_addr = "",
      from_addr_field,
      to_name   = "",
      to_addr   = "",
      to        = "",
      num_field,
      uid_field,
      unixtime,
      date_field,
      received_field,
      maxdelaysteps,
      delay_step,
      delay_field,
      attachment,
      personal,
      subject_field,
      arg_subject_field,
      messageid,
      thread_field,
      history_field,
      size_field;
    array(string) to_names, to_addrs, tos;
      
    mail = o->mails_contents[currentmail];

    unread_state   = !has_value (mail->imap->FLAGS, "\\Seen") ? "unread" : "";
    answered_state = has_value (mail->imap->FLAGS, "\\Answered") ? "answered" : "";
    flagged_state  = has_value (mail->imap->FLAGS, "\\Flagged") ? "flagged" : "";
    deleted_state  = has_value (mail->imap->FLAGS, "\\Deleted") ? "deleted" : "";
    draft_state    = has_value (mail->imap->FLAGS, "\\Draft") ? "draft" : "";
    recent_state   = has_value (mail->imap->FLAGS, "\\Recent") ? "recent" : "";
		
    id->misc->camas->curmail_unseen = unread_state;

    // TODO: maybe setting these stings outside this function would be more efficient

    num_field      = (string)(o->mails_contents[currentmail]->number + 1);
    uid_field      = (string)(o->mails_contents[currentmail]->imap->UID);
    unixtime       = (string)CAMAS.DateTools.unix_date(mail->imap->INTERNALDATE);
    date_field     = CAMAS.DateTools.mailindex_date(id, mail->imap->ENVELOPE[DATE_IDX],
				   mail->imap->INTERNALDATE, camas_main->QUERY (indexdateshortformat),
							   camas_main->QUERY (indexdatelongformat));
    received_field = CAMAS.DateTools.mailindex_date(id, mail->imap->INTERNALDATE,
				   mail->imap->ENVELOPE[DATE_IDX], camas_main->QUERY (indexdateshortformat),
							   camas_main->QUERY (indexdatelongformat));
    maxdelaysteps  = zero_type(args->delaysteps) ? 2 : (string)((int)args->delaysteps-1);
    delay_step     = (string)CAMAS.DateTools.compute_delaystep(CSESSION->delaycalc, (int)CSESSION->delaytime, (int)unixtime);
    delay_step     = ((int)delay_step > (int)maxdelaysteps) ? maxdelaysteps : delay_step;
    delay_field    = CAMAS.DateTools.mailindex_delay(mail->imap->INTERNALDATE, mail->imap->ENVELOPE[DATE_IDX],
    camas_main->QUERY(delaystepnumber), camas_main->QUERY(indexdelayformat));

    // Here we try to determine if the mail has attachments or not
    // Instead of recursively counting the number of children of mail->imap->BODY which could be time consuming,
    // just look if the last element of mail->imap->BODY is "mixed" or "related"
    // We don't look for "alternative" coz if the root part is alternative, then it shoudln't contain attachments,
    // or attachments in some of the alternative parts
    // Bertrand
    attachment     = search( ({ "mixed" , "related" }) , mail->imap->BODY[-1])!=-1 ? "attachment" : "";

    if (mail->imap->ENVELOPE[FROM_IDX])
    {
      from_name = mail->imap->ENVELOPE[FROM_IDX][0][0];
      from_addr = mail->imap->ENVELOPE[FROM_IDX][0][2] + "@" + mail->imap->ENVELOPE[FROM_IDX][0][3];
      from_addr = replace (from_addr, ({ "@.MISSING-HOST-NAME.", "@0" }), ({ "", "" }));
    }
    else
    {
      from_name += "-";
    }
    from_name = CAMAS.Tools.fix_header (from_name ? from_name : from_addr);
    from_name_field = from_name;
    from_addr_field = from_addr;

    if (mail->imap->ENVELOPE[TO_IDX])
    {
      to_names = ({ });
      to_addrs = ({ });
      tos      = ({ });
      foreach(mail->imap->ENVELOPE[TO_IDX], array recipients)
      {
        string _to_name = "", _to_addr = "";
        if(recipients[0] && recipients[0] != "?")
          _to_name = recipients[0];
        if(recipients[2] && recipients[3])
          _to_addr = "<" + replace(recipients[2] + "@" + recipients[3],
      			({ "@.MISSING-HOST-NAME.", "@0" }), ({ "", "" })) + ">";
        to_names += ({ _to_name });
        to_addrs += ({ _to_addr });
        tos += ({ _to_name + " " + _to_addr });
      }
      to_name = to_names * ", ";
      to_addr = to_addrs * ", ";
      to = CAMAS.Tools.fix_header(tos * ", ");
    }
    else
    {
      to_name = "-";
    }
    to_name = CAMAS.Tools.fix_header (to_name ? to_name : to_addr);
    to_addr = CAMAS.Tools.fix_header (to_addr);

    personal      = to_addr == CSESSION->address ? "personal" : ""; 
    subject_field = CAMAS.Tools.fix_header (mail->imap->ENVELOPE[SUBJECT_IDX]);
    arg_subject_field = subject_field;

    messageid = mail->imap->ENVELOPE[MESSAGEID_IDX];

    history_field = MSG(M_HISTSEARCH);
    thread_field = MSG(M_THREADSHOW);

    size_field = CAMAS.Tools.display_size(mail->imap["RFC822.SIZE"]);

    array outlet =
			({
				([
					"num"          : num_field,
					"uid"          : uid_field,
					"unread"       : unread_state,
					"answered"     : answered_state,
					"flagged"      : flagged_state,
					"deleted"      : deleted_state,
					"draft"        : draft_state,
					"recent"       : recent_state,
					"personal"     : personal,
					"delay"        : delay_field,
					"delaystep"    : delay_step,
					"date"         : date_field,
					"received"     : received_field,
					"unixtime"     : unixtime,
					"from_name"    : from_name_field,
					"from_addr"    : from_addr_field,
					"to_name"      : to_name,
					"to_addr"      : to_addr,
					"to"           : to,
					"subject"      : subject_field,
					"size"         : size_field,
					"messageid"    : messageid,
					"attachment"   : attachment,
				])
			});
    
    string s;
    s = parser->run(contents,
                     ([
			"camas_input_selectmail"  : container_camas_mailindex_mail_contents_tags,	
										 ]),
                     ([
                        "camas_href"        : container_camas_mailindex_mail_contents_href,
                        "camas_date"        : container_camas_mailindex_mail_contents_date,
                        "camas_delay"       : container_camas_mailindex_mail_contents_delay,
                      ]),
                     CSESSION->layout, __LINE__ + "/" + pos,
		     id, currentmail, from_addr, arg_subject_field, o);

    out += do_output_tag(args, outlet, s, id);

    m_delete(id->misc->camas, "curmail_unseen");
  }
  if(CSESSION->sessionsortorder == "backward")
      o->mails_contents = reverse(o->mails_contents);

  out += "<!-- End of contents -->";
  return out;
}

//! method: container_camas_mailindex_mail_contents_tags
//!  Handler for <camas_mailindex><camas_mail><camas_contents></></></> nes ted tags
string container_camas_mailindex_mail_contents_tags(string tag_name, int pos, mapping args, object id, int currentmailid, string from_addr, string subject_field, object o)
{
	string out = "";

	switch(tag_name)
	{
		case "camas_input_selectmail":
			//! subtag: camas_input_selectmail
			//!  Widget for selecting mail
			//! parentcontainer: camas_contents
			//! attribute: autocheck
			//!	 Automatically checks the mail if it is currently displayed
			//! attribute:
			//!  All attributes applicable to <input />
			//! note: screen: MAILINDEX
			int currentmailuid = o->mails_contents[currentmailid]->imap->UID;
			if(args->autocheck && (string)CSESSION->cmailuid==(string)currentmailuid)
				args->checked = "1";
			m_delete(args, "autocheck");

			args->type = "checkbox";
			args->name = "msg" + currentmailuid;
			args->value = "1";

			out += CAMAS.Tools.make_tag("input", args);
			break;

		default:
			CDEBUG(tag_name+" not supported");
			out += tag_name+" not supported";
	}

	return out;
}

//! subcontainer : camas_href
//! parentcontainer : camas_contents
//!  Clickable zone delimiter
//! attribute: action
//! note: screen: mailindex
string container_camas_mailindex_mail_contents_href(string tag_name, int pos, mapping args, string contents, object id, int currentmail, string from_addr, string subject_field, object o)
{
  string out = "";

  mapping vars = ([ ]);

  switch(args->action)
  {
    case "read":
      vars =
				([ 
					"actionread" : "1",
					"msguid"     : (string)o->mails_contents[currentmail]->imap->UID,
					"mbox"       : o->mails_contents[currentmail]->imap->MAILBOX
				]);
      args->href = CAMAS.Tools->make_get_url(id, args, vars);
      break;

    case "history":
      vars =
				([
					"history"            : "1",
					"text1"              : from_addr,
					"searchfield1"       : "from",
					"searchfield2"       : "to",
					"actionsearchmail"   : "1",
				]);
      args->href = CAMAS.Tools->make_get_url(id, args, vars);
      break;

    case "tracking":
      vars =
				([
					"tracking"          : "1",
					"text1"             : subject_field,
					"searchfield1"      : "subject",
					"nohistsearch"      : "1",
					"actionsearchmail"  : "1"
				]);
      args->href = CAMAS.Tools->make_get_url(id, args, vars);
      break;

    default:
      CDEBUG("The arg "+args->action+" in container_camas_mailindex_mail_contents_href is not available");
  }

  out += CAMAS.Tools.make_container("a", args, contents);
  
  return out;
}

//! subcontainer: camas_date
//! parentcontainer: camas_contents
//!  Zone processed if context needs to display message date
//! note: screen: mailindex
string container_camas_mailindex_mail_contents_date(string tag_name, int pos, mapping args, string contents, object id, object o)
{
  if(CAMASFOLDERTOOLS.is_a_special_folder(CSESSION->mailbox, CSESSION))
    return contents;
  
  return "";
}

//! subcontainer: camas_delay
//! parentcontainer: camas_contents
//!  Zone processed if context needs to display message delay
//! note: screen: mailindex
string container_camas_mailindex_mail_contents_delay(string tag_name, int pos, mapping args, string contents, object id, object o)
{
  if(!CAMASFOLDERTOOLS.is_a_special_folder(CSESSION->mailbox, CSESSION))
    return contents;
  
  return "";
}

//! subcontainer : camas_searchresult
//! parentcontainer : camas_mail
//!  Code to be displayed if current screen displays mails and it is a result of a search
//! note: screen: mailindex
string container_camas_mailindex_mail_searchresult(string tag_name, int pos, mapping args, string contents, object id, object o)
{
  if(CSESSION->searchstring && sizeof (CSESSION->searchstring))
  {
    mapping tags =
    ([
      "camas_checknewmail"      : tag_camas_mailindex,
      "camas_deletemarked"      : tag_camas_mailindex,
      "camas_deletealltrash"    : tag_camas_mailindex,
      "camas_movetotrash"       : tag_camas_mailindex,
      "camas_movemarkedbutton"  : tag_camas_mailindex,
      "camas_movemarkedlist"    : tag_camas_mailindex,
      "camas_archivemarked"     : tag_camas_mailindex,
      "camas_selectall"         : tag_camas_mailindex,
      "camas_showallaftersearch": tag_camas_mailindex,
      "camas_replymarked"       : tag_camas_mailindex,
      "camas_replymovemarked"   : tag_camas_mailindex,
      "camas_replytoallmarked"  : tag_camas_mailindex,
      "camas_replytoallmovemarked": tag_camas_mailindex,
      "camas_sendmarkeddraftsmove": tag_camas_mailindex,
      "camas_sendmarkeddrafts": tag_camas_mailindex,
    ]);

    contents = parser->run(contents, tags, ([ ]), CSESSION->layout, __LINE__  + "/" + pos, id, o);

    return contents;
  }
  
  return "";
}

//! subcontainer : camas_nosearchresult
//! parentcontainer : camas_mail
//!  Code to be displayed if current screen displays mails and it is not a result of a search
//! note: screen: mailindex
string container_camas_mailindex_mail_nosearchresult(string tag_name, int pos, mapping args, string contents, object id, object o)
{
  if(!(CSESSION->searchstring && sizeof (CSESSION->searchstring)))
  {
    mapping tags =
    ([
      "camas_checknewmail"      : tag_camas_mailindex,
      "camas_deletemarked"      : tag_camas_mailindex,
      "camas_deletealltrash"    : tag_camas_mailindex,
      "camas_movetotrash"       : tag_camas_mailindex,
      "camas_movemarkedbutton"  : tag_camas_mailindex,
      "camas_movemarkedlist"    : tag_camas_mailindex,
      "camas_archivemarked"     : tag_camas_mailindex,
      "camas_selectall"         : tag_camas_mailindex,
      "camas_showallaftersearch": tag_camas_mailindex,
      "camas_replymarked"       : tag_camas_mailindex,
      "camas_replymovemarked"   : tag_camas_mailindex,
      "camas_replytoallmarked"  : tag_camas_mailindex,
      "camas_replytoallmovemarked": tag_camas_mailindex,
      "camas_sendmarkeddraftsmove": tag_camas_mailindex,
      "camas_sendmarkeddrafts": tag_camas_mailindex,
    ]);
  
    contents = parser->run(contents, tags, ([ ]), CSESSION->layout, __LINE__  + "/" + pos, id, o);
    
    return contents;
  }
  
  return "";
}

//! subcontainer : camas_searchresult
//! parentcontainer : camas_nomail
//!  Code to be displayed if current screen don't displays mails and is result of a search
//! note: screen: mailindex
string container_camas_mailindex_nomail_searchresult(string tag_name, int pos, mapping args, string contents, object id, object o)
{
  string out = "<!-- Begining of nosearchresult -->";

  if (CSESSION->searchstring && sizeof (CSESSION->searchstring))
  {
    mapping tags =
    ([
      "camas_checknewmail"      : tag_camas_mailindex,
      "camas_deletemarked"      : tag_camas_mailindex,
      "camas_deletealltrash"    : tag_camas_mailindex,
      "camas_movetotrash"       : tag_camas_mailindex,
      "camas_movemarkedbutton"  : tag_camas_mailindex,
      "camas_movemarkedlist"    : tag_camas_mailindex,
      "camas_archivemarked"     : tag_camas_mailindex,
      "camas_selectall"         : tag_camas_mailindex,
      "camas_showallaftersearch": tag_camas_mailindex,
      "camas_replymarked"       : tag_camas_mailindex,
      "camas_replymovemarked"   : tag_camas_mailindex,
      "camas_replytoallmarked"  : tag_camas_mailindex,
      "camas_replytoallmovemarked": tag_camas_mailindex,
      "camas_sendmarkeddraftsmove": tag_camas_mailindex,
      "camas_sendmarkeddrafts": tag_camas_mailindex,
    ]);

    contents = parser->run(contents, tags, ([ ]), CSESSION->layout, __LINE__  + "/" + pos, id, o);

    out += contents;
  }
  
  return out;
}

//! subcontainer : camas_nosearchresult
//! parentcontainer : camas_nomail
//!  Code to be displayed if current screen doesn't display mails and is not result of a search
//! note: screen: mailindex
string container_camas_mailindex_nomail_nosearchresult(string tag_name, int pos, mapping args, string contents, object id, object o)
{
  string out = "<!-- Begining of searchresult -->";
  if (!(CSESSION->searchstring && sizeof (CSESSION->searchstring)))
  {
    mapping tags =
    ([
      "camas_checknewmail"      : tag_camas_mailindex,
      "camas_deletemarked"      : tag_camas_mailindex,
      "camas_deletealltrash"    : tag_camas_mailindex,
      "camas_movetotrash"       : tag_camas_mailindex,
      "camas_movemarkedbutton"  : tag_camas_mailindex,
      "camas_movemarkedlist"    : tag_camas_mailindex,
      "camas_archivemarked"     : tag_camas_mailindex,
      "camas_selectall"         : tag_camas_mailindex,
      "camas_showallaftersearch": tag_camas_mailindex,
      "camas_replymarked"       : tag_camas_mailindex,
      "camas_replymovemarked"   : tag_camas_mailindex,
      "camas_replytoallmarked"  : tag_camas_mailindex,
      "camas_replytoallmovemarked": tag_camas_mailindex,
      "camas_sendmarkeddraftsmove": tag_camas_mailindex,
      "camas_sendmarkeddrafts": tag_camas_mailindex,
    ]);

    contents = parser->run(contents, tags, ([ ]), CSESSION->layout, __LINE__  + "/" + pos, id, o);

    out += contents;
  }
  return out;
}

/* code for most mailindex screen tags */
string tag_camas_mailindex(string tag_name, int pos, mapping args, object id, object o)
{
  object camas_main = CAMAS_MODULE;
  if (!objectp (camas_main)) {
    CDEBUG("module camas_main is not present");
    return "foobar";
  }

  object camas_features = id->conf->get_provider("camas_features");
  if (!objectp (camas_features)) {
    CDEBUG("module camas_features is not present");
    return "foobar";
  }

  string out = "";

  zero_type (args->showpecialfolders) ? 0 : args->showspecialfolders;

  switch (tag_name) {


  case "camas_replymarked":
    //! tag: camas_replymarked
    //!  Button for replying to many mails at the same time
    //! note: screen: mailindex
    if(o->sizeof_mails > 0)
      out = CAMAS.Tools.formdrawbutton(id,"m_replymarked", "actionreplymarked", MSG(M_REPLYMARKED), args);
    else
      out = "<!-- Button camas_replymarked not available because we are in a special folder or "
      "there are no mails to reply to -->\n";
    break;
    
  case "camas_replymovemarked":
    //! tag: camas_replymovemarked
    //!  Button for replying to many mails at the same time and move them in the
    //!  answered folder
    //! note: screen: mailindex
    if(o->sizeof_mails > 0)
    {
      if(!camas_features->QUERY(moveanswered))
      {
				out = "<!-- You must enable Move answered in CAMAS: Features: Features to get this button -->\n";
				break;
      }
		  out = CAMAS.Tools.formdrawbutton(id,"m_replymovemarked", "actionreplymovemarked", MSG(M_REPLYMOVEMARKED), args);
    }
    else
      out = "<!-- No mails to reply -->\n";
    break;
 
  case "camas_replytoallmarked":
    //! tag: camas_replymarked
    //!  Button for replying to many mails at the same time
    //! note: screen: mailindex
    if(o->sizeof_mails > 0)
      out = CAMAS.Tools.formdrawbutton(id,"m_replytoallmarked", "actionreplytoallmarked", MSG(M_REPLYTOALL), args);
    else
      out = "<!-- Button camas_replymarked not available because we are in a special folder or "
      "there are no mails to reply to -->\n";
    break;
    
  case "camas_replytoallmovemarked":
    //! tag: camas_replymovemarked
    //!  Button for replying to many mails at the same time and move them in the
    //!  answered folder
    //! note: screen: mailindex
    if(o->sizeof_mails > 0)
    {
      if(!camas_features->QUERY(moveanswered))
      {
				out = "<!-- You must enable Move answered in CAMAS: Features: Features to get this button -->\n";
				break;
      }
			out = CAMAS.Tools.formdrawbutton(id,"m_replytoallmovemarked", "actionreplytoallmovemarked", MSG(M_REPLYTOALLMOVE), args);
    }
    else
      out = "<!-- No mails to reply -->\n";
    break;
    
  case "camas_deletemarked":
    //! tag: camas_deletemaked
    //!  Button for permanently deleting the selected mails.
    //! note: screen: mailindex
    if (o->sizeof_mails && camas_features->feature(FEAT_MAILBOXES))
    {
      if(CAMASFOLDERTOOLS.is_trash (CSESSION->mailbox, CSESSION->trashfolder) ||
				camas_features->QUERY(deletemethod) != "move to trash")
			{
				out = CAMAS.Tools.formdrawbutton(id,"m_deletemarked", "actiondelete", MSG(M_DELETEMARKED), args);
			}
    }
    break;

  case "camas_deletealltrash":
    //! tag: camas_deletealltrash
    //!  Button for deleting all trash.
    //! note: screen: mailindex
    if (camas_features->feature(FEAT_MAILBOXES))
    {
      if (CAMASFOLDERTOOLS.is_trash(CSESSION->mailbox, CSESSION->trashfolder))
      {
        out = CAMAS.Tools.formdrawbutton(id,"m_deletealltrash", "actiondeleteall", MSG(M_DELETEALLTRASH), args);
      }
      else
      {
        out = "<!-- &lt;camas_deletealltrash&gt; cannot be used because we are not in the trash -->";
      }
    }
    else
    {
      out = "<!-- &lt;camas_deletemarked&gt; cannot be used because of FEAT_MAILBOXES -->";
    }
    break;

  case "camas_movetotrash":
    //! tag: camas_movetotrash
    //!  Button for moving selected mails in trash
    //! note: screen: mailindex
    if (CSESSION->searchstring && sizeof(CSESSION->searchstring) && CSESSION->nothingfound)
    {
      CDEBUG("<camas_movetotrash> : we are in the case of a search giving no result");
    }
    else if(o->sizeof_mails)
    {
      if(!(camas_features->QUERY(deletemethod)=="both" || camas_features->QUERY(deletemethod)=="move to trash") && sizeof (CSESSION->trashfolder))
      {
        out = " &lt;camas_movetotrash&gt; cannot be used either because deletemethod is neither both or move to trash either trashfolder is not set ";
      }
      else if (!CAMASFOLDERTOOLS.is_trash(CSESSION->mailbox, CSESSION->trashfolder)) 
			// we are not in the trash
      {	
				out = CAMAS.Tools.formdrawbutton(id,"m_movetotrash", "actiontrash", MSG(M_MOVETOTRASH), args);
      }
    }
    break;

  case "camas_sendmarkeddrafts":
    //! tag: camas_sendmarkeddrafts
    //!  Button for sending selected drafts
    //! note: It works only when in a drafts folder
    //! note: screen: mailindex
    if(CAMASFOLDERTOOLS.is_draft(CSESSION->mailbox, CSESSION->draftsfolder))
    {
      out = CAMAS.Tools.formdrawbutton(id,"m_sendmarkeddrafts", "actionsendmarkeddrafts",MSG(M_SENDMARKEDDRAFTS), args);
    }
    else
    {
      out = "<!-- camas_sendmarkeddrafts: we are not into a drafts folder -->";
      CDEBUG("camas_sendmarkeddrafts: we are not into a drafts folder");
    }
    break;

  case "camas_sendmarkeddraftsmove":
    //! tag: camas_sendmarkeddraftsmove
    //!  Button for sending selected drafts and moving them into sent-mail
    //! note: It works only when in a drafts folder
    //! note: screen: mailindex
    if(CAMASFOLDERTOOLS.is_draft(CSESSION->mailbox, CSESSION->draftsfolder))
    {
      out = CAMAS.Tools.formdrawbutton(id,"m_sendmarkeddraftsmove", "actionsendmarkeddraftsmove", MSG(M_SENDMARKEDDRAFTSMOVE), args);
    }
    else
    {
      out = "<!-- camas_sendmarkeddraftsmove: we are not into a drafts folder -->";
      CDEBUG("camas_sendmarkeddraftsmove: we are not into a drafts folder");
    }
    break;

  case "camas_movemarkedbutton":
    //! tag: camas_movemarkedbutton
    //!  Button for moving selected mails into a folder
    //! note: It works with camas_movemarkedlist
    //! note: screen: mailindex
    if(CSESSION->searchstring && sizeof(CSESSION->searchstring) && CSESSION->nothingfound)
    {
      CDEBUG("<camas_movemarkedbutton> : we are in the case of a search giving no result");
      out = "<!-- camas_movemarkedbutton : we are in the case of a search giving no result -->";
    }
    else
    {
      if(o->sizeof_mails)
        out = CAMAS.Tools.formdrawbutton(id,"m_movemarked", "actionmovemarked", MSG(M_MOVEMARKED), args);
    }
    break;

  case "camas_movemarkedlist":
    //! tag: camas_movemarkedlist
    //!  List of directories for moving mails into
    //! attribute: restrictmailboxes
    //!  Lists folders only in current mailbox
    //! attribute: showspecialfolders
    //!  Lists special folders like .sent-mail, .Drafts, .Trash et all
    //! attribute: onlymailboxes
    //!  Lists only mailboxes, not folders in it
    //! attribute: setshareduserhome
    //!  In case each people have a "private" working area in shared hierarchy,
    //!  sets the list automatically to its name for conveniance.
    //!  Ex : User John Doe have a working area in a shared directory.
    //!      If "shared.directory.John Doe" exists, it will be automatically selected in &lt;option&gt;&lt;/&gt;
		//! attribute: option_*
		//!  All attributes applicable to <option></>
    //! note: It works with camas_movemarkedbutton
    //! note: screen: mailindex
    if(CSESSION->searchstring && sizeof(CSESSION->searchstring) && CSESSION->nothingfound)
    {
      CDEBUG("<camas_movemarkedlist> : we are in the case of a search giving no result");
      out = "<!-- camas_movemarkedlist : we are in the case of a search giving no result -->";
    }
    else
    {
      if(o->sizeof_mails)
      {
        array folders = CAMASFOLDERTOOLS.list_folders(id,!args->restrictmailboxes,args->showspecialfolders,args->onlymailboxes);
        array currentparts = ({ });

        mapping optionargs = CAMAS.Tools.extract_html_attributes(args, "option");

        optionargs->value = "imhonomailbox";
        string contents = CAMAS.Tools.make_container("option", optionargs, MSG(M_SELECT)+"...")+"\n";
        if(folders)
          foreach(folders, int folder)
          {
            currentparts = (CSESSION->mailboxes[folder][MB_HIERARCHY_IDX]);
	          optionargs->value = HTML_ENCODE_STRING (CSESSION->mailboxes[folder][MB_FOLDERNAME_IDX]);
	  
       	    contents += CAMAS.Tools.make_container("option", optionargs, HTML_ENCODE_STRING(CAMAS.FolderTools.translate_frommboxindex(id, folder)));
          }

        m_delete(args, "restrictmailboxes");
        m_delete(args, "showspecialfolders");
        m_delete(args, "onlymailboxes");
        m_delete(args, "setshareduserhome");
        m_delete(args, "value");
        args->name = "mboxmove";

        out += CAMAS.Tools.make_container("select", args, contents) + "\n";
      }
    }
    break;

  case "camas_archivemarked":
    //! tag: camas_archivemarked
    //!  Moves marked mails to sent-mail
    //! note: desactivated by default in specialfolders
    //! note: screen: mailindex
    if(!CAMASFOLDERTOOLS.is_a_special_folder(CSESSION->mailbox, CSESSION))
    {
      out += CAMAS.Tools.formdrawbutton(id,"m_archivemarked","actionarchivemarked", MSG(M_ARCHIVE), args);
    }
    out = "<!-- camas_archivemarked: we are not in a special folder -->";
    break;

  case "camas_selectall":
    //! tag: camas_selectall
    //!  Provides a checkbox that select/deselect all mail displayed in the screen
    //! note: Require a JavaScript-enabled browser
    //! note: screen: mailindex
    if(CSESSION->searchstring && sizeof(CSESSION->searchstring) && CSESSION->nothingfound)
    {
      CDEBUG("<camas_selectall> : we are in the case of a search giving no result");
    }
    else if(o->sizeof_mails && id->supports->javascript)
    {
			args->type = "checkbox";
			args->name = "all";
			args->onclick = "for(var i=0;i<this.form.elements.length;i++) {var inpt=this.form.elements[i];if(inpt.name.substr(0,3)=='msg') inpt.checked=this.form.all.checked}";
			out = CAMAS.Tools.make_tag("input", args);
    }
    else
    {
      out = "<!-- camas_selectall: either there is no mail, either browser do not support javascript -->";
      CDEBUG("<camas_selectall> : either there is no mail, either browser do not support javascript");
    }
    break;

  case "camas_showallaftersearch":
    //! tag: camas_showallaftersearch
    //!  Button for reseting search
    //! note: screen: mailindex
    if(CSESSION->searchstring && sizeof(CSESSION->searchstring) > 0)
    {
      out += CAMAS.Tools.formdrawbutton(id,"m_showallaftersearch","actionreload", MSG(M_SHOWALLAFTERSEARCH), args);
    }
    else
    {
      out = "<!-- camas_showallaftersearch : currently not in a search result screen -->";
      CDEBUG("<camas_showallaftersearch> : currently not in a search result screen");
    }
    break;

  default:
    out += "<!-- " + tag_name + " is not supported yet -->";
    CDEBUG("tag_camas_mailindex: " + tag_name + " is not supported yet");
  } // switch

  return out;
}

/*                                                                             
 * Local Variables:                                                            
 * c-basic-offset: 2                                                           
 * End:                                                                        
 *                                                                             
 * vim: softtabstop=2 tabstop=2 expandtab autoindent formatoptions=croqlt smartindent cindent shiftwidth=2
 */
