/*
 * Calcurse - text-based organizer
 * Copyright (c) 2004-2005 Frederic Culot
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Send your feedback or comments to : calcurse@culot.org
 * Calcurse home page : http://culot.org/calcurse
 *
 */

#include <ncurses.h>		/* ncurses.h includes stdio.h */
#include <panel.h>
#include <time.h>		/* to get the date */
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "io.h"
#include "help.h"
#include "calendar.h"
#include "utils.h"
#include "vars.h"
#include "event.h"
#include "todo.h"
#include "args.h"

struct tm *ptrtime;

time_t timer;
char current_day[3];
char current_month[3];
char current_year[5];
char current_time[15];		//time in the format : hh:mmPM
char cal_date[30];		// date to be written in calendar
int year, month, day;
int sel_year, sel_month, sel_day;
int x_cal, y_cal, x_app, y_app, x_tod, y_tod;
int nl_cal, nc_cal, nl_app, nc_app, nl_tod, nc_tod;
int nb_tod, hilt_tod, hilt_app, sav_hilt_tod, sav_hilt_app;
int which_pan;
int show_event;
int number_events_inday;
int first_todo_onscreen;
int first_app_onscreen;

/* Variables for user configuration */
int colr, layout;
char confirm_quit[4] = "yes";
char confirm_delete[4] = "yes";
char auto_save[4] = "yes";
char really_quit[4] = "ini";

/* Variables for drawing screen */
WINDOW *my_wins[3];
PANEL *my_panels[3];
PANEL *top;

char event_start_time[100];
char event_end_time[100];

 /* External functions */
void get_date();
void init_wins(WINDOW ** wins, int n);
void win_show(WINDOW * win, char *label, int label_color);
void add_todo();
void add_apts();
void load_conf();
void update_todo_panel();
void update_app_panel();
void extract_data(char *dst_data, const char *org, int len);
void redraw_screen();
void general_config();
void print_general_options();
void switch_option(int option, char *);
void print_option_incolor(char *option, int pos_x, int pos_y);
void del_event();

/*
 * Calcurse  is  a text-based personal organizer which helps keeping track
 * of events and everyday tasks. It contains a calendar, a 'todo' list,
 * and puts your appointments in order. The user interface is configurable,
 * and one can choose between different color schemes and layouts. 
 * All of the commands are documented within an online help system.
 */
int main(int argc, char **argv)
{
	int ch;
	int i;
	int non_interactive;

	/* parse command line arguments and initialize data paths */
	non_interactive = parse_args(argc, argv);

	/* handle command line arguments if in non-interactive mode */
	if (non_interactive) {
		return 0;
	} else {		/* start interactive mode */

		/* Begin of the interactive mode */
		initscr();	/* start the curses mode */
		cbreak();	/* control characters generate a signal */
		keypad(stdscr, TRUE);	/* enables function keys */
		noecho();	/* controls the echoing of typed characters */

		if (has_colors()) {
			start_color();

			/* 
			 * Color assignment
			 */
			init_pair(1, COLOR_RED, COLOR_BLACK);
			init_pair(2, COLOR_GREEN, COLOR_BLACK);
			init_pair(3, COLOR_BLUE, COLOR_BLACK);
			init_pair(4, COLOR_CYAN, COLOR_BLACK);
			init_pair(5, COLOR_YELLOW, COLOR_BLUE);
			init_pair(6, COLOR_BLACK, COLOR_GREEN);
			init_pair(7, COLOR_BLACK, COLOR_BLUE);
			init_pair(8, COLOR_RED, COLOR_BLUE);
			init_pair(9, COLOR_WHITE, COLOR_BLACK);
		}

		/* Variable Init */
		colr = 1;
		layout = 1;
		which_pan = 0;	/*panel number: 0=Cal, 1=Apps, 2=ToDo */
		nb_tod = 0;	/* number of tasks in the todo list */
		hilt_tod = 0;	/* no todo event highlighted */
		hilt_app = 0;	/* no appointment highlighted */
		show_event = 0;	/*if =1, then show event in a popup */
		first_todo_onscreen = 1;
		first_app_onscreen = 1;
		get_date();	/* get the current date from system */
		/* Windows creation */
		redraw_screen();
		/* checks if data files exist, if not : create them */
		check_data_files(colr);
		load_conf();	/* loads the user configuration from file */
		redraw_screen();	// to take into account the user's layout configuration
		nb_tod = load_todo();	/* loads data for ToDo panel from file */
		load_app();	/* loads data for the Appointment panel */
		border_color(my_wins[0], colr);	/* here we choose to begin in calendar mode */
		border_color(my_wins[1], 9);	/* and we put colors around the calendar view */
		border_color(my_wins[2], 9);	/* but we leave the two other panels in normal color */
		update_app_panel();	/* update the appointment panel */
		update_todo_panel();
		update_cal_panel(y_cal, x_cal, nl_cal, nc_cal, sel_month,
				 sel_year, sel_day, day, month, year);
		/* loads the calendar */
		status_bar(which_pan, colr);


		/* ==== User input ==== */
		for (;;) {
			/* check if the terminal is large enough to display calcurse */
			getmaxyx(stdscr, row, col);
			if ((col < 80) | (row < 24)) {
				clear();
				mvprintw(1, 1,
					 "Please resize your terminal screen");
				mvprintw(2, 1, "(to at least 80x24)");
				mvprintw(3, 1, "and restart CalCurse");
				getch();
				return 1;
			}
			/* get user input */
			ch = getch();
			switch (ch) {

			case 9:	/* The TAB key was hit */
				/* save the previous highlighted event */
				if (which_pan == 2) {
					sav_hilt_tod = hilt_tod;
					hilt_tod = 0;
				}
				if (which_pan == 1) {
					sav_hilt_app = hilt_app;
					hilt_app = 0;
				}
				/* switch to the selected panel */
				if (which_pan == 2)
					which_pan = 0;
				else
					++which_pan;

				/* select the event to highlight */
				if (which_pan == 1) {
					if ((sav_hilt_app ==
					     0) & (number_events_inday !=
						   0))
						hilt_app = 1;
					else
						hilt_app = sav_hilt_app;
				} else if (which_pan == 2) {
					if ((sav_hilt_tod ==
					     0) & (nb_tod != 0))
						hilt_tod = 1;
					else
						hilt_tod = sav_hilt_tod;
				}
				break;

			case 'R':
			case 'r':	/* redraw screen */
				redraw_screen();
				break;

			case 'G':
			case 'g':	/* goto function */
				erase_status_bar();
				get_date();
				goto_day(colr, day, month, year,
					 &sel_day, &sel_month, &sel_year);
				break;

			case 'V':
			case 'v':
				if ((which_pan == 1) & (hilt_app != 0)) {
					show_event = 1;
					update_app_panel();
					show_event = 0;
					doupdate();
				}
				if ((which_pan == 2) & (hilt_tod != 0)) {
					show_event = 1;
					update_todo_panel();
					show_event = 0;
					doupdate();
				}
				break;

			case 'C':
			case 'c':
				erase_status_bar();
				config_bar(colr);
				while ((ch = getch()) != 'q') {
					switch (ch) {
					case 'C':
					case 'c':
						colr =
						    color_config(colr);
						break;
					case 'L':
					case 'l':
						layout =
						    layout_config(layout, colr);
						break;
					case 'G':
					case 'g':
						general_config();
						break;
					}
					clear();
					redraw_screen();
					erase_status_bar();
					config_bar(colr);
				}
				redraw_screen();
				break;

			case 'A':
			case 'a':
				if (which_pan == 1)
					add_apts();	//Add an item in the Appointment list
				if (which_pan == 2)
					add_todo();	//Add an item in the ToDo list
				break;

			case 'D':
			case 'd':
				del_event();
				break;

				/* calls the help screen */
			case '?':
				status_bar(which_pan, colr);
				help_screen(which_pan, colr);
				doupdate();
				break;

			case 'S':
			case 's':
				save_cal(auto_save, confirm_quit,
					 confirm_delete, colr, layout);
				break;

			case (261):	// right arrow
			case ('L'):
			case ('l'):
				if (which_pan == 0) {
					if ((sel_day == 31) & (sel_month == 12))	//next year
					{
						sel_day = 0;
						sel_month = 1;
						sel_year++;
					}
					if (sel_day == days[sel_month - 1])	//next month
					{
						sel_month = sel_month + 1;
						sel_day = 1;
					} else
						sel_day = sel_day + 1;
				}
				break;

			case (260):	// left arrow
			case ('H'):
			case ('h'):
				if (which_pan == 0) {
					if ((sel_day == 1) & (sel_month == 1))	//previous year
					{
						sel_day = 32;
						sel_month = 12;
						sel_year--;
					}
					if (sel_day == 1)	//previous month
					{
						sel_day =
						    days[sel_month - 2];
						sel_month = sel_month - 1;
					} else
						sel_day = sel_day - 1;
				}
				break;

			case (259):	// up arrow
			case ('K'):
			case ('k'):
				if (which_pan == 0) {
					if ((sel_day <= 7) & (sel_month == 1))	//previous year
					{
						sel_day =
						    31 - (7 - sel_day);
						sel_month = 12;
						sel_year--;
						break;
					}
					if (sel_day <= 7)	// previous month
					{
						sel_day =
						    days[sel_month - 2] -
						    (7 - sel_day);
						sel_month = sel_month - 1;
					} else
						sel_day = sel_day - 7;	//previous week
				} else {
					if ((which_pan == 1) & (hilt_app >
								1)) {
						--hilt_app;
						if (hilt_app <
						    first_app_onscreen)
							--first_app_onscreen;	//scroll down one line
					}
					if ((which_pan == 2) & (hilt_tod >
								1)) {
						--hilt_tod;
						if (hilt_tod <
						    first_todo_onscreen)
							--first_todo_onscreen;	//scroll up one line
					}
				}
				break;

			case (258):	// down arrow
			case ('J'):
			case ('j'):
				if (which_pan == 0) {
					if ((sel_day >= days[sel_month - 1] - 7) & (sel_month == 12))	//next year
					{
						sel_day =
						    (7 - (31 - sel_day));
						sel_month = 1;
						sel_year++;
						break;
					}
					if (sel_day > days[sel_month - 1] - 7)	//next month
					{
						sel_day =
						    (7 -
						     (days[sel_month - 1] -
						      sel_day));
						sel_month = sel_month + 1;
					} else
						sel_day = sel_day + 7;	// next week
				} else {
					if ((which_pan == 1) & (hilt_app <
								number_events_inday))
					{
						++hilt_app;
						if (hilt_app -
						    first_app_onscreen ==
						    floor((nl_app -
							   4) / 3))
							++first_app_onscreen;	//scroll down one line
					}
					if ((which_pan == 2) & (hilt_tod <
								nb_tod)) {
						++hilt_tod;
						if (hilt_tod -
						    first_todo_onscreen ==
						    nl_tod - 4)
							++first_todo_onscreen;	//scroll down one line
					}
				}
				break;

			case ('Q'):
			case ('q'):
				if (strncmp(auto_save, "yes", 3) == 0)
					save_cal(auto_save,confirm_quit,
						 confirm_delete,
						 colr, layout);

				if (strncmp(confirm_quit, "yes", 3) == 0) {
					while ((strncmp
						(really_quit, "yes",
						 3) !=
						0) & (strncmp(really_quit,
							      "no",
							      3) != 0)) {
						erase_status_bar();
						attron(COLOR_PAIR(colr));
						mvprintw(row - 2, 0,
							 "Do you really want to quit ? [yes or no]");
						getstring(really_quit, 0,
							  row - 1);
						attroff(COLOR_PAIR(colr));
					}
					if (strncmp(really_quit, "yes", 3)
					    == 0) {
						endwin();
						return 0;
					} else {
						erase_status_bar();
						strcpy(really_quit, "ini");
						break;
					}
				} else {
					endwin();
					return 0;
				}
				break;

			}	//end case statement

			if (which_pan == 0) {	/* puts color on the selected panel's border */
				border_color(my_wins[0], colr);
				border_color(my_wins[1], 9);
				border_color(my_wins[2], 9);
			}
			if (which_pan == 1) {
				border_color(my_wins[1], colr);
				border_color(my_wins[2], 9);
				border_color(my_wins[0], 9);
			}
			if (which_pan == 2) {
				border_color(my_wins[2], colr);
				border_color(my_wins[0], 9);
				border_color(my_wins[1], 9);
			}
			top = (PANEL *) panel_userptr(top);
			top_panel(top);

			status_bar(which_pan, colr);
			update_panels();

			update_todo_panel();
			update_app_panel();
			update_cal_panel(y_cal, x_cal, nl_cal, nc_cal,
					 sel_month, sel_year, sel_day, day,
					 month, year);
		}
	}

}				/* end of interactive mode */

/* ==== External functions ==== */

  /* redraw screen when 'R' is pressed */
void redraw_screen()
{
	clear();

	/* Get the screen configuration */
	getmaxyx(stdscr, row, col);

	/* window placement definition */
	nl_cal = 12;
	nc_cal = 30;
	nc_app = col - nc_cal;
	nl_app = row - 2;
	nc_tod = nc_cal;
	nl_tod = row - (nl_cal + 2);
	nl_app = row - 2;

	/* defining the layout */
	switch (layout) {
	case 1:
		y_app = 0;
		x_app = 0;
		x_tod = nc_app;
		y_tod = nl_cal;
		x_cal = nc_app;
		y_cal = 0;
		break;
	case 2:
		y_app = 0;
		x_app = 0;
		x_tod = nc_app;
		y_tod = 0;
		x_cal = nc_app;
		y_cal = nl_tod;
		break;
	case 3:
		y_app = 0;
		x_app = nc_cal;
		x_tod = 0;
		y_tod = nl_cal;
		x_cal = 0;
		y_cal = 0;
		break;
	case 4:
		y_app = 0;
		x_app = nc_cal;
		x_tod = 0;
		y_tod = 0;
		x_cal = 0;
		y_cal = nl_tod;
		break;
	}

	init_wins(my_wins, 3);

	/* Attach a panel to each window *//* Order is bottom up */
	my_panels[0] = new_panel(my_wins[0]);	/* Push 0, order: stdscr-0 */
	my_panels[1] = new_panel(my_wins[1]);	/* Push 1, order: stdscr-0-1 */
	my_panels[2] = new_panel(my_wins[2]);	/* Push 2, order: stdscr-0-1-2 */

	/* Set up the user pointers to the next panel */
	set_panel_userptr(my_panels[0], my_panels[1]);
	set_panel_userptr(my_panels[1], my_panels[2]);
	set_panel_userptr(my_panels[2], my_panels[0]);

	/* Update the stacking order. calendar panel will be on top */
	top = my_panels[2];
	update_panels();
	doupdate();
	update_app_panel();	/* update the appointment panel */
	update_todo_panel();
	update_cal_panel(y_cal, x_cal, nl_cal, nc_cal, sel_month, sel_year,
			 sel_day, day, month, year);
	/* loads the calendar */
	status_bar(which_pan, colr);
}



/* Get current date */
void get_date()
{
	timer = time(NULL);
	ptrtime = localtime(&timer);
	strftime(current_time, 15, "%H:%M%p", ptrtime);	//to get the time
	strftime(cal_date, 30, "%a %B %Y", ptrtime);
	strftime(current_day, 3, "%d", ptrtime);
	strftime(current_month, 3, "%m", ptrtime);
	strftime(current_year, 5, "%Y", ptrtime);
	month = atoi(current_month);
	day = atoi(current_day);
	year = atoi(current_year);
	sel_year = year;
	sel_month = month;
	sel_day = day;
}

/* Put all the windows */
void init_wins(WINDOW ** wins, int n)
{
	char label[80];

	wins[0] = newwin(nl_cal, nc_cal, y_cal, x_cal);
	sprintf(label, "Calendar");
	win_show(wins[0], label, colr);
	wins[1] = newwin(nl_app, nc_app, y_app, x_app);
	sprintf(label, "Appointments");
	win_show(wins[1], label, colr);
	wins[2] = newwin(nl_tod, nc_tod, y_tod, x_tod);
	sprintf(label, "ToDo");
	win_show(wins[2], label, colr);
}

/* Show the window with a border and a label */
void win_show(WINDOW * win, char *label, int label_color)
{
	int startx, starty, height, width;

	getbegyx(win, starty, startx);
	getmaxyx(win, height, width);

	box(win, 0, 0);
	mvwaddch(win, 2, 0, ACS_LTEE);
	mvwhline(win, 2, 1, ACS_HLINE, width - 2);
	mvwaddch(win, 2, width - 1, ACS_RTEE);

	print_in_middle(win, 1, 0, width, label, COLOR_PAIR(label_color));
}

/* General configuration */
void general_config()
{
	WINDOW *conf_win;
	PANEL *conf_panel;
	char label[80];
	int ch;

	clear();
	conf_win = newwin(row - 2, col, 0, 0);
	box(conf_win, 0, 0);
	sprintf(label, "CalCurse %s | general options", VERSION);
	win_show(conf_win, label, colr);
	conf_panel = new_panel(conf_win);
	attron(COLOR_PAIR(colr));
	mvprintw(row - 2, 2,
		 "Enter the option number to change its value [Q to quit] : ");
	attroff(COLOR_PAIR(colr));
	update_panels();
	doupdate();
	print_general_options();
	while ((ch = getch()) != 'q') {
		switch (ch) {
		case '1':	// auto-save
			switch_option(1, auto_save);
			break;
		case '2':	// confirm-quit
			switch_option(2, confirm_quit);
			break;
		case '3':	// confirm-delete
			switch_option(3, confirm_delete);
			break;
		}
		print_general_options();
	}
	delwin(conf_win);
}

/* prints the general options */
void print_general_options()
{
	int x_pos, y_pos;
	char option1[] = "auto_save = ";
	char option2[] = "confirm_quit = ";
	char option3[] = "confirm_delete = ";

	x_pos = 3;
	y_pos = 4;

	mvprintw(y_pos, x_pos, "[1] %s      ", option1);
	print_option_incolor(auto_save, y_pos,
			     x_pos + 4 + strlen(option1));
	mvprintw(y_pos + 1, x_pos,
		 "(if set to YES, automatic save is done when quitting)");

	mvprintw(y_pos + 3, x_pos, "[2] %s      ", option2);
	print_option_incolor(confirm_quit, y_pos + 3,
			     x_pos + 4 + strlen(option2));
	mvprintw(y_pos + 4, x_pos,
		 "(if set to YES, confirmation is required before quitting)");

	mvprintw(y_pos + 6, x_pos, "[3] %s      ", option3);
	print_option_incolor(confirm_delete, y_pos + 6,
			     x_pos + 4 + strlen(option3));
	mvprintw(y_pos + 7, x_pos,
		 "(if set to YES, confirmation is required before deleting an event)");

	refresh();
}

/* print the option with appropriate color */
void print_option_incolor(char *option, int pos_y, int pos_x)
{
	int color;

	if (strncmp(option, "yes", 3) == 0)
		color = 2;
	else if (strncmp(option, "no", 2) == 0)
		color = 1;
	else {
		clear();
		mvprintw(1, 1,
			 "option not defined - Problem in print_option_incolor()");
		exit(1);
	}
	attron(COLOR_PAIR(color));
	mvprintw(pos_y, pos_x, "%s", option);
	attroff(COLOR_PAIR(color));
	refresh();
}

  /* Change an option value */
void switch_option(int option, char *opt)
{
	if (strncmp(opt, "yes", 3) == 0) {
		strcpy(opt, "no ");
	} else if (strncmp(opt, "no", 2) == 0) {
		strcpy(opt, "yes");
	}
	if (option == 1)
		strcpy(auto_save, opt);
	if (option == 2)
		strcpy(confirm_quit, opt);
	if (option == 3)
		strcpy(confirm_delete, opt);
}

  /* Delete an event from the ToDo or Appointment lists */
void del_event()
{
	char really_delete[3] = "ini";
	long date;

	if (which_pan == 1 && hilt_app != 0) {	//appointment event to delete
		date = date2sec(sel_year, sel_month, sel_day, 0, 0);
		if (strncmp(confirm_delete, "yes", 3) == 0) {
			while ((strncmp(really_delete, "yes", 3) != 0) &
			       (strncmp(really_delete, "no", 3) != 0)) {
				erase_status_bar();
				attron(COLOR_PAIR(colr));
				mvprintw(row - 2, 0,
					 "Do you really want to delete this Appointment ? [yes or no]");
				getstring(really_delete, 0, row - 1);
				attroff(COLOR_PAIR(colr));
			}
			if (strncmp(really_delete, "yes", 3) == 0) {
				if (number_events_inday != 0) {
					event_delete_bynum(date, hilt_app - 1);	//todo number to delete
					--number_events_inday;
				}
			} else {
				erase_status_bar();
				strcpy(really_delete, "ini");
				return;
			}
		} else {
			if (number_events_inday != 0) {
				event_delete_bynum(date, hilt_app - 1);
				--number_events_inday;
			}
		}

	} else if (which_pan == 2 && hilt_tod != 0) {	// todo event to delete
		if (strncmp(confirm_delete, "yes", 3) == 0) {
			while ((strncmp(really_delete, "yes", 3) != 0) &
			       (strncmp(really_delete, "no", 3) != 0)) {
				erase_status_bar();
				attron(COLOR_PAIR(colr));
				mvprintw(row - 2, 0,
					 "Do you really want to delete this ToDo ? [yes or no]");
				getstring(really_delete, 0, row - 1);
				attroff(COLOR_PAIR(colr));
			}
			if (strncmp(really_delete, "yes", 3) == 0) {
				if (nb_tod > 0) {
					todo_delete_bynum(hilt_tod - 1);
					//todo number to delete
					--nb_tod;
				}
			} else {
				erase_status_bar();
				strcpy(really_delete, "ini");
				return;
			}
		} else {
			if (nb_tod > 0) {
				todo_delete_bynum(hilt_tod - 1);
				//todo number to delete
				--nb_tod;
			}
		}
	}
}

  /* Adds an item in the ToDo list */
void add_todo()
{
	char mesg[] = "Enter the new ToDo event : ";
	char todo_input[500];

	erase_status_bar();
	mvprintw(row - 2, 0, "%s", mesg);
	refresh();
	getstring(todo_input, 0, row - 1);
	if (strlen(todo_input) != 0) {
		todo_insert(todo_input);
		++nb_tod;
		update_todo_panel();
	}
	erase_status_bar();
	status_bar(which_pan, colr);
}

  /* Adds An Item In The Appointments List */
void add_apts()
{
	char mesg_1[] = "Enter the start time [hh:mm] : ";
	char mesg_2[] = "Enter the event duration [in minutes] : ";
	char mesg_3[] = "Enter the event description :";
	char format_message_1[] = "You entered an invalid start time";
	char format_message_2[] =
	    "Please enter the event duration in minutes";
	int i;
	char event_time[500];
	char event_mesg[500];
	long event_duration;
	struct event_s *event_pointeur;
	unsigned heures, minutes, duration;

	// get the starting time
	strcpy(event_time, "     ");
	while (check_event_time(event_time) == 0) {
		erase_status_bar();
		mvprintw(row - 2, 0, "%s", mesg_1);
		refresh();
		getstring(event_time, 0, row - 1);
		if (strlen(event_time) == 0)
			return;	//nothing entered, cancel event add
		if (check_event_time(event_time) == 0) {
			erase_status_bar();
			attron(COLOR_PAIR(colr));
			mvprintw(row - 2, 0, "%s", format_message_1);
			mvprintw(row - 1, 0, "Press [Enter] to continue");
			attroff(COLOR_PAIR(colr));
			getch();
		} else
			sscanf(event_time, "%u:%u", &heures, &minutes);
	}
	// get the event duration
	strcpy(event_time, " ");
	while (is_all_digit(event_time) == 0) {
		erase_status_bar();
		mvprintw(row - 2, 0, "%s", mesg_2);
		refresh();
		getstring(event_time, 0, row - 1);
		if (strlen(event_time) == 0)
			return;	//if nothing entered, cancel event add
		if (is_all_digit(event_time) == 0) {
			erase_status_bar();
			attron(COLOR_PAIR(colr));
			mvprintw(row - 2, 0, "%s", format_message_2);
			mvprintw(row - 1, 0, "Press [Enter] to continue");
			attroff(COLOR_PAIR(colr));
			getch();
		} else
			event_duration = atoi(event_time);
	}
	// get the appointment description
	erase_status_bar();
	mvprintw(row - 2, 0, "%s", mesg_3);
	refresh();
	getstring(event_mesg, 0, row - 1);
	if (strlen(event_mesg) != 0) {
		// insert the appointment in list
		event_pointeur =
		    event_new(event_mesg,
			      date2sec(sel_year, sel_month, sel_day,
				       heures, minutes),
			      min2sec(event_duration));
		// update the panel
		update_app_panel();
	}
	erase_status_bar();
	status_bar(which_pan, colr);
}

  /* Updates the ToDo panel */
void update_todo_panel()
{
	WINDOW *popup_win;
	struct todo_s *i;
	int len = nc_tod - 8;
	char buf[len];
	int num;
	int y_offset;

	num = 0;
	y_offset = 0;
	if (first_todo_onscreen > 1) {
		attron(COLOR_PAIR(colr));
		mvprintw(y_tod + 3, x_tod + 1, "^");
		attroff(COLOR_PAIR(colr));
		refresh();
	}
	for (i = todolist; i != 0; i = i->next) {
		num++;
		y_offset = y_tod + 3 + num - first_todo_onscreen;
		if (num >= first_todo_onscreen) {	// checks the first item to print
			if (num - first_todo_onscreen < nl_tod - 4) {	//checks the available line space
				if (show_event == 0) {	// checks if normal or popup mode
					if (num == hilt_tod)
						attron(COLOR_PAIR(colr));	//checks the selected item
					if (strlen(i->mesg) < len)
						mvprintw(y_offset,
							 x_tod + 2, "- %s",
							 i->mesg);
					else {
						strncpy(buf, i->mesg,
							len - 1);
						buf[len - 1] = '\0';
						mvprintw(y_offset,
							 x_tod + 2,
							 "- %s...", buf);
					}
					if (num == hilt_tod)
						attroff(COLOR_PAIR(colr));
					refresh();
				} else {	//popup a window with the selected item
					if (num == hilt_tod) {
						popup_win =
						    popup(row - 4, col - 2,
							  1, 1, "To Do :",
							  colr);
						scroller(i->mesg, 1, 1,
							 row - 4, col - 2,
							 colr);
						getch();
						delwin(popup_win);
					}
				}	//end if normal or popup mode
			}
			if (num - first_todo_onscreen >= nl_tod - 4) {	//prints "v" at end of display if too many items
				attron(COLOR_PAIR(colr));
				mvprintw(y_tod + nl_tod - 2, x_tod + 1,
					 "v");
				attroff(COLOR_PAIR(colr));
				refresh();
			}	// end if num < nl_tod -4 loop
		}		// end if num >= first_todo_onscreen loop
	}			//end for loop
}

  /* Updates the Appointment panel */
void update_app_panel()
{
	WINDOW *popup_win;
	int j, pos_x;
	struct event_s *i;
	long date_start;
	int len = nc_app - 8;
	char buf[len];
	int max_events = floor((nl_app - 4) / 3);
	int y_offset;

	number_events_inday = 0;
	y_offset = 0;

	// prints the selected date 
	pos_x = x_app + nc_app - (strlen(monthnames[sel_month - 1]) + 11);
	attron(COLOR_PAIR(colr));
	if (sel_day < 10)
		mvprintw(y_app + 3, pos_x, " %s %d, %d",
			 monthnames[sel_month - 1], sel_day, sel_year);
	else
		mvprintw(y_app + 3, pos_x, "%s %d, %d",
			 monthnames[sel_month - 1], sel_day, sel_year);
	refresh();
	attroff(COLOR_PAIR(colr));
// prints the appointments for selected day
	date_start = date2sec(sel_year, sel_month, sel_day, 0, 0);
	if (first_app_onscreen > 1) {
		attron(COLOR_PAIR(colr));
		mvprintw(y_app + 3, x_app + 1, "^");
		attroff(COLOR_PAIR(colr));
		refresh();
	}

	for (i = eventlist; i != 0; i = i->next) {
		if (event_inday(i, date_start)) {
			number_events_inday++;
			y_offset =
			    y_app + 8 + 3 * (number_events_inday - 1) -
			    (first_app_onscreen * 3);
			event_sec2str(i, date_start, event_start_time,
				      event_end_time);
			if (number_events_inday >= first_app_onscreen) {
				if (number_events_inday -
				    first_app_onscreen < max_events) {
					if (show_event == 0)	//no popup mode
					{
						if (number_events_inday ==
						    hilt_app)
							attron(COLOR_PAIR
							       (colr));
						mvprintw(y_offset - 1,
							 x_app + 1,
							 " - %s -> %s",
							 event_start_time,
							 event_end_time);
						//check message length
						if (strlen(i->mesg) < len)
							mvprintw(y_offset,
								 x_app + 5,
								 "%s",
								 i->mesg);
						else {
							strncpy(buf,
								i->mesg,
								len - 1);
							buf[len - 1] =
							    '\0';
							mvprintw(y_offset,
								 x_app + 5,
								 "%s...",
								 buf);
						}

						if (number_events_inday ==
						    hilt_app)
							attroff(COLOR_PAIR
								(colr));
						refresh();
					} else	//popup mode
					{
						if (number_events_inday ==
						    hilt_app) {
							popup_win =
							    popup(row - 4,
								  col - 2,
								  1, 1,
								  "Appointment :",
								  colr);
							mvprintw(4, 4,
								 " - %s -> %s",
								 event_start_time,
								 event_end_time);
							scroller(i->mesg,
								 1, 1,
								 row - 4,
								 col - 2,
								 colr);
							getch();
							delwin(popup_win);
						}
					}	//end if normal or popup mode
				}
				if (number_events_inday -
				    first_app_onscreen >= max_events) {
					attron(COLOR_PAIR(colr));
					mvprintw(y_app + nl_app - 2,
						 x_app + 1, "v");
					attroff(COLOR_PAIR(colr));
					refresh();
				}
			}
		}
	}
}

  /* Load the user configuration */
void load_conf()
{
	FILE *data_file;
	char mesg[] = "failed to open config file";
	char buf[100], e_conf[100];
	int var;

	data_file = fopen(path_conf, "r");
	if (data_file == NULL) {
		cal_error(mesg);
	}
	refresh();
	var = 0;
	for (;;) {
		if (fgets(buf, 99, data_file) == NULL) {
			break;
		}
		extract_data(e_conf, buf, strlen(buf));
		if (var == 1) {
			strcpy(auto_save, e_conf);
			var = 0;
		} else if (var == 2) {
			strcpy(confirm_quit, e_conf);
			var = 0;
		} else if (var == 3) {
			strcpy(confirm_delete, e_conf);
			var = 0;
		} else if (var == 4) {
			colr = atoi(e_conf);
			var = 0;
		} else if (var == 5) {
			layout = atoi(e_conf);
			var = 0;
		}
		if (strncmp(e_conf, "auto_save=", 10) == 0)
			var = 1;
		else if (strncmp(e_conf, "confirm_quit=", 13) == 0)
			var = 2;
		else if (strncmp(e_conf, "confirm_delete=", 15) == 0)
			var = 3;
		else if (strncmp(e_conf, "color-theme=", 12) == 0)
			var = 4;
		else if (strncmp(e_conf, "layout=", 7) == 0)
			var = 5;
	}
	fclose(data_file);
	erase_status_bar();
}

  /* get data from file */
void extract_data(char *dst_data, const char *org, int len)
{
	for (;;) {
		if (*org == '\n' || *org == '\0')
			break;
		*dst_data++ = *org++;
	}
	*dst_data = '\0';
}
