# Copyright (C) 2009 Aaron Bentley
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA


from bzrlib import (
    builtins,
    errors,
    merge,
    revision as _mod_revision, trace,
    ui,
)
from bzrlib.branch import Branch
from bzrlib.bzrdir import BzrDir
from bzrlib.commands import Command
from bzrlib.option import Option, RegistryOption
from bzrlib.osutils import pathjoin
from bzrlib.switch import switch
from bzrlib.workingtree import WorkingTree
from bzrlib.plugins.pipeline.pipeline import (
    dwim_pipe,
    NoSuchPipe,
    PipeManager,
    PipeStorage,
    tree_to_pipeline,
)


def is_light_checkout(tree):
    return (tree.branch.bzrdir.root_transport.base !=
            tree.bzrdir.root_transport.base)


def require_light_checkout(tree):
    if not is_light_checkout(tree):
        raise errors.BzrCommandError('Directory is not a lightweight'
                                     ' checkout.')


class PipeCommand(Command):

    def _get_manager(self, location, after, before):
        tree, manager = self._get_checkout_manager(location, allow_tree=True)
        if after is not None:
            if before is not None:
                raise errors.BzrCommandError('Cannot specify --before and'
                                             ' --after.')
            manager = PipeManager(manager.storage.find_pipe(after), tree)
        if before is not None:
            manager = PipeManager(manager.storage.find_pipe(before), tree)
        return tree, manager

    def _get_location_manager(self, location='.'):
        branch = Branch.open_containing(location)[0]
        return PipeManager(branch, None)

    @staticmethod
    def _get_tree_and_branch(location, tree_optional):
        if tree_optional:
            tree, branch = BzrDir.open_containing_tree_or_branch(location)[:2]
        else:
            tree = WorkingTree.open_containing(location)[0]
            branch = tree.branch
        return tree, branch

    @classmethod
    def _get_checkout_manager(klass, location='.', checkout_optional=False,
                              pipe=None, allow_tree=False):
        try:
            tree, branch = klass._get_tree_and_branch('.', checkout_optional)
            branch = PipeStorage(branch).find_pipe(location)
        except (errors.NotBranchError, NoSuchPipe):
            tree, branch = klass._get_tree_and_branch(location,
                                                      checkout_optional)
        if tree is not None and not allow_tree:
            require_light_checkout(tree)
        manager = PipeManager(branch, tree)
        if pipe is not None:
            manager = PipeManager(manager.storage.find_pipe(pipe), tree)
        return tree, manager

    def _get_revision_id(self, branch, revision, manager, insert_before):
        if revision is None:
            if not insert_before:
                return None
            prev_revision_id = manager.get_prev_revision_id()
            if prev_revision_id is not None:
                return prev_revision_id
            else:
                return branch.last_revision()
        if len(revision) > 1:
            raise errors.BzrCommandError('Only one revision may be supplied.')
        return revision[0].as_revision_id(branch)


class cmd_add_pipe(PipeCommand):
    """Add a pipe to the pipeline.

    By default, the pipe is added after the current active pipe, at the
    current revision.

    This command can be used to start a new pipeline.
    """

    takes_args = ['pipe', 'neighbour?']
    takes_options = [RegistryOption.from_kwargs('position',
                     'Position to insert the new pipe at.',
                     after='Insert after the selected pipe.',
                     before='Insert before the selected pipe.',
                     value_switches=True, enum_switch=False),
                     Option('interactive', short_name='i',
                             help='Interactively decide which changes to place'
                             ' in the new pipe.'),
                     Option('no-switch', help="Do not switch to new pipe."),
                     Option('directory', short_name='d', type=unicode,
                            help='Directory of the pipe to add to, rather than'
                            ' the one for the working directory.'),
                     'revision',]

    def run(self, pipe, neighbour=None, revision=None, interactive=False,
            no_switch=False, directory=None, position='after'):
        tree, manager = self._get_checkout_manager(directory, pipe=neighbour,
            allow_tree=True)
        if not is_light_checkout(tree):
            raise errors.BzrCommandError('add-pipe should be run in a '
                'lightweight checkout.  See bzr help pipeline for details.')
        insert_before = (position == 'before')
        tree.branch.lock_read()
        try:
            revision_id = self._get_revision_id(tree.branch, revision,
                                                manager, insert_before)
        finally:
            tree.branch.unlock()
        try:
            new_br = Branch.open(pipe)
        except errors.NotBranchError:
            new_br = manager.storage.insert_pipe(pipe, revision_id,
                                                 before=insert_before)
        else:
            manager.storage.insert_branch(new_br, insert_before)
        if revision_id is None and no_switch:
            PipeManager(new_br, tree).store_uncommitted(interactive)
        if no_switch:
            trace.note('Created pipe "%s".' % pipe)
        if not no_switch:
            switch(tree.bzrdir, new_br)
            trace.note('Created and switched to pipe "%s".' % pipe)


class cmd_rename_pipe(PipeCommand):
    """Rename a pipe to a different name.

    This will rename the branch directory and update the pipeline metadata.
    It is not connected to the branch nick.
    """

    takes_args = ['new_name']

    def run(self, new_name):
        tree, manager = self._get_checkout_manager('.')
        manager.rename_pipe(new_name)

class cmd_merge(builtins.cmd_merge):
    #Support merge --uncommitted PIPE
    __doc__ = builtins.cmd_merge.__doc__

    def get_merger_from_uncommitted(self, tree, location, pb):
        try:
            pipe = PipeManager(dwim_pipe(PipeManager(tree.branch, tree),
                                         location), tree)
        except NoSuchPipe:
            return builtins.cmd_merge.get_merger_from_uncommitted(
                self, tree, location, pb)
        merger = pipe.make_uncommitted_merger(self)
        if merger is None:
            # Have to return a merger of some kind, so we don't try another
            # code path.
            merger = merge.Merger.from_revision_ids(None, tree,
                _mod_revision.NULL_REVISION, _mod_revision.NULL_REVISION)
        return merger


class cmd_reconfigure_pipeline(PipeCommand):
    """Reconfigure a tree with branch into a lightweight checkout of a pipe.

    The pipeline will be stored in a new "pipes" subdirectory of .bzr.

    This is suitable if you have a standalone tree, but if you have a
    shared repository with its own organization scheme already, it's probably
    better to just create a lightweight checkout.
    """

    def run(self):
        tree = WorkingTree.open_containing('.')[0]
        tree_to_pipeline(tree)


class cmd_remove_pipe(PipeCommand):
    """Remove a pipe from the pipeline.

    By default, the current pipe is removed, but a pipe may be specified as
    the first parameter.  By default, only the association of the pipe with
    its pipeline is removed, but if --branch is specified, the branch is
    also deleted.
    """

    takes_args = ['pipe?']
    takes_options = [Option('branch', help="Remove pipe's branch.")]

    def run(self, pipe=None, branch=False):
        tree, manager = self._get_checkout_manager(location=pipe,
                                                   checkout_optional=True,
                                                   allow_tree=True)
        target_branch = manager.get_next_pipe()
        if target_branch is None:
            target_branch = manager.get_prev_pipe()
        if target_branch is None:
            raise errors.BzrCommandError('Branch is not connected to a'
                                         ' pipeline.')
        if (tree is not None and
            is_light_checkout(tree) and
            tree.branch.base == manager.storage.branch.base):
            switch(tree.bzrdir, target_branch, quiet=True)
        manager.storage.disconnect()
        if branch:
            manager.storage.branch.bzrdir.root_transport.delete_tree('.')


class cmd_switch_pipe(PipeCommand):
    """Switch from one pipe to another.

    Any uncommitted changes are stored.  Any stored changes in the target
    pipe are restored.
    """
    
    aliases = ['swp']

    takes_args = ['pipe']
    takes_options = [
        Option('directory', type=unicode, short_name='d',
               help='Directory of the checkout to switch, rather than the'
                    ' current directory.')]

    def run(self, pipe, directory=None):
        checkout, manager = self._get_checkout_manager(directory)
        old = checkout.branch.nick
        target = dwim_pipe(manager, pipe)
        manager.switch_to_pipe(target)
        trace.note('Switched from "%s" to "%s".' % (old, target.nick))


class cmd_store(PipeCommand):

    """Store uncommitted changes in the pipe."""

    hidden = True

    def run(self):
        checkout, manager = self._get_checkout_manager('.')
        manager.store_uncommitted(checkout)


class cmd_show_pipeline(PipeCommand):
    """Show the current pipeline.

    All pipes are listed with the beginning of the pipeline at the top and the
    end of the pipeline at the bottom. These indicators are used::

      * - The current pipe.
      U - A pipe holding uncommitted changes.

    Uncommitted changes are automatically restored by the 'switch-pipe'
    command.
    """

    takes_args = ['location?']
    aliases = ['pipes']

    def run(self, location='.'):
        manager = self._get_location_manager(location)
        for pipe in manager.list_pipes():
            if pipe is manager.storage.branch:
                selected = '*'
            else:
                selected = ' '
            if PipeStorage(pipe).has_stored_changes():
                uncommitted = 'U'
            else:
                uncommitted = ' '
            self.outf.write('%s%s %s\n' % (selected, uncommitted, pipe.nick))


class cmd_pump(PipeCommand):
    """From this pipe onward, merge all pipes into their next pipe and commit.

    If the merge is successful, the changes are automatically committed, and
    the process repeats for the next pipe.  Eventually, the last pipe will
    have all the changes from all of the affected pipes.  On success, the
    checkout's initial state is restored.

    If the merge produces conflicts, the process aborts and no commit is
    performed.  You should resolve the conflicts, commit, and re-run pump.
    """

    takes_options = [
        Option('directory', short_name='d', type=unicode,
               help='Directory in the pipeline to pump from.'),
        Option('from-submit', help="Start from the first pipe's submit"
               " branch."),
    ]

    def run(self, directory=None, from_submit=False):
        tree, manager = self._get_checkout_manager(directory)
        if from_submit:
            manager = PipeManager(manager.get_first_pipe(), tree)
        if not manager.pipeline_merge(from_submit):
            trace.note('Please resolve conflicts, commit, and re-run pump.')


class cmd_pipe_patches(PipeCommand):
    """Export the pipeline as a collection of patches, one per pipe.

    The patch name begins with a sequence number, and ends with the pipe
    name.
    """
    takes_args = ['patch_location?']
    takes_options = [Option('directory', short_name='d', type=unicode,
               help='Directory of the pipeline.'),
    ]
    def run(self, patch_location='.', directory=None):
        checkout, manager = self._get_checkout_manager(directory,
                                                       checkout_optional=True,
                                                       allow_tree=True)
        for num, pipe in enumerate(manager.list_pipes()):
            pipe.lock_read()
            try:
                patch = PipeManager(pipe, checkout).get_patch()
            finally:
                pipe.unlock()
            if patch is None:
                continue
            filename = pathjoin(patch_location,
                                '%.2d-%s.patch' % (num, pipe.nick))
            my_file = open(filename, 'wb')
            try:
                my_file.write(patch)
            finally:
                my_file.close()


class cmd_sync_pipeline(PipeCommand):
    """Synchronise the contents of this pipeline with another copy.

    The first argument is the location of one of the pipes in the remote
    pipeline.  It defaults to the push location.  If it does not exist, the
    whole remote pipeline will be created.  If any remote pipes are missing,
    they will be created.

    The pipelines are then synchronized by pulling and pushing between
    pipes, depending on which is newer.

    If pipes have diverged, the process will abort.  You should then merge the
    remote pipe into the local pipe and re-run sync-pipeline.
    """

    takes_args = ['location?']

    def run(self, location=None):
        checkout, manager = self._get_checkout_manager(checkout_optional=True,
                                                       allow_tree=True)
        remote = None
        if location is None:
            branchless_location = None
            for pipe in manager.list_pipes():
                location = pipe.get_push_location()
                if location is not None:
                    try:
                        remote = Branch.open(location)
                    except errors.NotBranchError:
                        if branchless_location is None:
                            branchless_location = location
                        continue
                    manager = PipeManager(pipe, checkout)
                    break
            else:
                if branchless_location is not None:
                    location = branchless_location
                else:
                    raise errors.BzrCommandError(
                        'No location specified and none remembered.')
        try:
            manager.sync_pipeline(location, remote)
        finally:
            ui.ui_factory.clear_term()
