# Copyright (C) 2010 Canonical Ltd
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


from PyQt4 import QtCore, QtGui

from bzrlib import urlutils
from bzrlib.plugins.explorer.lib import (
    experimental_mode,
    location_selector,
    )


class LocationViewer(QtGui.QWidget):
    """A widget for viewing and navigating between views."""

    def __init__(self, action_map, selector_key, history_manager, *args):
        QtGui.QWidget.__init__(self, *args)
        self._action_map = action_map
        self._selector_key = selector_key
        self._history_manager = history_manager
        self._key_model_by_view = {}
        self._reopen_required_next_refresh = set()
        self.setLayout(self._layout_ui())

    def _layout_ui(self):
        # Build the view stack and selector
        self._stack = QtGui.QStackedWidget()
        self._selector = self._build_selector()
        north_ui, east_ui, west_ui, south_ui = self._selector.ui_decorators()

        # Put them together
        inner = QtGui.QVBoxLayout()
        inner.setContentsMargins(0, 0, 0, 0)
        if north_ui:
            inner.addWidget(north_ui)
        inner.addWidget(self._stack, 1)
        if south_ui:
            inner.addWidget(south_ui)
        # XXX: Maybe the West-Inner-East layout should be a QSplitter?
        # Or the selector should tell us per direction re that?
        outer = QtGui.QHBoxLayout()
        outer.setContentsMargins(0, 0, 0, 0)
        if west_ui:
            outer.addWidget(west_ui)
        outer.addLayout(inner, 1)
        if east_ui:
            outer.addWidget(east_ui)
        return outer

    def _build_selector(self):
        selector_factory = location_selector.location_selector_registry.get(
            self._selector_key)
        selector = selector_factory(self._action_map)
        self.connect(selector, QtCore.SIGNAL("currentChanged"),
            self._select_new_view)
        return selector

    def _select_new_view(self, key, model):
        if not key:
            # We've closed the last page. We could emit a key of None
            # here but doing nothing seems equally sensible.
            #self.emit(QtCore.SIGNAL("currentChanged"), key)
            return
        view_opened = False
        for i in range(0, self._stack.count()):
            widget = self._stack.widget(i)
            widget_key, widget_model = self._key_model_by_view[widget]
            if key == widget_key:
                self._stack.setCurrentIndex(i)
                view_opened = True
                break
        if not view_opened:
            #print "opening view for %s" % (key,)
            # The selector has a location we're yet to open.
            # Open it now.
            view = model.view()
            self._key_model_by_view[view] = (key, model)
            self._stack.addWidget(view)
            index = self._stack.count() - 1
            self._stack.setCurrentIndex(index)
            self._record_history(key, model)
        self.emit(QtCore.SIGNAL("currentChanged"), key)

    def add_location(self, key, model):
        """Add a tab for a location.

        :param key: a unique key for this location
        :param model: an instance of AbstractLocationModel
        """
        key = self._normalise_key(key)
        # Get the view & remember the mapping from view to model
        view = model.view()
        self._key_model_by_view[view] = (key, model)

        # Add the view and make it active
        self._stack.addWidget(view)
        self._selector.add_location(key, model)
        self._record_history(key, model)

    def _record_history(self, key, model):
        """Record this location as recently opened."""
        if self._history_manager:
            self._history_manager.add_recent(key, model.kind(),
                model.display_name())

    def _normalise_key(self, key):
        # If the key is a file URL, use the path instead
        if key.startswith("file://"):
            return urlutils.local_path_from_url(key)
        return key

    def select_view_for_key(self, key):
        """Change the selected view to be the one for key.
        
        :return: True if succeeded, False if key not found.
        """
        key = self._normalise_key(key)
        try:
            self._selector.select_key(key)
            return True
        except KeyError:
            return False

    def current_key(self):
        """What's the current key?
        
        :return: None if no locations exist.
        """
        widget = self._stack.currentWidget()
        if widget:
            key, model = self._key_model_by_view.get(widget)
            return key
        else:
            return None

    def current_model(self):
        """What's the current location model?
        
        :return: None if no locations exist.
        """
        widget = self._stack.currentWidget()
        if widget:
            key, model = self._key_model_by_view.get(widget)
            return model
        else:
            return None

    def close_current(self):
        """Close the current location."""
        current_widget = self._stack.currentWidget()
        self._stack.removeWidget(current_widget)
        self._selector.close_current()
        del self._key_model_by_view[current_widget]

    def refresh_current(self, reopen=False):
        """Refresh the current location."""
        key, model = self._key_model_by_view.get(self._stack.currentWidget())
        if reopen or key in self._reopen_required_next_refresh:
            self._reopen_required_next_refresh.discard(key)
            model.reopen_location()
            self._selector.update_current_label()
        model.refresh_view()

    def set_reopen_on_next_refresh(self):
        """Remember to reopen the current location next time it is refreshed."""
        key, model = self._key_model_by_view.get(self._stack.currentWidget())
        self._reopen_required_next_refresh.add(key)

    def refresh_view_for_key(self, key):
        """Refresh the location for key if any.
        
        :return: True if succeeded, False if key not found.
        """
        key = self._normalise_key(key)
        for i in range(0, self._stack.count()):
            widget = self._stack.widget(i)
            widget_key, model = self._key_model_by_view[widget]
            if key == widget_key:
                model.refresh_view()
                return True
        return False
