# -*- coding: utf-8 -*-
### BEGIN LICENSE
# Copyright (C) 2009 Rick Spencer rick.spencer@canonical.com
#This program is free software: you can redistribute it and/or modify it 
#under the terms of the GNU General Public License version 3, as published 
#by the Free Software Foundation.
#
#This program is distributed in the hope that it will be useful, but 
#WITHOUT ANY WARRANTY; without even the implied warranties of 
#MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
#PURPOSE.  See the GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License along 
#with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

import sys
import os

try:
 import pygtk
 pygtk.require("2.0")
 import gtk
 import gobject
 from quickly.widgets.grid_filter import (
    BlankFilterBox,
    GridFilter,
    NumericFilterBox,
    StringFilterBox,
    TagsFilterBox)
 from quickly.widgets.dictionary_grid import DictionaryGrid
 from quickly.widgets.grid_column import IntegerColumn
 import webbrowser

except Exception, inst:
 print "some dependencies for BugsPane are not available"
 raise inst

class BugsPane( gtk.VBox ):
 """BugsPane: A widget for diplaying a list of launchpad bug tasks.
 Useful  for embedding in other widgets. Supports filtering, and sorting.

 """
 def __init__(self, bug_tasks, keys=None ):
  """Create a BugsPane

  arguments:
  bug_tasks -- a gtk.ListStore full of bug_tasks fromLaunchpadUtils.get_empty_list_store

  """

  gtk.VBox.__init__( self, False, 5 )
  self.keys = keys
  self.__setup_grid(bug_tasks)
  self.__setup_popup_menu()
 
 def __setup_grid(self, bug_tasks):
  """set_up_treeview: sets up the treeview for the BugPan. 
  Sets up the headings, the treeview, the model, and moin_header string.

  arguments:
  bug_tasks -- a gtk.ListStore set up for the Pane
  """

  sw_filter = BlankFilterBox()
  sw_filter.append("=",lambda x,y: convert_to_num(x) == float(y) )
  sw_filter.append("<",lambda x,y: convert_to_num(x) < float(y) )
  sw_filter.append(">",lambda x,y: convert_to_num(x) > float(y) )
  sw_filter.append("<=",lambda x,y: convert_to_num(x) <= float(y) )
  sw_filter.append(">=",lambda x,y: convert_to_num(x) >= float(y) )

  sw_filter2 = BlankFilterBox()
  sw_filter2.append("=",lambda x,y: convert_to_num(x) == float(y) )
  sw_filter2.append("<",lambda x,y: convert_to_num(x) < float(y) )
  sw_filter2.append(">",lambda x,y: convert_to_num(x) > float(y) )
  sw_filter2.append("<=",lambda x,y: convert_to_num(x) <= float(y) )
  sw_filter2.append(">=",lambda x,y: convert_to_num(x) >= float(y) )

  filter_hints = {"status":sw_filter,"importance":sw_filter2}
  type_hints = {"gravity":IntegerColumn,"affected users":IntegerColumn}
  if self.keys is not None:
   self.grid = DictionaryGrid(bug_tasks,keys=self.keys,type_hints=type_hints,editable=False)
  else:
   self.grid = DictionaryGrid(bug_tasks,type_hints=type_hints,editable=False)
  grid_filt = GridFilter(self.grid,filter_hints)
  grid_filt.show()
  self.grid.show()
  
  #add the gridview to the top of the pane
  self.pack_start(grid_filt, False)

  #wire the treeview to the signal handlers
  self.grid.connect("row-activated",self.__row_clicked)
  self.grid.connect("button_press_event", self.__treeview_clicked)
  self.grid.connect("cursor_changed", self.selection_changed)
  self.grid.connect("move-cursor", self.__cursor_moved)
  self.grid.connect("select-all", self.selection_all)
  self.grid.connect("select-cursor-row", self.selection_changed)
  self.grid.connect("unselect-all", self.selection_none)
  self.grid.connect("toggle-cursor-row", self.selection_changed)

  #create a scrolled window for the treeview 
  grid_scroller = gtk.ScrolledWindow()
  grid_scroller.add_with_viewport(self.grid)
  grid_scroller.show()
  
  #add the scrolled window to the bottom of the pane
  self.pack_start(grid_scroller, True, True)

 def __setup_popup_menu(self):
  """" create_popup_menu: sets up a context menu to support commands directed
  at bug_tasks selected in the treeview
    
  """

  self.popup = gtk.Menu()
  open_item = gtk.MenuItem("Open in Browser")
  self.popup.append(open_item)
  open_item.show()
  open_item.connect("activate",self.__open_in_browser)

  cp_txt = gtk.MenuItem("Copy as Text")
  self.popup.append(cp_txt)
  cp_txt.show()
  cp_txt.connect("activate", self.__copy_as_text, (",","",""))

  cp_lnk = gtk.MenuItem("Copy as URL")
  self.popup.append(cp_lnk)
  cp_lnk.show()
  cp_lnk.connect("activate",self.__copy_as_url)

  cp_hyp = gtk.MenuItem("Copy as Hyperlink")
  self.popup.append(cp_hyp)
  cp_hyp.show()
  cp_hyp.connect("activate",self.__copy_as_hyper_text)

  moin = gtk.MenuItem("Copy as Moin")
  self.popup.append(moin)
  moin.show()
  moin.connect("activate",self.__copy_as_text, (" || ","|| "," ||"))

  mt = gtk.MenuItem("Copy as Moin Table")
  self.popup.append(mt)
  mt.show()
  mt.connect("activate",self.__copy_as_text, (" || ","|| "," ||"), True)

 def __treeview_clicked(self, treeview, event):
  """treeview_clicked: signal handler to detect right clicks
  and present the popup menu at the clicked location

  """
  
  #if rows were selected before the mouse click, store those rows
  rows = treeview.get_selection().get_selected_rows()
  
  if event.button == 3:  #event.button == 3 is a right mouse click
   #determine the coordinates and time of the click for passing to the popup
   x = int(event.x)
   y = int(event.y)
   time = event.time

   #find the path to the cell that was clicked
   pthinfo = treeview.get_path_at_pos(x, y)

   #pthinfo is None if the click was not on a row in the treeview
   if pthinfo is not None:
    #focus the treeview and set the cursor to the clicked row
    #and pop up the menu where the user clicked
    path, col, cellx, celly = pthinfo
    treeview.grab_focus()
    treeview.set_cursor( path, col, 0)
    self.popup.popup( None, None, None, event.button, time)

    #select any previously selected row, so the selection does not change
    for r in rows[1]:
     treeview.get_selection().select_path(r)
   return 1

 def selected_for_each(self, func, remove = False):
  """selected_for_each: runs a function for each bug_task selected in the BugsPane.
  If no bug_tasks are selected, it does nothing.

  arguments:
  func -- a function to run for each bug_task. Required signature is:
  func(bug_task), with bug_task being the Launchpad BugTask being acted upon.
  
  keyword arguments:
  remove -- if True, will remove the selected rows from the grid immediately
  """
  #extract iters
  model, rows = self.grid.get_selection().get_selected_rows()
  selected_bug_tasks = []

  iters = [model.get_iter(path) for path in rows]

  #create a list of BugTasks becuase if we remove the rows
  #we won't be able to get to them through the ListStore anymore
  for i in iters:
   selected_bug_tasks.append(model.get_value(i,10))

  #immediately remove if asked to remove for instant UI feedback
  if remove:
   for i in iters:
    gtk.gdk.threads_enter()
    self.grid.get_model().remove(i)
    gtk.gdk.threads_leave()

  #call the function for each selected bug task
  for bug_task in selected_bug_tasks:
   func(bug_task)

 def __open_in_browser(self, widget, data = None):
  """open_in_browser: signal handler for opening selected bug_tasks in the browser"""
  for row in self.grid.selected_rows:
   url = "http://bugs.launchpad.net/bugs/" + str(row["id"])
   webbrowser.open(url)    

 def __copy_as_url(self, widget, data = None ):
  """copy_as_url: signal hanlder to copy selected bugs as URLs."""
  #get the selected rows as a TreeviewModel 
  rows = self.grid.selected_rows

  #iterate through the selected rows bulding a string of URLs
  strng = ""
  i = 0
  for r in rows:
   bug_num = r["id"]
   strng += "http://bugs.launchpad.net/bugs/" + str(bug_num)

   #add a return to the string if this is not the last row 
   if i < len(rows):
    strng += "\n"
   i += 1
 
  self.__add_to_clipboard(strng)

 def __copy_as_hyper_text(self, widget, data = None ):
  """copy_as_hyper_text: signal hanler to copy selected rows as hyper text"""
  rows = self.grid.selected_rows

  strng = ""

  for i, r in enumerate(rows):
   strng += "<A src=\""
   bug_num = r["id"]
   strng += "http://bugs.launchpad.net/bugs/" + str(bug_num)
   strng += "\">"
   strng += "Bug " + str(bug_num)
   strng += ":" + str(bug_num) + ":" + r["title"]
   strng += "</A>"
   if i < len(rows):
    strng += "\n"

  self.__add_to_clipboard(strng)


 def __copy_as_text(self, widget, seperators, include_header=False):
  """copy_as_hyper_text: signal hanler to copy selected rows as hyper text"""

  delim, line_start, line_end = seperators
  
  strng = ""

  #build a header with the keys if requested
  if include_header:
   strng += line_start
   for i, k in enumerate(self.grid.keys):
    strng += k

    #don't add the delimeter to the last header
    if i < len(self.grid.keys) -1:
        strng += delim

   strng += line_end
   strng += "\n"

  #build each row
  rows = self.grid.selected_rows
  for i, r in enumerate(rows):
   strng += line_start
   for j, k in enumerate(self.grid.keys):

    #if the key does not exist in the row, just use an empty string
    try:
     if k.lower() == "id":
        cell_text = "Bug:" + str(r[k])
     else:
      cell_text = str(r[k])
    except:
     cell_text = ""     
    strng += cell_text

    #don't add the delimeter to the last item in the row
    if j < len(self.grid.keys) -1:
     strng += delim

   #finish off the row, don't include \n for the last row
   strng += line_end
   if i < len(rows) -1:
    strng += "\n"

  self.__add_to_clipboard(strng)

 def __add_to_clipboard(self, text):
  """add_to_clipboard: helper function to put text onto the system clipboard"""
  clipboard=gtk.clipboard_get(selection="CLIPBOARD")
  clipboard.set_text(text)
  
 def __row_clicked(self, treeview, path, col,  data=None):
  """row_clicked: open all the selected bugs if the user clicked on them."""
  self.__open_in_browser(treeview)
  
 #signal handlers to track selection in the treeview
 #utlmiately these all emit the "selection_changed signal
 def __cursor_moved(self, grid, step, count, data=None):
  self.selection_changed(grid)

 def selection_all(self, treeview, data=None):
  self.emit("selection_changed", self.grid.selected_rows)

 def selection_none(self, treeview, data=None):
  self.emit("selection_changed", [])

 def selection_changed(self, grid, data=None):
  self.emit("selection_changed", grid.selected_rows)

 def select_next(self):
  """select_next: public function to move the treeview cursor ahead one row.
  If no rows or the last row is selected, the first row will be selected. 

  """
  #get the current path to where the cursor is currently set
  current_path, col = self.grid.get_cursor()
  model = self.grid.get_model()
  
  #find the current row from the path, and set the next one
  if current_path == None:
    row = 0
  else:
    row = current_path[0] + 1
  if row >= len(model):
    row = 0
  self.grid.set_cursor(row)
  
 def select_previous(self):
  """select_previus: public function to move the treeview cursor back one row.
  If no row of the first row is selected, the last row will be selected.
  """

  #get the path to current cursor and the current model
  current_path, col = self.grid.get_cursor()
  model = self.grid.get_model()

  if current_path == None:
   #if no row is selected, select the last row
   if len(model) > 0:
     self.grid.set_cursor( len(model)-1 )

  else:
   #find the current row, and move selection to the next one
   #unless the current row is the last one, then select the first
   current_row = current_path[0]
   prev_row = current_row - 1
   if prev_row < 0:
    prev_row = len(model) - 1
   self.grid.set_cursor(prev_row)

 __gsignals__ = {'selection_changed' : (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE,
        (gobject.TYPE_PYOBJECT,)),
        }

def convert_to_num(string):
 val = 0.0
 try:
  val = float(string.split(" ")[0])
 except:
  print "could not convert " + string + " to float"
 return val
