#pragma implementation

#include "config.h"

#include "Config.h"
#include "Environment.h"

#include <sys/types.h>  // getpwuid, stat, mkdir, getuid
#include <sys/stat.h>   // stat, mkdir
#include <pwd.h>        // getpwuid
#include <unistd.h>     // stat, getuid

#include <errno.h>

using namespace std;

static bool getbool(xmlpp::Element* el)
{
	string val = el->get_child_text()->get_content();
	if (val == "true")
		return true;
	if (val == "false")
		return false;

	string nodename = el->get_name();
	warning("Unknown value %.*s in node %.*s\n",
			PFSTR(val), PFSTR(nodename));
	return false;
}

static void setbool(xmlpp::Element* el, bool value)
{
	el->set_child_text(value ? "true" : "false");
}


static int getint(xmlpp::Element* el)
{
	string val = el->get_child_text()->get_content();
	return atoi(val.c_str());
}

static void setint(xmlpp::Element* el, int value)
{
	el->set_child_text(stringf::fmt(value));
}

static xmlpp::Element* get_first_el(xmlpp::Element* parent, const std::string& name) throw ()
{
	xmlpp::Node::NodeList nl = parent->get_children(name);
	if (nl.empty())
		return 0;
	else
		return dynamic_cast<xmlpp::Element*>(*nl.begin());
}

Config::Config() throw (SystemException, ConsistencyCheckException) 
	: doc_conf(0), _root(0), _general(0), _view(0), _view_empty(0), _view_read(0), _locations(0)
{
	struct passwd* udata = getpwuid(getuid());
	rcfile = udata->pw_dir;
	rcfile += "/.buffy";

	struct stat rcfile_stat;
	if (stat(rcfile.c_str(), &rcfile_stat) == -1)
	{
		feedback("Creating new configuration file %.*s.\n", PFSTR(rcfile));
		
		// Create the configuration from scratch
		doc_conf = new xmlpp::Document();
		_root = doc_conf->create_root_node("buffy");
	} else {
		if (S_ISDIR(rcfile_stat.st_mode))
			throw ConsistencyCheckException(rcfile + " already exists and is a directory");
		
		if (access(rcfile.c_str(), R_OK) == -1)
			throw ConsistencyCheckException(rcfile + " already exists and is not readable");

		feedback("Reading configuration from %.*s.\n", PFSTR(rcfile));
	
		// Parse the existing config file
		try {
			// Enable when we'll have a DTD
			//_xmlparser.set_validate();
			_xmlparser.set_substitute_entities(); // Resolve/unescape text automatically
			_xmlparser.parse_file(rcfile);
			if (_xmlparser)
				doc_conf = _xmlparser.get_document();
			else
				throw ConsistencyCheckException("Parser did not parse " + rcfile);
		} catch (const std::exception& ex) {
			throw ConsistencyCheckException(string(ex.what()) + " when parsing configuration file " + rcfile);
		}
	}
	//printf("Init debtags environment; rcdir is %.*s\n", PFSTR(rcdir));
}

xmlpp::Element* Config::el_root() throw ()
{
	if (!_root)
		_root = doc_conf->get_root_node();
	return _root;
}

xmlpp::Element* Config::el_general() throw ()
{
	if (!_general)
	{
		xmlpp::Node::NodeList nl = el_root()->get_children("general");
		if (nl.empty())
			_general = el_root()->add_child("general");
		else
			_general = dynamic_cast<xmlpp::Element*>(*nl.begin());
	}
	return _general;
}

xmlpp::Element* Config::el_view() throw ()
{
	if (!_view)
	{
		xmlpp::Node::NodeList nl = el_general()->get_children("view");
		if (nl.empty())
			_view = el_general()->add_child("view");
		else
			_view = dynamic_cast<xmlpp::Element*>(*nl.begin());
	}
	return _view;
}

xmlpp::Element* Config::el_locations() throw ()
{
	if (!_locations)
	{
		xmlpp::Node::NodeList nl = el_general()->get_children("locations");
		if (nl.empty())
			_locations = el_general()->add_child("locations");
		else
			_locations = dynamic_cast<xmlpp::Element*>(*nl.begin());
	}
	return _locations;
}

xmlpp::Element* Config::el_view_empty() throw ()
{
	if (!_view_empty)
	{
		xmlpp::Node::NodeList nl = el_view()->get_children("empty");
		if (nl.empty())
		{
			_view_empty = el_view()->add_child("empty");
			setbool(_view_empty, true);
		}
		else
			_view_empty = dynamic_cast<xmlpp::Element*>(*nl.begin());
	}
	return _view_empty;
}

xmlpp::Element* Config::el_view_read() throw ()
{
	if (!_view_read)
	{
		xmlpp::Node::NodeList nl = el_view()->get_children("read");
		if (nl.empty())
		{
			_view_read = el_view()->add_child("read");
			setbool(_view_read, true);
		}
		else
			_view_read = dynamic_cast<xmlpp::Element*>(*nl.begin());
	}
	return _view_read;
}

xmlpp::Element* Config::el_view_important() throw ()
{
	if (!_view_important)
	{
		xmlpp::Node::NodeList nl = el_view()->get_children("important");
		if (nl.empty())
		{
			_view_important = el_view()->add_child("important");
			setbool(_view_important, true);
		}
		else
			_view_important = dynamic_cast<xmlpp::Element*>(*nl.begin());
	}
	return _view_important;
}

xmlpp::Element* Config::el_interval() throw ()
{
	if (!_interval)
	{
		xmlpp::Node::NodeList nl = el_general()->get_children("interval");
		if (nl.empty())
		{
			_interval = el_general()->add_child("interval");
			setint(_interval, 30);
		}
		else
			_interval = dynamic_cast<xmlpp::Element*>(*nl.begin());
	}
	return _interval;
}

xmlpp::Element* Config::el_programs() throw ()
{
	if (!_programs)
	{
		xmlpp::Node::NodeList nl = el_general()->get_children("programs");
		if (nl.empty())
			_programs = el_general()->add_child("programs");
		else
			_programs = dynamic_cast<xmlpp::Element*>(*nl.begin());
	}
	return _programs;
}


bool Config::get_view_empty() throw () { return getbool(el_view_empty()); }
bool Config::get_view_read() throw () { return getbool(el_view_read()); }
bool Config::get_view_important() throw () { return getbool(el_view_important()); }
int Config::get_update_interval() throw () { return getint(el_interval()); }

void Config::set_view_empty(bool val) throw () { setbool(el_view_empty(), val); }
void Config::set_view_read(bool val) throw () { setbool(el_view_read(), val); }
void Config::set_view_important(bool val) throw () { setbool(el_view_important(), val); }
void Config::set_update_interval(int val) throw () { setint(el_interval(), val); }


std::vector<std::string> Config::get_folder_locations() throw ()
{
	xmlpp::Node::NodeList nl = el_locations()->get_children("location");
	if (nl.empty())
	{
		struct passwd* udata = getpwuid(getuid());
		vector<string> deflt;
		deflt.push_back(string("/var/mail/") + udata->pw_name);
		deflt.push_back(string(udata->pw_dir) + "/Maildir");
		deflt.push_back(string(udata->pw_dir) + "/Mail");
		deflt.push_back(string(udata->pw_dir) + "/mail");

		for (vector<string>::const_iterator i = deflt.begin();
				i != deflt.end(); i++)
			el_locations()->add_child("location")->set_child_text(*i);

		return deflt;
	}
	else
	{
		vector<string> res;
		for (xmlpp::Node::NodeList::const_iterator i = nl.begin();
				i != nl.end(); i++)
			if (xmlpp::Element* e = dynamic_cast<xmlpp::Element*>(*i))
				res.push_back(e->get_child_text()->get_content());
		return res;
	}
}

void Config::set_folder_locations(const std::vector<std::string>& locations) throw ()
{
	xmlpp::Node::NodeList nl = el_locations()->get_children("location");

	// First remove all location nodes
	for (xmlpp::Node::NodeList::const_iterator i = nl.begin();
			i != nl.end(); i++)
		el_locations()->remove_child(*i);

	// Then add the new ones
	for (vector<string>::const_iterator i = locations.begin();
			i != locations.end(); i++)
		el_locations()->add_child("location")->set_child_text(*i);
}


std::vector<MailProgram> Config::get_mail_programs() throw ()
{
	xmlpp::Node::NodeList nl = el_programs()->get_children("mail");
	if (nl.empty())
	{
		std::vector<MailProgram> deflt;
		deflt.push_back(MailProgram("Mutt", "/usr/bin/x-terminal-emulator -e /usr/bin/mutt -f %p", true));
		deflt.push_back(MailProgram("Other", "/usr/bin/sample-mail-editor --folder %p"));

		for (vector<MailProgram>::const_iterator i = deflt.begin();
				i != deflt.end(); i++)
		{
			xmlpp::Element* item = el_programs()->add_child("mail");
			if (i->selected())
				item->set_attribute("selected", "true");
			xmlpp::Element* val = item->add_child("name");
			val->set_child_text(i->name());
			val = item->add_child("command");
			val->set_child_text(i->command());
		}

		return deflt;
	}
	else
	{
		vector<MailProgram> res;
		for (xmlpp::Node::NodeList::const_iterator i = nl.begin();
				i != nl.end(); i++)
			if (xmlpp::Element* e = dynamic_cast<xmlpp::Element*>(*i))
			{
				bool sel = false;
				if (xmlpp::Attribute* at = e->get_attribute("selected"))
					if (at->get_value() == "true")
						sel = true;

				string name;
				if (xmlpp::Element* el = get_first_el(e, "name"))
					name = el->get_child_text()->get_content();

				string command;
				if (xmlpp::Element* el = get_first_el(e, "command"))
					command = el->get_child_text()->get_content();
				
				res.push_back(MailProgram(name, command, sel));
			}
		return res;
	}
}

MailProgram Config::get_selected_mail_program() throw ()
{
	vector<MailProgram> progs = get_mail_programs();
	for (vector<MailProgram>::iterator i = progs.begin();
			i != progs.end(); i++)
		if (i->selected())
			return *i;
	return MailProgram();
}

void Config::set_mail_programs(const std::vector<MailProgram>& programs) throw ()
{
	xmlpp::Node::NodeList nl = el_programs()->get_children("mail");

	// First remove all mail nodes
	for (xmlpp::Node::NodeList::const_iterator i = nl.begin();
			i != nl.end(); i++)
		el_programs()->remove_child(*i);

	// Then add the new ones
	for (vector<MailProgram>::const_iterator i = programs.begin();
			i != programs.end(); i++)
	{
		xmlpp::Element* item = el_programs()->add_child("mail");
		if (i->selected())
			item->set_attribute("selected", "true");
		xmlpp::Element* val = item->add_child("name");
		val->set_child_text(i->name());
		val = item->add_child("command");
		val->set_child_text(i->command());
	}
}


void Config::save() throw (ConsistencyCheckException)
{
	try {
		if (doc_conf)
		{
			feedback("Saving configuration to %.*s.\n", PFSTR(rcfile));
			doc_conf->write_to_file_formatted(rcfile);
		} else
			feedback("No configuration present: ignoring save request.\n");
	} catch (const std::exception& ex) {
		throw ConsistencyCheckException(string(ex.what()) + " when saving configuration file " + rcfile);
	}
}

// vim:set ts=4 sw=4:
