#!/bin/sh

# Which executable to use for the tests.
BSDTAR=bsdtar

# Remove and recreate the directory we'll use for these tests
rm -rf /tmp/bsdtar-test
mkdir -p /tmp/bsdtar-test
cd /tmp/bsdtar-test

# We need some files to archive; generate some random files and a few
# with very long names or other special attributes
echo a: Setting up some files for basic archiving/copy tests
rm -rf a
mkdir -p a/original
cd a/original
# Create some long files with random text data
for f in f0 f1 f2 f3 f4 f5 f6 f7 f8 f9
do
    dd if=/dev/random bs=1k count=100 2>/dev/null | uuencode $f > $f
done
# A sparse file
dd if=/dev/zero of=sparse bs=1 count=1 oseek=100000 2>/dev/null
# A file with a long name
touch abcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyz
# A file with a long pathname
mkdir -p 1abcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyz/abcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyz/abcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyz/abcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyz
cd ..

# Basic test of archiving/dearchiving
echo a: Basic copy test: default format
mkdir copy-default
(cd original && ${BSDTAR} -cf - .) | (cd copy-default; ${BSDTAR} -xf -)
diff -r original copy-default || echo XXX FAILED XXX

# Exercise gzip compression (test compressed output with gunzip -t
echo Compression interoperability tests
echo a: tar -cz \| gunzip -t
(cd original && ${BSDTAR} -czf - .) | gunzip -tq

# Ensure our compression works with gunzip program
echo a: tar -cz \| gunzip \| tar -x
mkdir copy-gzip2
(cd original && ${BSDTAR} -czf - .) | gunzip -q | (cd copy-gzip2; ${BSDTAR} -xf -)
diff -r original copy-gzip2 || echo XXX FAILED XXX

# Ensure our decompression works with gzip program
echo a: tar -c \| gzip \| tar -x
mkdir copy-gunzip
(cd original && ${BSDTAR} -cf - .) | gzip | (cd copy-gunzip; ${BSDTAR} -xf -)
diff -r original copy-gunzip || echo XXX FAILED XXX

# Ensure our gzip compression/decompression work with each other
echo a: tar -cz \| tar -x
mkdir copy-gzip-gunzip
(cd original && ${BSDTAR} -czf - .) | (cd copy-gzip-gunzip; ${BSDTAR} -xf -)
diff -r original copy-gzip-gunzip || echo XXX FAILED XXX

# Ensure our decompression works with bzip2 program
echo a: tar -c \| bzip2 \| tar -x
mkdir copy-bunzip
(cd original && ${BSDTAR} -cf - .) | bzip2 | (cd copy-bunzip; ${BSDTAR} -xf -)
diff -r original copy-bunzip || echo XXX FAILED XXX

# Ensure our compression works with bunzip2 program
echo a: tar -cy \| bunzip2 \| tar -x
mkdir copy-bzip2
(cd original && ${BSDTAR} -cyf - .) | bunzip2 -q | (cd copy-bzip2; ${BSDTAR} -xf -)
diff -r original copy-bzip2 || echo XXX FAILED XXX

# Ensure our bzip2 compression/decompression work with each other
echo a: tar -cy \| tar -x
mkdir copy-bzip2-bunzip2
(cd original && ${BSDTAR} -cyf - .) | (cd copy-bzip2-bunzip2; ${BSDTAR} -xf -)
diff -r original copy-bzip2-bunzip2 || echo XXX FAILED XXX

# Filtering exercises different areas of the library.
echo a: Basic filter test: 'tar->tar'
mkdir filter-tar-tar
(cd original && ${BSDTAR} -cf - .) | ${BSDTAR} -cf - @- | (cd filter-tar-tar; ${BSDTAR} -xf -)
diff -r original filter-tar-tar || echo XXX FAILED XXX

# Filtering as format conversion
echo a: Basic filter test: 'tar->cpio'
mkdir filter-tar-cpio
(cd original && ${BSDTAR} -cf - .) | ${BSDTAR} -cf - --format=cpio @- | (cd filter-tar-cpio; ${BSDTAR} -xf -)
diff -r original filter-tar-cpio || echo XXX FAILED XXX

# Test basic --include selection logic
echo a: Filter selection
mkdir filter-tar-selected
(cd original && ${BSDTAR} -cf - .) | ${BSDTAR} -cf - --format=cpio --include=f3 @- | (cd filter-tar-selected; ${BSDTAR} -xf -)
diff -r original/f3 filter-tar-selected/f3 || echo XXX FAILED XXX
# Should be no files in copy except for 'f3'
(cd filter-tar-selected ; ls | grep -v f3 | grep .) && echo XXX FAILED XXX

# Test --include with wildcards
echo a: Wildcard filter selection
mkdir filter-tar-selected2
(cd original && ${BSDTAR} -cf - .) | ${BSDTAR} -cf - --format=cpio --include='f*' @- | (cd filter-tar-selected2; ${BSDTAR} -xf -)
for f in f1 f2 f3 f4 f5 f6 f7 f8 f9
do
    diff -r original/$f filter-tar-selected2/$f || echo XXX FAILED XXX
done
# Should be no files in copy except for 'f[0-9]'
(cd filter-tar-selected2 ; ls | grep -v 'f[0-9]' | grep .) && echo XXX FAILED XXX

# Check read/write of basic odc cpio format
echo a: Basic copy test: cpio format
mkdir copy-cpio
(cd original && ${BSDTAR} -cf - --format cpio .) | (cd copy-cpio; ${BSDTAR} -xf -)
diff -r original copy-cpio || echo XXX FAILED XXX

# Ensure we can read gtar archives
echo a: Interoperability test: gtar
mkdir copy-gtar
(cd original && gtar -cf - .) | (cd copy-gtar; ${BSDTAR} -xf -)
diff -r original copy-gtar || echo XXX FAILED XXX

# Ensure we can read svr4crc cpio archives
echo a: Interoperability test: svr4crc cpio format
mkdir copy-svr4crc
(cd original && find . | cpio -o -H crc 2>/dev/null) | (cd copy-svr4crc; ${BSDTAR} -xf -)
diff -r original copy-svr4crc || echo XXX FAILED XXX

# Ensure we generate proper shar output
echo a: Interoperability test: shar output
mkdir copy-shar
(cd original && ${BSDTAR} -cf - --format=shar --exclude=sparse .) | (cd copy-shar; /bin/sh >/dev/null)
diff -r --exclude=sparse original copy-shar || echo XXX FAILED XXX

# Check that -u (update) picks up no new files
echo a: -u test 1
(cd original && ${BSDTAR} -cf ../test-u.tar -b 1 .)
cp test-u.tar test-u2.tar
(cd original && ${BSDTAR} -uf ../test-u2.tar .)
# Unfortunately, bsdtar -u truncates the end-of-file padding.
# Someday, this should be fixed.  For now, workaround the issue.
dd if=/dev/zero bs=1024 count=1 >>test-u2.tar 2>/dev/null
diff test-u.tar test-u2.tar || echo XXX FAILED XXX

# Check that -u (update) does pick up actual changed files
echo a: -u test 2
(cd original && echo hello >>f0)
(cd original && ${BSDTAR} -uf ../test-u2.tar .)
# Unfortunately, bsdtar -u truncates the end-of-file padding.
# Someday, this should be fixed.  For now, workaround the issue.
dd if=/dev/zero bs=1024 count=1 >>test-u2.tar 2>/dev/null
# All this really tests is that the archive did change.
cmp -s test-u.tar test-u2.tar && echo XXX FAILED XXX
# Now, unpack the archive and verify the contents (including the change to f0)
mkdir copy-u-test2
(cd copy-u-test2 && ${BSDTAR} -xf ../test-u2.tar)
diff -r original copy-u-test2 || echo XXX FAILED XXX

# We're done with the 'a' tests
cd ..

# Check that file flags are preserved properly across copies
echo b: Flag copy test
mkdir b
cd b
mkdir original
FLAGS='uchg opaque nodump uappnd'
for f in $FLAGS
do
    touch original/test.$f
    chflags $f original/test.$f
done
mkdir copy
(cd original && ${BSDTAR} -cf - .) | (cd copy; ${BSDTAR} -xpf -)
for f in $FLAGS
do
    [ "$f" = `ls -ol copy/test.$f | awk '{print $5;}'` ] || echo XXX FAIL: $f XXX
done
for f in $FLAGS
do
    if [ $f = 'nodump' ]
    then
	chflags dump original/test.$f
	chflags dump copy/test.$f
    else
	chflags no$f original/test.$f
	chflags no$f copy/test.$f
    fi
done
cd ..

# TODO: Verify that flags are not preserved if -p is omitted.

# Check that archiving obeys the 'nodump' flag
echo c: 'nodump' test
mkdir c
cd c
mkdir original
cd original
touch a
touch b
touch c
chflags nodump b
cd ..
mkdir copy
(cd original && ${BSDTAR} -cf - --nodump .) | (cd copy; ${BSDTAR} -xf -)
if [ -e copy/b ] ; then echo XXX Copied nodump file XXX; fi
diff -r --exclude=b original copy || echo XXX FAILURE XXX
cd ..

# Stress the deep directory logic; the actual depth here seems to
# be limited by the shell.  This should be restructured to get around
# that limit (possibly by using perl to build the deep tree?)
echo d: Deep copy test 1
mkdir d
cd d
mkdir original
cd original
I=0
while [ $I -lt 220 ]
do
    mkdir a$I
    cd a$I
    I=$(($I + 1))
done
while [ $I -gt 0 ] ; do cd ..; I=$(($I - 1)); done
cd ..
mkdir copy
(cd original; ${BSDTAR} -cf - .) | (cd copy; ${BSDTAR} -xf -)
diff -r original copy || echo XXX FAILURE XXX

# Make sure we correctly handle gtar long filenames.
echo d: Deep copy test '(gtar compatibility)'
mkdir copy-gtar
(cd original; gtar -cf - .) | (cd copy-gtar; ${BSDTAR} -xf -)
diff -r original copy-gtar || echo XXX FAILURE XXX
cd ..

