/******************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BsC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BsC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
---------------------------------------------------------------------*/
#include "Tree.h"
#include "StartDir.h"
#include "Shared.h"
#include "Busy.h"
#include "Config.h"
#include <qapplication.h>
#include <qcursor.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qheader.h>
#include <qdir.h>
#include <qlabel.h>
#include <qmessagebox.h>

/*------- local constants:
-------------------------------------------------------------------*/
const QString Tree::Caption     = QT_TR_NOOP( "Tree of directories" );
const QString Tree::Dirs        = QT_TR_NOOP( "Dirs: " );
const QString Tree::CantReadDir = QT_TR_NOOP( "You don't have permission rights to read this directory:\n%1" );
const QString Tree::DirNotValid = QT_TR_NOOP( "Directory specification not applicable:\n%1" );
const QString Tree::DirNotFound = QT_TR_NOOP( "The directory not found:\n%1" );


//*******************************************************************
// Tree                                                  CONSTRUCTOR
//*******************************************************************
Tree::Tree( QWidget* const in_parent, const QString& in_current_dir )
: QDialog( in_parent )
, d_from_dir( QString::null )
, d_current_dir( in_current_dir )
, d_view( new QListView( this ))
, d_select_btn( new QPushButton( tr(Shared::SelectBtnLabel), this ))
, d_return_btn( new QPushButton( tr(Shared::BreakBtnLabel), this ))
, d_break( FALSE )
, d_finished( FALSE )
, d_counter_lb( new QLabel( "0", this ))
, d_counter( 0 )
{
	setCaption( tr(Caption) );
	d_select_btn->setEnabled( FALSE );
		
	d_view->setRootIsDecorated( FALSE );
	d_view->addColumn( "" );
	d_view->header()->hide();
	d_view->setPaletteBackgroundColor( Config::instance()->lfs_default_bkg_color() );
	
	QVBoxLayout* const main_layout = new QVBoxLayout( this );
	main_layout->setMargin( Shared::LayoutMargin  );
	main_layout->setSpacing( Shared::LayoutSpacing );
	main_layout->addWidget( d_view );
		
	QHBoxLayout* const btn_layout = new QHBoxLayout;
	btn_layout->addWidget( new QLabel( tr(Dirs), this ));
	btn_layout->addWidget( d_counter_lb );
	btn_layout->addStretch( Shared::OverStretch );
	btn_layout->addWidget( d_select_btn );
	btn_layout->addWidget( d_return_btn );
	main_layout->addLayout( btn_layout );
	
	connect( d_select_btn, SIGNAL( clicked() ), this, SLOT( slot_select() ));
	connect( d_return_btn, SIGNAL( clicked() ), this, SLOT( slot_return() ));
}
// end of Tree

//*******************************************************************
// slot_select                                          PRIVATE slot
//*******************************************************************
void Tree::slot_select()
{
	const ViewItem* const item = dynamic_cast<ViewItem*>( d_view->selectedItem() );
	if( item ) {
		accept();
	}
}
// end of slot_select

//*******************************************************************
// slot_return                                          PRIVATE slot
//*******************************************************************
void Tree::slot_return()
{
	if( d_finished ) {
		reject();
	}
	else if( FALSE == d_break ) {
		d_break = TRUE;
		d_return_btn->setEnabled( FALSE );
	}
}
// end of slot_return

//*******************************************************************
// polish                                          PRIVATE inherited
//*******************************************************************
void Tree::polish()
{
	Shared::polish( this, 30, 50 );
}
// end of polish

//*******************************************************************
// keyPressEvent                                   PRIVATE inherited
//*******************************************************************
void Tree::keyPressEvent( QKeyEvent* e )
{
	if( Qt::Key_Escape == e->key() ) {
		d_break = TRUE;
	}
	QDialog::keyPressEvent( e );
}
// end of keyPressEvent

//*******************************************************************
// show                                             PRIVATE inerited
//*******************************************************************
void Tree::show()
{
	bool ok = FALSE;
	
	StartDir* start = new StartDir( this );
	if( start ) {
		if( QDialog::Accepted == start->exec() ) {
			start->get_from_dir( d_from_dir );
		}
		delete start;
		start = 0;
		
		if( d_from_dir ) {
			if( Shared::is_regular_file( d_from_dir ) ) {
				const QFileInfo fi( d_from_dir );
				if( fi.exists() && fi.isDir() ) {
					if( fi.isExecutable() && fi.isReadable() ) {
						ok = TRUE;
					}
					else {
						QMessageBox::information(	0, tr(Shared::Information),
														tr(CantReadDir).arg( d_from_dir ),
														QMessageBox::Ok );
					}
				}
				else {
					QMessageBox::information(	0, tr(Shared::Information),
														tr(DirNotFound).arg( d_from_dir ),
														QMessageBox::Ok );
				}
			}
			else {
					QMessageBox::information(	0, tr(Shared::Information),
														tr(DirNotValid).arg( d_from_dir ),
														QMessageBox::Ok );
			}
		}
	}
	
	if( FALSE == ok ) {
		reject();
		return;
	}
	
	QDialog::show();
	Shared::idle();

	QApplication::setOverrideCursor( Qt::WaitCursor );
	d_view->setHScrollBarMode( QListView::AlwaysOff );

	d_finfo.setFile( d_from_dir );
	ViewItem* root = new ViewItem( d_view, &d_finfo );
	root->setOpen( TRUE );
	update_counter();
	//........................................................
	parse( root, d_from_dir, 0 );

	QApplication::restoreOverrideCursor();
	Shared::idle();
	d_view->setUpdatesEnabled( TRUE );
	d_view->adjustColumn( 0 );
	d_view->setHScrollBarMode( QListView::Auto );

	if( d_break ) {
		reject();
	}
	else {
		d_finished = TRUE;
		d_select_btn->setEnabled( TRUE );
		d_return_btn->setText( tr(Shared::CloseBtnLabel) );
		select();
	}
}
// end of show

//*******************************************************************
// parse                                                     PRIVATE
//*******************************************************************
void Tree::parse( ViewItem* in_parent, const QString& in_path, const int in_level )
{
	d_view->setUpdatesEnabled( 0 == in_level );
	Shared::idle();

	if( FALSE == d_break ) {
		QFileInfo* finfo = 0;
		d_finfo.setFile( in_path);

		if( d_finfo.isDir() && !d_finfo.isSymLink() ) {
			if( d_finfo.isReadable() && d_finfo.isExecutable() ) {
				QDir dir( in_path, QString::null, (QDir::Name | QDir::IgnoreCase), QDir::Dirs  );
				const QFileInfoList* const file_list = dir.entryInfoList();
				if( file_list ) {
					QFileInfoListIterator it( *file_list );
					while(( FALSE == d_break ) && ((	finfo = it.current() ) != 0 )) {
						if( Shared::is_regular_file( finfo->fileName() )) {
							ViewItem* const item = new ViewItem( in_parent, finfo );
							update_counter();
							parse( item, finfo->absFilePath(), in_level + 1 );
							d_view->setUpdatesEnabled( 0 == in_level );
						}
						Shared::idle();
						++it;
					}
				}
			}
		}
	}
}
// end of parse

//*******************************************************************
// select                                                    PRIVATE
//*******************************************************************
void Tree::select()
{
	if( d_current_dir ) {
		QListViewItemIterator it( d_view );
		while ( it.current() ) {
			ViewItem* const item = dynamic_cast<ViewItem*>( it.current() );
			if( item ) {
				if( item->d_info.absFilePath() == d_current_dir ) {
					d_view->setCurrentItem( item );
					d_view->ensureItemVisible( item );
					break;
				}
			}
			++it;
		}
	}
}
// end of select

//*******************************************************************
// update_counter                                            PRIVATE
//*******************************************************************
inline void Tree::update_counter()
{
	d_counter_lb->setText( Shared::num2str( ++d_counter ));
}
// end of update_counter
