
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

//#define DEBUG

#include <math.h>

#include "bristol.h"
#include "bristolmm.h"

#include "dco.h"

extern tableEntry defaultTable[];
extern void inthandler();

/*
 * Used by the a440 sine generator.
 */
int s440holder = 0;

/*
 * Some of these are global parameters and should be moved into the baudio,
 * some are voice specific and should be moved into the voice structure. Some of
 * them will eventually need to be linked up to midi controller values for real
 * time expression.
static float extgain = 0;
 */

doAudioOps(audioMain *audiomain, float *outbuf, float *startbuf)
{
	register bristolVoice *voice = audiomain->playlist;
	register int i;
	register Baudio *thisaudio;
	register float *extmult, *leftch, *rightch;

	/*
	 * Clear the output buffer at this point.
	 */
	bristolbzero(outbuf, audiomain->segmentsize * 2);

#ifdef DEBUG
	if ((audiomain->debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG5)
		printf("doAudioOps\n");
#endif
//printf("doAudioOps\n");

	/*
	 * Configure flags to force one iteration of the preops, and to enable
	 * postops if a voice is active.
	 */
	thisaudio = audiomain->audiolist;
//printf("preops\n");
	while (thisaudio != NULL)
	{
		/*
		 * See if we are finnished with the baudio (ie, gui has exit).
		 */
		if (thisaudio->mixflags & BRISTOL_REMOVE)
		{
			Baudio *holder = thisaudio;
			bristolVoice *vl;

			thisaudio = thisaudio->next;

			freeBristolAudio(audiomain, holder);

			if (audiomain->audiolist == NULL)
			{
				/*
				 * Last synth GUI has exit, we may as well do the same.
				 */
				inthandler();
				return;
			}
			/*
			 * We now need to make sure none of the voices are using this
			 * remove audio.
			 */
			for (vl = voice; vl != NULL; vl = vl->next)
				if (vl->baudio = holder)
					vl->baudio = NULL;

			continue;
		}

		thisaudio->mixflags |= BRISTOL_MUST_PRE;
		thisaudio->mixflags &= ~BRISTOL_HAVE_OPED;
		thisaudio->lvoices = thisaudio->cvoices;
		thisaudio->cvoices = 0;

		thisaudio = thisaudio->next;
	}

	if (voice->baudio != NULL)
	{
#ifdef DEBUG
		if ((audiomain->debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG5)
			printf("No audio assigned to voice\n");
#endif
		/*
		 * a440 has to be done here, since it must be active even when no keys
		 * are pressed.
		 */
		if (voice->baudio->mixflags & A_440)
			a440(audiomain, outbuf, audiomain->samplecount);
	}

	/*
	 * Should drop startbuf and outbuf.
	bristolbzero(startbuf, audiomain->iosize);
	 */

	/*
	 * We need to look through our voice list, and see if any are active. If
	 * so then start running the voice structures through the sound structures.
	 *
	 * This is the polyphonic mixing process.
	 */
//printf("polyops\n");
	while (voice != NULL)
	{
		if (voice->baudio == NULL)
		{
			voice = voice->next;
			continue;
		}

		if (voice->baudio->mixflags & (BRISTOL_HOLDDOWN|BRISTOL_REMOVE))
			continue;

		if (!(voice->flags & BRISTOL_KEYDONE))
		{
			if (voice->baudio->mixflags & BRISTOL_MUST_PRE)
			{
				if (voice->baudio->preops)
					voice->baudio->preops(audiomain,
						voice->baudio, voice, outbuf, startbuf);

				/*
				 * Keep a pointer to the first voice that was active on any 
				 * given bristolAudio sound. This is the most recent keypress.
				 */
				voice->baudio->firstVoice = voice;
				voice->baudio->mixflags &= ~BRISTOL_MUST_PRE;
				voice->baudio->mixflags |= BRISTOL_HAVE_OPED;
			}

			/*
			 * Do something - find the buffers and put them in the IO structures
			 * then get the locals and params, and call the operator, summing
			 * the outputs into our single float buf.
			 *
			 * Each voice, or channel if you want, has its own independant 
			 * limit on voice counts.
			 */
			if (++voice->baudio->cvoices > voice->baudio->voicecount)
				voice->flags |= BRISTOL_KEYOFF;
			voice->baudio->operate(audiomain,
				voice->baudio, voice, startbuf);

			bristolbzero(startbuf, audiomain->iosize);

			/*
			 * Then clear the Key on flags to indicate that this is not a new
			 * note on event.
			voice->flags &= ~(BRISTOL_KEYON|BRISTOL_KEYREON);
			 */
		}
		voice = voice->next;
	}

	/*
	 * See if any of the voices have postoperators configured.
	 */
	thisaudio = audiomain->audiolist;
	while (thisaudio != NULL)
	{
		if (thisaudio->postops != NULL)
		{
			if (thisaudio->mixflags & BRISTOL_HAVE_OPED)
				thisaudio->postops(audiomain,
					thisaudio, thisaudio->firstVoice, startbuf);
		}
		thisaudio = thisaudio->next;
	}

	voice = audiomain->playlist;
	while (voice != NULL)
	{
		voice->flags &= ~(BRISTOL_KEYON|BRISTOL_KEYREON);
		voice = voice->next;
	}

	/*
	 * At this point all the voices have put their output onto the leftbuf and
	 * rightbuf (leftbuf only if mono). We should go through the baudio 
	 * structures and see if we have any effects active.
	 *
	 * It is not possible to use the HAVE_OPED flag for the effects processes.
	 * If they use reverb and echo they need to roll on way past the end of
	 * poly voice processing.
	 */
	thisaudio = audiomain->audiolist;
//printf("postops\n");
	while (thisaudio != NULL)
	{
		/*
		 * Need to ensure this audio structure is actually assigned
		 */
		if (thisaudio->mixflags & BRISTOL_HOLDDOWN)
		{
			thisaudio = thisaudio->next;
			continue;
		}

		/*
		 * nc-17/06/02:
		 * At the moment this only works for a single effect on the list. Need
		 * to add another for/while loop inside this if statement.
		 */
		if ((thisaudio->effect != NULL) &&
			(thisaudio->effect[0] != NULL))
		{
			int index;

			index = (*thisaudio->effect[0]).index;
			/*
			 * One inbuf, two outbuf.
			 */
			audiomain->palette[index]->specs->io[0].buf = thisaudio->leftbuf;
			audiomain->palette[index]->specs->io[1].buf = thisaudio->leftbuf;
			audiomain->palette[index]->specs->io[2].buf = thisaudio->rightbuf;

			if ((thisaudio->firstVoice == NULL)
				|| (thisaudio->firstVoice->baudio == 0))
			{
				thisaudio = thisaudio->next;
				continue;
			}
#warning - this voice may have been reassigned. check midi channel
			if (thisaudio->firstVoice != NULL)
				(*thisaudio->effect[0]).operate(
					audiomain->palette[index],
					thisaudio->firstVoice,
					(*thisaudio->effect[0]).param,
					thisaudio->FXlocals[0][0]);
			/*
			 * If we have FX, apply them. They will all be stereo, and will
			 * eventually also be chained.
			 */
			rightch = thisaudio->rightbuf;
		} else {
			if (thisaudio->mixflags & BRISTOL_STEREO)
				rightch = thisaudio->rightbuf;
			else
				/*
				 * Otherwise the synth is mono.
				 */
				rightch = thisaudio->leftbuf;
		}

		leftch = thisaudio->leftbuf;

		extmult = outbuf;
		/*
		 * Interleave the results for output to a stereo device.
		 */
		for (i = 0; i < audiomain->samplecount; i+=8)
		{
			*extmult++ += *leftch++; *extmult++ += *rightch++;
			*extmult++ += *leftch++; *extmult++ += *rightch++;
			*extmult++ += *leftch++; *extmult++ += *rightch++;
			*extmult++ += *leftch++; *extmult++ += *rightch++;
			*extmult++ += *leftch++; *extmult++ += *rightch++;
			*extmult++ += *leftch++; *extmult++ += *rightch++;
			*extmult++ += *leftch++; *extmult++ += *rightch++;
			*extmult++ += *leftch++; *extmult++ += *rightch++;
		}

		bristolbzero(thisaudio->leftbuf, audiomain->segmentsize);
		bristolbzero(thisaudio->rightbuf, audiomain->segmentsize);

		thisaudio = thisaudio->next;
	}
}

fillFreqTable(Baudio *baudio, register bristolVoice *voice, register float *buf,
	register int size, int glide)
{
	register int i;

	if ((baudio->debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG5)
		printf("fillFreqTable(%x, %x, %i)\n",
			voice, buf, size);

	/*
	 * Take a buffer, and fill in some frequency value to make an oscillator
	 * reproduce the correct note. Will cater for polyphonic portamento, and
	 * eventually also for some generic mods.
	 */
	if ((voice->cFreq == voice->dFreq) || (glide == 0))
	{
		/*
		 * Note has no glide, just fill the table.
		 */
		for (i = 0; i < size; i++)
			*buf++ = voice->dFreq;
	} else {
		if (voice->lastkey < voice->key.key)
		{
			/*
			 * Glide frequency up at some rate.
			 */
			for (i = 0; i < size; i++)
			{
				if (voice->cFreq > voice->dFreq)
				{
					voice->cFreq = voice->dFreq;
					for (; i < size; i++)
						*buf++ = voice->cFreq;
					break;
				}
				*buf++ = (voice->cFreq += voice->cFreqstep);
			}
		} else {
			/*
			 * Glide frequency down at some rate.
			 */
			for (i = 0; i < size; i++)
			{
				if (voice->cFreq < voice->dFreq)
				{
					voice->cFreq = voice->dFreq;
					for (; i < size; i++)
						*buf++ = voice->cFreq;
					break;
				}
				*buf++ = (voice->cFreq += voice->cFreqstep);
			}
		}
	}
}

/*
 * Take a buffer and fill it with a sinewave at 440 hz.
 */
a440(audioMain *audiomain, float *buf, int count)
{
	float *sine, index = s440holder, step;
	int obp;

	/*
	 * There is a sine wave in buffer zero of the first operator - the DCO.
	 */
	sine = ((bristolDCO *)audiomain->palette[0]->specs)->wave[0];
	step = defaultTable[69].defnote;

	for (obp = 0; obp < count * 2;)
	{
		buf[obp++] = sine[(int) index] * 12;
		buf[obp++] = sine[(int) index] * 12;
		if ((index += step) > DCO_WAVE_SZE)
			index -= DCO_WAVE_SZE;
	}

	s440holder = index;
}

