
/*
 *  Diverse Bristol midi routines.
 *  Copyright (c) by Nick Copeland <nickycopeland@hotmail.com> 1996,2008
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * This code should open the midi device and read data from it.
 *
 * All requests will be dispatched to the relevant handlers, with support for
 * OSS (/dev/midi*), ALSA (hw:0,0) and as of release 0.8 also the ALSA sequencer
 * (with client.port config).
 *
 * All events will be converted into a bristol native message format for
 * independency from the driver.
 */

/*#define DEBUG */

#include <unistd.h>

#include "bristolmessages.h"
#include "bristol.h"
#include "bristolmidi.h"

bristolMidiMain bmidi;

extern int bristolMidiFindDev();
extern int bristolMidiFindFreeHandle();

extern int bristolMidiTCPOpen();
extern int bristolMidiOSSOpen();
extern int bristolMidiALSAOpen();
extern int bristolMidiSeqOpen();
#ifdef _BRISTOL_JACK_MIDI
extern int bristolMidiJackOpen();
#endif

extern int bristolMidiTCPClose();
extern int bristolMidiOSSClose();
extern int bristolMidiALSAClose();
extern int bristolMidiSeqClose();
#ifdef _BRISTOL_JACK_MIDI
extern int bristolMidiJackClose();
#endif

extern int bristolMidiSanity();
extern int bristolMidiDevSanity();
extern int bristolMidiALSARead();
extern int bristolMidiSeqRead();
extern int bristolMidiWrite();
extern int bristolPhysWrite();

extern int initMidiLib();

/*
 * This is the generic code. Calls the desired (configured) drivers.
 */
int
bristolMidiOpen(char *dev, int flags, int chan, int msgs, int (*callback)(),
void *param)
{
	int handle, devnum;

#ifdef DEBUG
	printf("bristolMidiOpen(%s, %x)\n", dev, flags);
#endif

	initMidiLib(flags);

	/*
	 * The value of -1 indicates all channels, otherwise we are only interested
	 * in messages on the given channel.
	 */
	if ((chan < -1) || (chan >= 16))
	{
		/*
		 * If this is a TCP connection then we can specify a channel greater
		 * than 1024 - it is interpreted as a TCP port, all channels.
		 */
		if ((chan < 1024)
			|| ((flags & BRISTOL_CONNMASK & BRISTOL_CONN_TCP) == 0))
			return(BRISTOL_MIDI_CHANNEL);
	}

	/*
	 * Make sure we have a handle available
	 */
	if ((handle = bristolMidiFindFreeHandle()) < 0)
		return(handle);

	bmidi.handle[handle].handle = handle; /* That looks kind of wierd! */
	bmidi.handle[handle].state = BRISTOL_MIDI_OK;
	bmidi.handle[handle].channel = chan;
	bmidi.handle[handle].dev = -1;
	bmidi.handle[handle].flags = BRISTOL_MIDI_OK;
	bmidi.handle[handle].messagemask = msgs;

	/*
	 * We should then check to see if this dev is open, and if so link
	 * another handle to it. We should support multiple destinations of the
	 * same midi information.
	 */
	if (((flags & BRISTOL_CONN_FORCE) == 0)
		&& ((devnum = bristolMidiFindDev(dev)) >= 0))
	{
		/*
		 * We have the device open, configure the new handle, and return it.
		 */
		bmidi.dev[devnum].handleCount++;
		bmidi.handle[handle].dev = devnum;
		bmidi.handle[handle].callback = callback;
		bmidi.handle[handle].param = param;
		bmidi.handle[handle].flags = bmidi.dev[devnum].flags;

		printf("reusing connection %x\n", bmidi.dev[devnum].flags);

		return(handle);
	}

	/*
	 * Now try and find a device that is free.
	 */
	if ((devnum = bristolMidiFindDev(NULL)) < 0)
		return(devnum);

	/*
	 * We have early returns here and we should consider cleaning up. The thing
	 * is that if this call fails then typically the engine will exit. Future
	 * code may change that where we poke around for driver availability.
	 */
	switch (flags & BRISTOL_CONNMASK) {
		case BRISTOL_CONN_TCP:
			if (handle != bristolMidiTCPOpen(dev, flags,
				chan, msgs, callback, param, devnum, handle))
				return(BRISTOL_MIDI_DRIVER);
			bmidi.handle[handle].channel = -1;
			break;
		case BRISTOL_CONN_MIDI:
			if (handle != bristolMidiALSAOpen(dev, flags,
					chan, msgs, callback, param, devnum, handle))
				return(BRISTOL_MIDI_DRIVER);
			break;
		case BRISTOL_CONN_OSSMIDI:
			if (handle != bristolMidiOSSOpen(dev, flags,
				chan, msgs, callback, param, devnum, handle))
				return(BRISTOL_MIDI_DRIVER);
			break;
		case BRISTOL_CONN_SEQ:
			if (handle != bristolMidiSeqOpen(dev, flags,
				chan, msgs, callback, param, devnum, handle))
				return(BRISTOL_MIDI_DRIVER);
			break;
		case BRISTOL_CONN_JACK:
#ifdef _BRISTOL_JACK_MIDI
			if (handle != bristolMidiJackOpen(dev, flags,
				chan, msgs, callback, devnum, devnum, handle))
			{
				bmidi.dev[devnum].state = -1;
				bmidi.handle[handle].state = -1;
				return(BRISTOL_MIDI_DRIVER);
			}
#endif
			bmidi.dev[devnum].fd = -1;
			break;
		case BRISTOL_CONN_UNIX:
		default:
			printf("conn type %x not supported\n", flags & BRISTOL_CONNMASK);
			return(BRISTOL_MIDI_DRIVER);
			break;
	}

	sprintf(bmidi.dev[devnum].name, "%s", dev);
	bmidi.dev[devnum].state = BRISTOL_MIDI_OK;
	bmidi.dev[devnum].handleCount = 1;
	bmidi.dev[devnum].bufindex = 0;
	bmidi.dev[devnum].bufcount = 0;
	bmidi.handle[handle].dev = devnum;
	bmidi.handle[handle].callback = callback;
	bmidi.handle[handle].param = param;

/*printf("returning handle %i, flags %x\n", handle, bmidi.dev[devnum].flags); */
	return(handle);
}

int
bristolMidiClose(int handle)
{
#ifdef DEBUG
	printf("bristolMidiClose()\n");
#endif

	/*
	 * We close a handle, not a device. We need to clean out the handle, and if
	 * no other handles have this device open then close the physical device as
	 * well.
	 */
#warning - close handle, and then device if no longer in use.
	if (bristolMidiSanity(handle) < 0)
		return(bristolMidiSanity(handle));

	switch (bmidi.dev[bmidi.handle[handle].dev].flags & BRISTOL_CONNMASK) {
		case BRISTOL_CONN_OSSMIDI:
			return(bristolMidiOSSClose(handle));
		case BRISTOL_CONN_MIDI:
			return(bristolMidiALSAClose(handle));
		case BRISTOL_CONN_SEQ:
			return(bristolMidiSeqClose(handle));
		case BRISTOL_CONN_JACK:
#ifdef _BRISTOL_JACK_MIDI
			return(bristolMidiJackClose(handle));
#endif
			break;
	}

	return(BRISTOL_MIDI_DRIVER);
}

int
bristolMidiDevRead(int dev, bristolMidiMsg *msg)
{
#ifdef DEBUG
	printf("bristolMidiDevRead(%i)\n", dev);
#endif

	if (bristolMidiDevSanity(dev) < 0)
		return(bristolMidiDevSanity(dev));

	/*
	 * ALSARead handles all the TCP interface as well. It should be reworked to
	 * manage OSS raw interfaces as well.....
	 */
	switch (bmidi.dev[dev].flags & BRISTOL_CONNMASK) {
		case BRISTOL_CONN_MIDI:
		case BRISTOL_CONN_TCP:
		case BRISTOL_CONN_OSSMIDI:
			return(bristolMidiALSARead(dev, msg));
		case BRISTOL_CONN_SEQ:
			return(bristolMidiSeqRead(dev, msg));
	}

	return(BRISTOL_MIDI_DRIVER);
}

int
bristolGetMidiFD(int handle)
{
	return(bmidi.dev[bmidi.handle[handle].dev].fd);
}

int
bristolMidiRead(int handle, bristolMidiMsg *msg)
{
#ifdef DEBUG
	printf("bristolMidiRead(%i): %i/%i\n", handle,
		bmidi.handle[handle].dev, bmidi.dev[bmidi.handle[handle].dev].fd);
#endif

	if (bristolMidiSanity(handle) < 0)
		return(bristolMidiSanity(handle));

	msg->command = 0xff;

	if (bmidi.handle[handle].callback == NULL)
	{
		while (msg->command == 0xff)
		{
/*printf("reading type %x\n", bmidi.dev[bmidi.handle[handle].dev].flags); */
			switch (bmidi.dev[bmidi.handle[handle].dev].flags
				& BRISTOL_CONNMASK)
			{
				case BRISTOL_CONN_MIDI:
				case BRISTOL_CONN_TCP:
				case BRISTOL_CONN_OSSMIDI:
					if (bristolMidiALSARead(bmidi.handle[handle].dev, msg) < 0)
						return(-1);
					break;
				case BRISTOL_CONN_SEQ:
					if (bristolMidiSeqRead(bmidi.handle[handle].dev, msg) < 0)
						return(-1);
					break;
			}
		}

		return(BRISTOL_MIDI_OK);
	} else {
		switch (bmidi.dev[handle].flags & BRISTOL_CONNMASK) {
			case BRISTOL_CONN_MIDI:
			case BRISTOL_CONN_TCP:
			case BRISTOL_CONN_OSSMIDI:
				return(bristolMidiALSARead(bmidi.handle[handle].dev, msg));
			case BRISTOL_CONN_SEQ:
				return(bristolMidiSeqRead(bmidi.handle[handle].dev, msg));
		}
	}

	return(BRISTOL_MIDI_DRIVER);
}

int
bristolMidiTerminate()
{
	printf("Terminate MIDI signalling\n");
	/*
	 * We need this since the library never actually returns when we are
	 * using the midiCheck routines. We are going to close all handles.
	 */
	bmidi.flags = BRISTOL_MIDI_TERMINATE;

	if (bmidi.dev[0].flags & BRISTOL_CONTROL_SOCKET)
	{
		/*
		 * We should send an all_notes_off and terminate the socket layer with
		 * a REQSTOP
		 */
		close(bmidi.dev[0].fd);
		unlink(BRISTOL_SOCKNAME);
	}

	/*
	 * Free up the semaphores
	 */

	return(0);
}

int
bristolMidiWrite(int dev, bristolMsg *msg, int size)
{
	unsigned char byte;

	if (bristolMidiDevSanity(dev) < 0)
		return(bristolMidiDevSanity(dev));

#ifdef DEBUG
	printf("bristolMidiWrite %i/%i, %i\n", dev, bmidi.dev[dev].fd, size);
#endif

	byte = MIDI_SYSEX;
	if (bristolPhysWrite(bmidi.dev[dev].fd, &byte, 1) != 0)
		return(1);

	if (bristolPhysWrite(bmidi.dev[dev].fd, (unsigned char *) msg, size) != 0)
		return(1);

	byte = MIDI_EOS;
	if (bristolPhysWrite(bmidi.dev[dev].fd, &byte, 1) != 0)
		return(1);

	return(0);
}

int
bristolMidiControl(int handle, int channel, int operator, int controller,
	int value)
{
	unsigned char comm = 0xb0, ctr, val;

/*	printf("Midi Control %i %i %i %i\n", channel, operator, controller, value); */

	comm |= channel;
	ctr = controller;
	val = value & 0x7f;

	bristolPhysWrite(bmidi.dev[bmidi.handle[handle].dev].fd, &comm, 1);
	bristolPhysWrite(bmidi.dev[bmidi.handle[handle].dev].fd, &ctr, 1);
	bristolPhysWrite(bmidi.dev[bmidi.handle[handle].dev].fd, &val, 1);

	return(0);
}

int
bristolPitchEvent(int handle, int op, int channel, int key)
{
	unsigned char comm = 0xe0, lsb, msb;

/*	printf("pitch event %i %i %i\n", op, channel, key); */

	comm |= channel;
	lsb = key & 0x7f;
	msb = (key >> 7) & 0x7f;

/*	printf("Sending %i %i %i on %i\n", comm, msb, lsb, */
/*		bmidi.dev[bmidi.handle[handle].dev].fd); */

	bristolPhysWrite(bmidi.dev[bmidi.handle[handle].dev].fd, &comm, 1);
	bristolPhysWrite(bmidi.dev[bmidi.handle[handle].dev].fd, &lsb, 1);
	bristolPhysWrite(bmidi.dev[bmidi.handle[handle].dev].fd, &msb, 1);

	return(0);
}

#ifdef NOT_THIS_DEBUG
static int toggle = 0;
#endif

int
bristolKeyEvent(int handle, int op, int channel, int key, int velocity)
{
	unsigned char comm;

	key &= 0x7f;
	velocity &= 0x7f;

#ifdef DEBUG
	printf("key event ch: %i, key: %i over fd %i\n", channel, key,
		bmidi.dev[bmidi.handle[handle].dev].fd);
#endif

	if (bristolMidiSanity(handle) < 0)
		return(bristolMidiSanity(handle));

	if (op == BRISTOL_EVENT_KEYON) {
		comm = MIDI_NOTE_ON | channel;
	} else
		comm = MIDI_NOTE_OFF | channel;

	bristolPhysWrite(bmidi.dev[bmidi.handle[handle].dev].fd, &comm, 1);
	bristolPhysWrite(bmidi.dev[bmidi.handle[handle].dev].fd, &key, 1);
	bristolPhysWrite(bmidi.dev[bmidi.handle[handle].dev].fd, &velocity, 1);

	return(0);
}

/*
 * This is the second implementation, the previous uses superfluous params....
 */
int
bristolMidiSendControlMsg(int handle, int channel, int ctrl, int value)
{
	unsigned char comm = 0xb0;

	comm |= channel;
	ctrl &= 0x7f;
	value &= 0x7f;

	/* printf("control event %i %i %i\n", channel, ctrl, value); */

	bristolPhysWrite(bmidi.dev[bmidi.handle[handle].dev].fd, &comm, 1);
	bristolPhysWrite(bmidi.dev[bmidi.handle[handle].dev].fd, &ctrl, 1);
	bristolPhysWrite(bmidi.dev[bmidi.handle[handle].dev].fd, &value, 1);

	return(0);
}

int
bristolMidiSendNRP(int handle, int channel, int nrp, int value)
{
	/*
	 * Send the NRP index.
	 */
	bristolMidiSendControlMsg(handle, channel, 99, (nrp >> 7) & 0x7f);
	bristolMidiSendControlMsg(handle, channel, 98, nrp & 0x7f);

	/*
	 * Send the controller values.
	 */
	bristolMidiSendControlMsg(handle, channel, 6, (value >> 7) & 0x7f);
	bristolMidiSendControlMsg(handle, channel, 38, value & 0x7f);

	return(0);
}

int
bristolMidiSendRP(int handle, int channel, int nrp, int value)
{
	/*
	 * Send the NRP index.
	 */
	bristolMidiSendControlMsg(handle, channel, 101, (nrp >> 7) & 0x7f);
	bristolMidiSendControlMsg(handle, channel, 100, nrp & 0x7f);

	/*
	 * Send the controller values.
	 */
	bristolMidiSendControlMsg(handle, channel, 6, (value >> 7) & 0x7f);
	bristolMidiSendControlMsg(handle, channel, 38, value & 0x7f);

	return(0);
}

int
bristolMidiSendKeyMsg(int handle, int channel, int operator, int key, int volume)
{
	if (operator != BRISTOL_EVENT_KEYON)
		operator = BRISTOL_EVENT_KEYOFF;

	bristolKeyEvent(bmidi.handle[handle].dev, operator, channel, key, volume);

	return(0);
}

int
bristolMidiSendMsg(int handle, int channel, int operator, int controller,
	int value)
{
	bristolMsg msg;

#ifdef DEBUG
	printf("bristolMidiSendMsg(%i, %i, %i, %i, %i)\n",
		handle, channel, operator, controller, value);
#endif

	if ((value > C_RANGE_MIN_1) || (value < 0))
		printf("controller %i/%i value %i outside range\n",
			operator, controller, value);

	value &= C_RANGE_MIN_1;

	/*
	 * Key on and off events are NOT sent as system messages, but as normal
	 * key events on the control channel. As such they require separate formats.
	 */
	if (operator == BRISTOL_EVENT_PITCH)
	{
		bristolPitchEvent(bmidi.handle[handle].dev, operator, channel, value);
		return(0);
	}
	if ((operator == BRISTOL_EVENT_KEYON) || (operator == BRISTOL_EVENT_KEYOFF))
	{
		bristolKeyEvent(bmidi.handle[handle].dev, operator,
			channel, value & BRISTOL_PARAMMASK, 120);
		return(0);
	}
	/*
	 * We are going to send this through to the control port as a basic format
	 * message. The parameter is not going to be scaled, just given as a value
	 * from 0 to 127. The receiver can decide how to scale the value.
	 */
	msg.SysID = 83;
	msg.L = 76;
	msg.a = 97;
	msg.b = 98;

	msg.msgLen = sizeof(bristolMsg);
	msg.msgType = MSG_TYPE_PARAM;

	msg.channel = channel;
	msg.from = handle;
	msg.operator = operator;
	msg.controller = controller;
	msg.valueLSB = value & 0x007f;
	msg.valueMSB = (value >> 7) & 0x007f;

	bristolMidiWrite(bmidi.handle[handle].dev, &msg, sizeof(bristolMsg));

	return(0);
}

