
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nickycopeland@hotmail.com> 1996,2008
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <signal.h>
#include <stddef.h>
#include <stdio.h>
#include <unistd.h>
#include "pthread.h"

#include "bristol.h"
#include "bristolmidi.h"
#include "bristolblo.h"

extern int dupfd;
void inthandler();

int exitReq = 0;
static int priority = 75, jh = -1;

extern void *audioThread();
extern void *midiThread();
extern void doNoteChanges();
extern int bristolMidiTerminate();

pthread_t spawnThread();

audioMain audiomain;

char *outputfile = NULL;

/*
 * We are going to create, initially, two threads. One will field MIDI events,
 * and translate the (for now) raw data into bristol MIDI messages. The messages
 * are queued, and then fielded by the audio thread. The audio thread reacts to
 * the midi events and starts sound generation operations. The MIDI thread will
 * remain responsible for MIDI parameter management, but not real-time messages.
 *
 * It could make sense to move the threads requests into the midi library for
 * use with other programmes?
 */
int
main(int argc, char **argv)
{
	int argCount = 1, harmonics = 31;
	pthread_t audiothread, midithread;

	bzero(&audiomain, sizeof(audioMain));
	audiomain.samplecount = 256; /* default this - gets overriden later */
	audiomain.iosize = 0; /* This needs to be calculated, samplecount * float */
	audiomain.preload = 8; /* This should be less, preferably 4. */
	audiomain.ingain = 0;
	audiomain.outgain = 4;
	/* default to OSS until ALSA goes 1.0 */
/*	audiomain.flags = BRISTOL_OSS|BRISTOL_MIDI_OSS; */
	audiomain.flags = BRISTOL_ALSA|BRISTOL_MIDI_SEQ;
	audiomain.samplerate = 44100; /* This is a default, gets overwritten */

	if (argc <= 2) {
		printf("You probably prefer to use the startBristol script.\n");
		exit(1);
	}

	/*
	 * This could be made more intelligent such as a name for what to connect
	 * to for example. That is for later study.
	 */
	if (getenv("BRISTOL_AUTOCONN") != NULL)
		audiomain.flags |= BRISTOL_AUTO_CONN;

	while (argc > argCount)
	{
		if (strcmp(argv[argCount], "-glwf") == 0)
		{
			bloflags |= BRISTOL_LWF;
			argCount++;
			continue;
		}

		if (strcmp(argv[argCount], "-blo") == 0)
			harmonics = atoi(argv[++argCount]);

		if ((strcmp(argv[argCount], "-o") == 0)
			&& (strlen(argv[argCount]) == 2))
				outputfile = argv[++argCount];

		if ((strcmp(argv[argCount], "-T") == 0)
			&& (strlen(argv[argCount]) == 2))
			audiomain.flags |= BRISTOL_MIDI_WAIT;

		if (strcmp(argv[argCount], "-mididev") == 0)
		{
			if (++argCount <= argc)
				audiomain.mididev = argv[argCount];
		} else if (strcmp(argv[argCount], "-midi") == 0) {
			if (++argCount <= argc)
			{
				/*
				 * Note that some of these types of interface to not actually
				 * apply to MIDI at the moment. OSS and ALSA are rawmidi specs.
				 * SEQ is ALSA sequencer support. The others may or may not
				 * provide midi messaging, however it is possible to have jack
				 * audio drivers with ALSA rawmidi interfacing, for example.
				 */
				if (strcmp(argv[argCount], "oss") == 0)
					audiomain.flags = (audiomain.flags & ~BRISTOL_MIDIMASK)
						|BRISTOL_MIDI_OSS;
				else if (strcmp(argv[argCount], "alsa") == 0)
					audiomain.flags = (audiomain.flags & ~BRISTOL_MIDIMASK)
						|BRISTOL_MIDI_ALSA;
				else if (strcmp(argv[argCount], "slab") == 0)
					audiomain.flags = (audiomain.flags & ~BRISTOL_MIDIMASK)
						|BRISTOL_MIDI_SLAB;
				else if (strcmp(argv[argCount], "seq") == 0)
					audiomain.flags = (audiomain.flags & ~BRISTOL_MIDIMASK)
						|BRISTOL_MIDI_SEQ;
				else if (strcmp(argv[argCount], "jack") == 0)
					audiomain.flags = (audiomain.flags & ~BRISTOL_MIDIMASK)
						|BRISTOL_MIDI_JACK;
				else if (strcmp(argv[argCount], "ladsp") == 0)
					audiomain.flags = (audiomain.flags & ~BRISTOL_MIDIMASK)
						|BRISTOL_MIDI_LADSP;
			}
		}

		if ((strcmp(argv[argCount], "-scala") == 0)
			|| (strcmp(argv[argCount], "-scl") == 0))
		{
			if (++argCount <= argc)
				audiomain.microTonalMappingFile =  argv[argCount];
		}

		if (strcmp(argv[argCount], "-audiodev") == 0)
		{
			if (++argCount <= argc)
				audiomain.audiodev = argv[argCount];
		} else if (strcmp(argv[argCount], "-audio") == 0) {
			if (++argCount <= argc)
			{
				if (strcmp(argv[argCount], "oss") == 0)
					audiomain.flags = (audiomain.flags & ~BRISTOL_AUDIOMASK)
						|BRISTOL_OSS;
				else if (strcmp(argv[argCount], "alsa") == 0)
					audiomain.flags = (audiomain.flags & ~BRISTOL_AUDIOMASK)
						|BRISTOL_ALSA;
				else if (strcmp(argv[argCount], "slab") == 0)
					audiomain.flags = (audiomain.flags & ~BRISTOL_AUDIOMASK)
						|BRISTOL_SLAB;
				else if (strcmp(argv[argCount], "jack") == 0) {
					audiomain.flags = (audiomain.flags & ~BRISTOL_AUDIOMASK)
						|BRISTOL_JACK;
					/* Default the devicename, can be overridden later */
					audiomain.audiodev = argv[0];
				} else if (strcmp(argv[argCount], "ladsp") == 0)
					audiomain.flags = (audiomain.flags & ~BRISTOL_AUDIOMASK)
						|BRISTOL_LADSP;
				else if (strcmp(argv[argCount], "dummy") == 0)
					audiomain.flags = (audiomain.flags &
						~(BRISTOL_AUDIOMASK|BRISTOL_MIDIMASK|BRISTOL_MIDI_ALSA))
						|BRISTOL_DUMMY|BRISTOL_MIDI_SEQ;
			}
		}

		/*
		 * Overall output gain for weak signals.
		 *
		 * the 'gain' option is now per synth, negotiated over the datalink.
		if (strcmp(argv[argCount], "-gain") == 0)
		{
			if (++argCount <= argc)
				audiomain.outgain = atoi(argv[argCount]);
		}
		 */
		if (strcmp(argv[argCount], "-outgain") == 0)
		{
			if (++argCount <= argc)
				audiomain.outgain = atoi(argv[argCount]);
		}

		/*
		 * Overall input gain for weak signals.
		 */
		if (strcmp(argv[argCount], "-ingain") == 0)
		{
			if (++argCount <= argc)
				audiomain.ingain = atoi(argv[argCount]);
		}

		/*
		 * This is the TCP control port connection only.
		 */
		if (strcmp(argv[argCount], "-port") == 0)
		{
			if (++argCount <= argc)
				audiomain.port = atoi(argv[argCount]);
		}

		if (strcmp(argv[argCount], "-oss") == 0)
			audiomain.flags = (audiomain.flags &
				~(BRISTOL_AUDIOMASK|BRISTOL_MIDIMASK))
				|BRISTOL_OSS|BRISTOL_MIDI_OSS;

		if (strcmp(argv[argCount], "-seq") == 0)
			audiomain.flags |= BRISTOL_MIDI_SEQ;

		if (strcmp(argv[argCount], "-alsa") == 0)
			audiomain.flags = (audiomain.flags &
				~(BRISTOL_AUDIOMASK|BRISTOL_MIDIMASK))
				|BRISTOL_ALSA|BRISTOL_MIDI_ALSA;

		if (strcmp(argv[argCount], "-jack") == 0)
		{
			audiomain.flags = (audiomain.flags &
				~(BRISTOL_AUDIOMASK|BRISTOL_MIDIMASK|BRISTOL_MIDI_ALSA))
				|BRISTOL_JACK|BRISTOL_MIDI_SEQ;
			/* Default the devicename, can be overridden later */
			audiomain.audiodev = argv[0];
		}

		if (strcmp(argv[argCount], "-autoconn") == 0)
			audiomain.flags |= BRISTOL_AUTO_CONN;

		if (strcmp(argv[argCount], "-dummyaudio") == 0)
			audiomain.flags = (audiomain.flags &
				~(BRISTOL_AUDIOMASK|BRISTOL_MIDIMASK|BRISTOL_MIDI_ALSA))
				|BRISTOL_DUMMY|BRISTOL_MIDI_SEQ;

		if ((strcmp(argv[argCount], "-rate") == 0) && (argc > argCount))
			audiomain.samplerate = atoi(argv[argCount++ + 1]);

		if ((strcmp(argv[argCount], "-count") == 0) && (argc > argCount))
			audiomain.samplecount = atoi(argv[argCount++ + 1]);

		if ((strcmp(argv[argCount], "-preload") == 0) && (argc > argCount))
			audiomain.preload = atoi(argv[argCount++ + 1]);

		if ((strcmp(argv[argCount], "-priority") == 0) && (argc > argCount))
		{
			if ((priority = atoi(argv[argCount++ + 1])) < 0)
				priority = 0;
		}

		argCount++;
	}

	generateBLOwaveforms(harmonics, BRISTOL_VPO);

	audiomain.atStatus = audiomain.mtStatus = BRISTOL_EXIT;
	audiomain.atReq = audiomain.mtReq = BRISTOL_OK;

	/*
	 * Create two threads, one for midi, then one for audio.
	 *
	 * We cannot create the audio thread yet, since we do not know how many 
	 * voices to create - this is a function of the midi start requests from
	 * some controller.
	 */
	printf("spawning midi thread\n");
	midithread = spawnThread(midiThread, SCHED_RR);

	signal(SIGINT, inthandler);
/*	signal(SIGSEGV, inthandler); */

	printf("parent going into idle loop\n");

	/*
	 * To support jack we are going to use this parent thread. We will have it
	 * reschedule itself to something less than that requested for the audio
	 * thread and have it handle MIDI events from the Jack interface at some
	 * higher priority than than the midi thread. The actual MIDI thread will
	 * still accept events from the GUI but they are generally long events 
	 * for audio reconfiguration.
	 * 
	 * We should put some lockout on this code - make sure that the MIDI
	 * thread has initialised since we may need some of that init here.
	 */
	while (audiomain.mtStatus != BRISTOL_OK)
	{
		if (audiomain.atReq == BRISTOL_REQSTOP)
			/*
			 * Midi thread init failed, exit to prevent a lockout. Might
			 * look odd that it is in the atReq and not in some mt status
			 * however the midi thread is rquesting the audiothread to exit.
			 * Since we have not started it yet then we can kind of exit
			 * here.
			 */
			exit(0);

		printf("Init waiting for midi thread OK status\n");
		usleep(100000);
	}
	printf("Got midi thread OK status\n");

	while (!exitReq)
	{
		/*
		 * We should wait for children here, and restart them. We should also 
		 * monitor any bristol system sysex messages, since they will be used
		 * to request we start the audio thread.
		 */
		if (audiomain.atReq & BRISTOL_REQSTART)
		{
			/*
			 * See if the audio thread is active, if not, start it. May 
			 * eventually support multiple audio threads.
			 */
			if (audiomain.atStatus == BRISTOL_EXIT)
			{
				int i = 100;

				printf("spawning audio thread\n");
				audiothread = spawnThread(audioThread, SCHED_FIFO);

				while (audiomain.atStatus != BRISTOL_OK)
				{
					if (audiomain.atStatus == BRISTOL_FAIL)
						break;

					printf("Jack init waiting for audio thread OK status\n");

					usleep(100000);

					if (--i < 0)
					{
						printf("failed to recieve audio thread OK, exiting\n");
						audiomain.mtStatus = BRISTOL_EXIT;
						return(-1);
					}
				}

				/*
				 * Request jack linkup to its midi sequencer. If we did not
				 * compile with jack for whatever reason then the library will
				 * report it.
				 *
				 * This call will not reschedule the thread, we need to do that,
				 * also this is not a thread but the parent process.
				 */
				if (audiomain.flags & BRISTOL_MIDI_JACK)
				{
					if ((jh = bristolMidiOpen("bristolmt",
						BRISTOL_CONN_JACK|BRISTOL_DUPLEX, -1,
						BRISTOL_REQ_NSX, midiMsgHandler, audiomain)) < 0)
						printf("requested jack midi did not link up\n");
					else
						printf("requested jack midi link up: %i\n", jh);
					if (priority - 1 > 0)
					{
						struct sched_param schedparam;

						if (sched_getparam(0, &schedparam) != 0)
							printf("Scheduler getparam failed...\n");

						schedparam.__sched_priority = priority - 1;
						if (sched_setscheduler(0, SCHED_FIFO, &schedparam) == 0)
							printf("set jmidi thread priority\n");
						else
							printf("could not set jmidi thread priority\n");
					} else
							printf("no request to set jmidi thread priority\n");
				}
			}
		}

		sleep(1);

		if (audiomain.atStatus == BRISTOL_FAIL)
		{
			printf("Audio Thread status failed: exiting. Is device in use?\n");
			break;
		}
	}

	/* Unregister the jack interface */
	if ((jh >= 0) && (audiomain.flags & BRISTOL_MIDI_JACK))
		bristolMidiClose(jh);

	sleep(1);

	printf("parent exiting\n");

	return(0);
}

pthread_t spawnThread(void * (*threadcode)(void *), int despolicy)
{
	int retcode;
	pthread_t thread;
	struct sched_param schedparam;
	int policy;

	/*
	 * Create thread, detach it.
	 */
	retcode = pthread_create(&thread, NULL, threadcode, &audiomain);
	if (retcode != 0)
		fprintf(stderr, "create a failed %d\n", retcode);

	retcode = pthread_detach(thread);
	if (retcode != 0)
		fprintf(stderr, "detach failed %d\n", retcode);

	/*
	 * If we are not after realtime fifo scheduling then leave out the thread
	 * scheduling request.
	 */
	if (despolicy != SCHED_FIFO)
		return(thread);

	/*
	 * See if we need to reschedule it. This may fail if we do not have root
	 * permissions (or if the audio library configures RRpri99 before we make
	 * this request?)
	 */
	if (pthread_getschedparam(thread, &policy, &schedparam) == 0)
	{
		if ((priority != 0) && ((policy != despolicy)
			|| (schedparam.__sched_priority != priority)))
		{
			policy = despolicy;
			schedparam.__sched_priority = priority;

			if (pthread_setschedparam(thread, policy, &schedparam) == 0)
				printf("rescheduled thread: %i\n", schedparam.__sched_priority);
			else
				printf("could not reschedule thread\n");
		} else
			printf("no requested to reschedule thread\n");

		/*
		 * Barfs on core2, early implementation?
		if (schedparam.__sched_priority != priority) {
			if (pthread_setschedprio(thread, priority) != 0)
				printf("failed to set priority to priority\n");
			else
				printf("set priority to priority\n");
		}
		 */
	} else
		printf("could not get thread schedule\n");

	return(thread);
}

void
midithreadexit()
{
	printf("Null palette\n");
	audiomain.atReq = BRISTOL_REQSTOP;
	audiomain.mtStatus = BRISTOL_EXIT;

	bristolMidiTerminate();

	pthread_exit(0);
}

void
inthandler()
{
	int i = 5;

	audiomain.atReq = BRISTOL_REQSTOP;
	exitReq = 1;

	for (; i > 0; i--)
	{
		if (audiomain.atStatus != BRISTOL_EXIT)
			sleep(1);
	}

	/*
	 * Cleanup the semaphores
	 */
	sem_destroy(&audiomain.sem_long);
	sem_destroy(&audiomain.sem_short);

	bristolMidiTerminate();
}

void
alterAllNotes(Baudio *baudio)
{
	bristolVoice *voice = audiomain.playlist;

	while (voice != NULL)
	{
		if (voice->baudio == baudio)
			doNoteChanges(voice);

		voice = voice->next;
	}
}

