package org.bouncycastle.pqc.jcajce.provider.test;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.Signature;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.bc.BCObjectIdentifiers;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.pqc.asn1.PQCObjectIdentifiers;
import org.bouncycastle.pqc.jcajce.spec.QTESLAParameterSpec;
import org.bouncycastle.util.Arrays;
import org.bouncycastle.util.encoders.Hex;
import org.bouncycastle.util.test.FixedSecureRandom;

public class QTESLATest
    extends KeyPairGeneratorTest
{
    static SecureRandom secureRandom = new SecureRandom();

    private void doTestSig(KeyPair kp)
        throws Exception
    {
        byte[] seed = Hex.decode("061550234D158C5EC95595FE04EF7A25767F2E24CC2BC479D09D86DC9ABCFDE7056A8C266F9EF97ED08541DBD2E1FFA1");
        byte[] msg = Hex.decode("D81C4D8D734FCBFBEADE3D3F8A039FAA2A2C9957E835AD55B22E75BF57BB556AC8");

        Signature signer = Signature.getInstance(kp.getPublic().getAlgorithm(), "BCPQC");

        signer.initSign(kp.getPrivate(), new FixedSecureRandom(seed));

        signer.update(msg);

        byte[] sig = signer.sign();

        signer = Signature.getInstance("qTESLA", "BCPQC");

        signer.initVerify(kp.getPublic());

        signer.update(msg);

        assertTrue(signer.verify(sig));
    }

    private void doTestKey(KeyPair kp)
        throws Exception
    {
        KeyFactory keyFact = KeyFactory.getInstance("qTESLA", "BCPQC");

        PublicKey pubKey = keyFact.generatePublic(new X509EncodedKeySpec(kp.getPublic().getEncoded()));
        PrivateKey privKey = keyFact.generatePrivate(new PKCS8EncodedKeySpec(kp.getPrivate().getEncoded()));

        assertEquals(kp.getPublic(), pubKey);
        assertEquals(kp.getPrivate(), privKey);
        assertEquals(kp.getPublic().hashCode(), pubKey.hashCode());
        assertEquals(kp.getPrivate().hashCode(), privKey.hashCode());
        assertEquals(kp.getPublic(), serialiseDeserialise(kp.getPublic()));
        assertEquals(kp.getPrivate(), serialiseDeserialise(kp.getPrivate()));
    }

    private Object serialiseDeserialise(Object o)
        throws Exception
    {
        ByteArrayOutputStream bOut = new ByteArrayOutputStream();
        ObjectOutputStream oOut = new ObjectOutputStream(bOut);

        oOut.writeObject(o);

        oOut.close();

        ObjectInputStream oIn = new ObjectInputStream(new ByteArrayInputStream(bOut.toByteArray()));

        return oIn.readObject();
    }


    public void testGenerateKeyPairSigningVerifyingPI()
        throws Exception
    {
        KeyPairGenerator kpGen = KeyPairGenerator.getInstance("qTESLA", "BCPQC");

        kpGen.initialize(new QTESLAParameterSpec(QTESLAParameterSpec.PROVABLY_SECURE_I), secureRandom);

        KeyPair kp = kpGen.generateKeyPair();

        kf = KeyFactory.getInstance("qTESLA");

        performKeyPairEncodingTest(kp);

        assertEquals(BCObjectIdentifiers.qTESLA_p_I, SubjectPublicKeyInfo.getInstance(kp.getPublic().getEncoded()).getAlgorithm().getAlgorithm());

        doTestSig(kp);
        doTestKey(kp);
    }

    public void testGenerateKeyPairSigningVerifyingPIII()
        throws Exception
    {

        KeyPairGenerator kpGen = KeyPairGenerator.getInstance("qTESLA", "BCPQC");

        kpGen.initialize(new QTESLAParameterSpec(QTESLAParameterSpec.PROVABLY_SECURE_III), secureRandom);

        KeyPair kp = kpGen.generateKeyPair();

        kf = KeyFactory.getInstance("qTESLA");

        performKeyPairEncodingTest(kp);

        assertEquals(BCObjectIdentifiers.qTESLA_p_III, SubjectPublicKeyInfo.getInstance(kp.getPublic().getEncoded()).getAlgorithm().getAlgorithm());

        doTestSig(kp);
        doTestKey(kp);
    }

    private void doTestKAT(ASN1ObjectIdentifier alg, byte[] pubKey, byte[] privKey, byte[] seed, byte[] msg, byte[] expected)
        throws Exception
    {
        KeyFactory keyFact = KeyFactory.getInstance("qTESLA", "BCPQC");

        PublicKey qPub = keyFact.generatePublic(new X509EncodedKeySpec(new SubjectPublicKeyInfo(new AlgorithmIdentifier(alg), pubKey).getEncoded()));
        PrivateKey qPriv = keyFact.generatePrivate(new PKCS8EncodedKeySpec(new PrivateKeyInfo(new AlgorithmIdentifier(alg), new DEROctetString(privKey)).getEncoded()));

        Signature signer = Signature.getInstance("qTESLA", "BCPQC");

        signer.initSign(qPriv, QTESLASecureRandomFactory.getFixed(seed, 256));

        signer.update(msg);

        byte[] sig = signer.sign();

        assertTrue(Arrays.areEqual(expected, Arrays.concatenate(sig, msg)));

        signer = Signature.getInstance("qTESLA", "BCPQC");

        signer.initVerify(qPub);

        signer.update(msg);

        assertTrue(signer.verify(sig));
    }


    /*
    # qTesla-p-I
    */
    public void testCatPIVector0()
        throws Exception
    {
        byte[] seed = Hex.decode("061550234D158C5EC95595FE04EF7A25767F2E24CC2BC479D09D86DC9ABCFDE7056A8C266F9EF97ED08541DBD2E1FFA1");
        int mlen = 33;
        byte[] msg = Hex.decode("D81C4D8D734FCBFBEADE3D3F8A039FAA2A2C9957E835AD55B22E75BF57BB556AC8");
        byte[] pk = Hex.decode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
        byte[] sk = Hex.decode("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");
        int smlen = 2625;
        byte[] sm = Hex.decode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

        doTestKAT(PQCObjectIdentifiers.qTESLA_p_I, pk, sk, seed, msg, sm);
    }

    /*
    # qTesla-p-III
    */
    public void testCatPIIIVector0()
        throws Exception
    {
        byte[] seed = Hex.decode("061550234D158C5EC95595FE04EF7A25767F2E24CC2BC479D09D86DC9ABCFDE7056A8C266F9EF97ED08541DBD2E1FFA1");
        int mlen = 33;
        byte[] msg = Hex.decode("D81C4D8D734FCBFBEADE3D3F8A039FAA2A2C9957E835AD55B22E75BF57BB556AC8");
        byte[] pk = Arrays.concatenate(Hex.decode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ex.decode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
        byte[] sk = Hex.decode("F402FF08041001F812050002F9010B04020CFEF8FA0BF3FF060003060709F00CFFF6F101FBFEF200FE100410FF0306000D0AFAFC01FDF2FB00F9F002FEFFFEF5F8F10503F9F8F5F406060508F9050101F7F507F2F902FCFBFD06EE0303F5FA0306F8F812FE0A04F50DFCF80BFCF8FA05FBFD00F0F8F7FFF606FB0BF70508FEFDFEFA0104FEFEFF090DFF03FC02F10E05FAF906FCF6F7FEFA020105FE03020B0F00FAFB05FD03F800F603060506FA0704FEFB0B0B0700FCF5FAFB0A0403F6FF04F80AFC00FE0BF3020701FE030B0AFA0908FCF8FD0708FEF60200EFFBFE0105F60CF5060607FFF400FE0CFCFD040AEE010606FBECFF0F010702FDFD080412FC02FFF601FFFD04000009080403FB0DFD0C04FAF70809030B06FE130607040607FB09F9E6FA06090A000108070E010B030203FA1108FB030B0213FF05F70004050DF008FCFE0D0DF4F70604F4FCF20EFFF90EE5FBF7F9E81301FCF8F7F5FC0901160500F109F9FA0907FAF800FCFBF3FBFC00000504F003060BF109F1040DF9E8040307F0FEFB02FFFE09060308FE0413FFF50602FB040205FE0808FF03E8FDFE02F7F90D0203F1020AF8F1F7FFFEF9FD09F906FE16F418F4FC0907F106EDFE00FA0104FF0101F9F8F0FDFE0BF809FF080B0EF90AFFF801F1FE04F20901F909040301ECFA010CFEF307F60DF5060303F5FEFC08F61100FA11000C05FF0DFD0102040003F710FE01020E0B0B03F2F8FA060CFB01FF01FDF309F505FE00FD0901F2F402FDFFF901F6F500040A02FFF2FD0EF1FFEC020BF9F007F3FAF6F20C060FF1EDFAFC0C03F20405FDFAF5E8100F06040B0CFC0008FEF8F201EA03FCF0FCFF0A02FEFC08F8FF070EF9020913FB09FB0500FFFE04FA0208FFF809FE01F801FEF8FCF5FEF40DF4F6F70503F2FD050BF5FC03FDF111FC010209080C10000D0AFF0AFD04FA04F80F06FF04FEFE0CFFFDF3F00BF00BFDFBF800010DF70CF807030D05FCFDFFF905FAF4F0FD01F802040004040DFFF902F7010000F7030501F501040EFB0A03FA08060709FDFD03FBFBEA090C04F4F109F708010205FBF4FDF4FAFFF20A0106F5F904FEF5FEFAFDFDF6070AF5060FF406F00A0E03F8FC00FA00F4FBFB1105000BF80EFB020011120004FCFF12FD0404FCFCFD0604F30DF401F9EB030C0800F903FAF7F0F3FC0508FB09FBF00C09F7FFE00508FE0AFB0D01000AFC06F0FEF2F2FB020604FEFC07F2F9FBFD0308040610FDFA07FE07FA04000C01F1021315FF0604020AF5090802F4FF0CFF0501F5F90702F1FEF50B010E0103FE060301FBF5F90BFFFEF50700F8FFFFFDF6F8FE030306FEFFFA09FFFAEFFD0EFEE5F906020500FB04F703010311F5FBF003FE01F405FBF606FDF3FF020704F805FEFA08F5000009F3FA13FAFDFC0F0AF9070FF80307FBFEF5F80105FEFBF108040301FC0709FDFC0B0CFAF3FD0100FBF1FE090E0CFFFD04FCFEED0201FEFF0D0AFE0000F804EF100C0CFDF903FF08EEF702FC00160207EDF8ECF7F30808F80AEA0B0908F10212FE04010AFC08F8050300F509F3F718FEFBFD01FAFA0A10F4020204F8FDF808FD15FC0201F8FF0307FBF5F1FEF204FB11F5FBFA0AFB0503FF0505EF08F7FEF8FF0603FD06F9F70BF70300040DFF0D0C09FAF502EF020FF10BFD00FB01080205070CF8F9090507080A12FCF8020BFF030C01FF03FDFDF8FE03F5F606F90700F3F8FAF5F5FCFCF8FE04F6FF03F90FF8F3F404F9FB05F3FDF7FEF70400FB0203FD000B0100FD0411F6F4020FF6FE160613FC0C04F9FEF9F802F701040003F8FE07F9EF0DF1FD0907FE02FB07F6F0F909EE0B0AF9FA07FB00000B01F70BFBF1EE090400F311FE09FEF2F1140403EF02F3F50C0001F80200FE01F5F4FFFCFA14F8FF0504FA050EFA08F007FE04040206F9090111EF020903FBF600090809F606020102F70BFDECFD12F40606FCFA0D03F70FF403FA0BFEFDFC0310F6F3FE06FE020105F8FB0902F307F8FDF103080706F20106F9F3000902010503060FF6F7FCFA0600F3FB04EFFFFA0304F508FDF103030AF1000E06E908F70E05FAF6020AF5FEFBFCFFF1FB0602F8F30E07F808EDFCFA03FEF3EE05F5060201F5F309020505FCFD0DFBF4F9FAF90AFA050B09F508FBFB0B00FD0B02FB11F602FB0D09090901FF0103FFFAFCF708020BFF0FFC0210F90E0A0D030006FAFD04F9FF1006FA05F90AFE050DFA0512FE04050105FFF7FA0C0903FB0CF8FC01FA06EEF6F8F4120402040AFE13FF0E0902FE04FDE605FC06F7F60C00000D0200FCFF10F2EBFD0302F4050DF9FFF60906FF010A0D05FFFC0705FB0BF800FAF806FB13FD030104FB0F0900FA0300F5FC080EFD1202040AFCFEEEFDF70404F612F90FFDF6F906FB0C0AFE08F8FFF905FD010B060104F9F4F8FDFBFA06F6FE12F70305FCF5FBF7FCF6FC00FA04F9F1FC0201FAFE04FD04F8FEFC04F6050BFEFE0308FDF200F50DF40EFDF702F8FDFF03F8F70D01FA01F90207FC0206FBF9FB0BFB0204000907000FFB04F40007FAFBF812FF04050002F301F10B01040003FC11FDF602FA0BF700FC0102F9FD0603FE01FAF00CF700F805FC05FBFB08FBF907FD0007FD00F6FAFCF702F9FEFD04FAFD0601F909FF0AFFF9FCFEFC0108F700FE0C0B050304FBF7F804030E02160B0B0AFEFD03FEFDF50AF7030CF5FBFCF10D0000F8000D020D0404EFFC0507FEFF00FE05FDFA0807FFF90607090C010406FA070201FCF70AED0B0CFB03FC0206F8F805FC0802F8FB030AF4F50607FE0302FFFA0109FB0E0A0403FE02F807030B18F1FEFB06F6F4FD08FBF9010004FDFEF60502030EFF0100FE0804F50AF10305EFF6FFFF0CF8070AF41102EFFBFE04090AF3FE080C0201F1F2F8FF0003F8020106FDFD0B0AFFFD070B00F704FA00FBF4EE01F906F40B09F802F80C0803FFF800F9FEF507FE09F2F70703EDFEF8F40608FE080201000607FBFE0E0AFB0A000CFFF3F4EEF606F8020012080DFEFA15FEFBFF0905FF00F6FD01F801FBFA03FE020B010107FE1105FA0306FC16FA040508F4020708F300FDFAFC000DFFF70AFDFDFB08F7FA150511FE0A00FDF4F8FAFD11FDF303F9FE01F60901FFEBFC080204FFFFF00506071004FD0A01F7F405081100FE0BFE0300F4FBF802F7FCFFEB02F60707030CFBFB00080603FEFEFBF2EC0000FCFE03FDFC02FD0EFC06FBF407050703F80400FD01FDFEFEFFF80B0102060B0001F8F902FA0EF9FE16F901FB06FA01F802F10A08FBF40402FEFE0B050702FD0C0602F90706FB08FFF702F20BFE05FB0809FA08F906FE020AFFF5FC05FFE9F9F8F7FFFDFBF5FEFC0204F8000405FF040C080A05F2F30201F3F3010007011106ED060200F50803FFFAFDFD06F60EFFFA01F705FAFE03050C08FFED080A12F5F50CFEFFF4FDFFFA0609FC040400040107F6FAF1FB0CF30C0C03ECFFF8FDFFFD010603F7F90805FF0902F0FD0A00FBFF0CF601F7FD02050DFE08FFF30C05070E010EF80710F105F6F70210090AF1FFF9F80000F7FE0903FEFA000C02FBFAF7FEFCF7F60608FF03F7FDF9FF01FF040A03FD0206F8F40A06FEFE0B01F4FAF5FDF9FBFF0500FE06F90401FFFF00EEF7FD02FE19F4050E06FEFB0005FDFEFA0307050102FAFAFE0EF60BFFFA0205FF0B040F0103FAFFFC00F106F9F9FEE9F80CFE080AFA070FFCF8F506ED01FFFF06F901F7F2040310FE0000F001F40BFF020C0307FCF609FA06010306FB040002FF1007FD07F9080901F207F5FEFEFFFE08100602F50505FB06F9FB0806FDFC1207FD08FC0001FC0807FAFDF9EF13FF000707020606FC090301FD01FEFCF6020AFD030F09FD04EB0DFD0712FAF704E6FC0C07040B020314FEF6F2020607FA04050CF7F4FB0E0305F808F20403060605F902FCF5F8050603030903FAF2F9020AFBFAFE07F7FEFFEE0CF7FBFDFEFA00F70D01F501030B09FB000211060DFB05FE060207F7F7FC03F40208FAFA0601F9FF080702F706FAF9FBFD11FD000409F7030903F405FBF8FC0BF9E91806FBFEF7F401F80602F8FF0DF7FC090909FA00F8F205000203FB0F0BF100040003FBF603F5060D0AFEFFFFFF0909F50310F4FBFAF603FAFEF60406F8EEF001FF070CF00509F9FAF607FDFD040405FF0201000FFC04FBFD0605FA00F6F8F300F90B0703FF0B080AFEFF0306FDF6000207F0F8F000060502F4F9FD0AF801F700F20CF90112FA0308E70400FA0303FB080806FD0302FF040EF0FBEDFBFC05F40901F407F7FBFE05FA0000100DF8F40C0B0200F900FCFCFA08EF01F7F4FE040101E405F409EB06FAFEFCFAF10301EE11F4F000F5000805FDF4010206FBFE0EF80F02FCFDFF02F9FE05FFFD02EEFD090400F70703FF00FE00F7FCF50FFCF8F9FBFA05FBF4FF05FA0404060B09F707F3FC07F6FB0606F10B04FAF901F9FE0702110602F7F905FEFDF5FEFC06F9FFF503090100F1F6FD0204FCF7F801FEF3F501FDFF01FC010205F3FD050AFC0401F700F2FF070E0CFB04FDFDF50C0EFA03FB06FD03F70605F8FF0700FA0CF40017010204FBFAF8FC0805FB0001F803F705F702080E05F307FE05F90E0B02FDFD04F800F804FB06EBF3E9ED1607FBFDFEF8F80EF00308EF0704FC04F9E8FF0B1202FE07F216EF050213F0F304F808F901ED030514010207000108F8F4FEF9F905F40C070304FF000309FB0EFE000500000BFD07F8F607F80C0301FEF9FBF5FC01F90709F509F6FA0302FFF5FFEBFA0D01000BFD0A02FFFBF6FB010508010D0407FCF300ECFE0B090808F6F5030DFF0A05FBFD06050A02FC02F9F6FAFBF0010507FC00F80404FE0406F1F20303FC010AFCFAF802010FFAFCF4F7FEF7020307F1F505F500FA0600F8FE07FCF70600F4F4E8E4FDF8010EF1F9FF0107F5F00303FE0DF3FD02070401FD00FCF6FEFBF0050400F60A0EF8FBFE030EFFFAF6FFEE05F7FCFA0CEBFDF6FA01FFF5FEFEFAF2FC04010512F9FB05000407F200FBFBF5F2F90BF5040003F403020A0603FBFFEF0C03FA0FFE06FF070B03FD0AFD04FC000AFEF7F306100108FFF512F2F50511FCF1020605F8F5040CFD08FA00EC0D05F7020602F9F2FB03F4FF00040B06FBFEFDFEF712FB18070FF8FBFBF500FB07FE10F40004070200F00C03FF0215F801F605FD04FFF8FB07F9000E0B03070B0AF4FEFDFA06F806FBF7EEFEFCF6F8FE0A0EFFFE03FFFD00F7FE01FDFDFA04FFF402FDFF050FF9F4F6FB12F901F0F0FFF806F40B0AFC0DF8F3F3FD090800F9F1FC0B06FD0604FD070711F406000CFDF8F507FCFE0605060E03FBF5010508F900FB03F8EB010AFEF7FB03FDF20402FC01F502FBFAF600FE01F407020402060B0111F201FD06FBFCF6F204090101F9F8FCFAF7030403F800FB0D03070003FD080EFF0506F90DF406FB07080AF705FC060205FF0E04F7F912FBFFEDFB0A120E01F3F402F907000404F404FD0205F60CF0070005F603F9F303FDFB07030B0413F113090001F60504FE05F4F60C010905F809FE01FAF90302FDF6F308F7F7FDFEFC0405F60B0B07FFF708F1070900F60100FAEEFD040107FB01020B01FDF902020A14F4FC0D070402F8FEFE070EF9FCFDF300F40601010213FB07F600FA020900FC03080407000501F609F8F300F8F508EE0C05030300FAF9080208060EFBF402FEF7020303F2F40D0AFC01FCFCFA0AFD080D020DFFFB09F80CF902FF0907FCFC00040D01EFF9F8FCECFDFAFC01F9080304F706F6F3FE0004080FFEFDFC00EDF700FF02FE17FBFF0701F5FBF5FB0801FA040EEAF30502010711FD04090A09FC1302F309FF0803F9FA00080704FBFD0003FFFAFE0E030007F902F900FBFFF8FD02FEF3FAF3FF02FE040DFDFD0400F014F9F8F704FEF2FDFE0600F3060D0000060AFE0BFC040A01EEF90E0402FAFD0610080206000E06FAFA0702180FFFFCFAF5FF060A050407040003FA0E02F7030E0EF8000200FA0105FBEF0902F808FDF709F9F7F609EBFBFBF416F902000601FA06F906FB080BFD11FC0603FA0DFF00FCF0F307FD0502040100000307FD030CFD0E0402FDFD00FE08F8050B05F103FEFE0501FEF703EEF80401F3050A1100FF0C02130804090008020012FAF3FF06F0070404040905FA06FCFB0203F4F9FC0604FEF2090EFD0515FC030308FA0400FDFDF90D05FD03FC0C040301F2FF061405F8F809F9F8FBF6FAF4FB050AF206FAF70702FA0502F10011F00009FA07EFE5000811FBF70405EF0412F3F902FBFFFDF9FF08F906FCFF0BFDF603F5FA0A05F9EFEB00E9FDFD03FB010302F404F503F3FBFAF8FEEF01FE03FA0AFB0B06FE05FAFAFFF3FCF3FC0BF7FF0B010DF103F6FB0412060503030A010107030901FFF6020E07FB06F70102FBFCFAFA0EEFFFF3F80800FDFDF103F8120707FCFA0C02FE0DFAFCEFFDF000FA01FFF701F802FFF9090C03F708FDF9F1F8100608010DFD09FD06FDF6F8F60AF90DF50C070CF9FDFBF100000A0DF8060808F9000608F804FBFEF6F80803F6030A1003F103FDF6FF0AFB00F609F6F601FB06020206FDFAF7FE10F5F9FEF8F8FDF8FB08F8F7070102FBFB000EEC03FF0B00FD140B0102FB01FF0801F605FFFE06EE02FDF0F8F506FBFFFF14FCFEFD040304090201FEFD0504F602F9FEFEFEF0FE05FF05F3FBFA0606030F0D0502F807021504FF03FA0A050601FF07FFFD04FD00FA0BFC1500FA01F80AFBF2040206F700FE00F402F6F7FF0FF901EE030CFE0AFB0310FD0003FDF2F005050DF6F717F803FD03F5FA0B0CFE000CFFFAFD0D0509F3030004FEF8FB080A01FA0403F2FC02F90BFA0807F80703FD0E040D030BFD02F004F0FB03FEF20105FE000206FD0908FA01F30AFA01F302F9FF05FE040DF500FD03080C09F6FDFEFF0A0006090EF108090409FFFB0008021100060BF80005F7FEFF0C01FEFA050B0300FDF0FE0CF7080014F6FBFEF208FE04FBFC0B010600F8FE0B03030C080604F306050A0002FA0507FF03030AFEFA07FA020B06FA00F600FD01000107FC01FFEFFE0306040EF6F6F7F6F30F090404E9F604030DF802FC03FDF20EF90202FBFAFEF507FEF303FFFE0CFFF3010006FB02F207FDF304FDFE03F8FF08FA0C030BFCFD070F010902F604010016F9FA06FF02080B080E11F8F6F7FA0E0003F7EEF4160202FCF509F5FEFC0A050406100702FE060B0601070C08030302FFFFFA1303F9F7F5FD1315F2FDFB010710F105F8F40EFBEF0F110604040D01FD050C0AF9FF0C030401F9FE05FBF60FF60808F5F8FD0EFCEA040DEE01FD03FAFB06F9F9FDF806FF000205FC0302FE09031404F6050301F60915F60D0C0D050C05FF0FFC05F4FE0502FAF507F0F70B14FEF801FBFDFCF3FAF3F7FCFB01EEFF00F9FB0306080503060801F7FA00020EF6FDF50400FD07FCFA0401060BF9F6F708F9F80DFB0314F10B01FBF80507FBF6F406FD06F507F6F006010BF4FB01FDEF08F7100103010206FC0903FF06090104FC11FFFD00F70D07FBFB05F5FFFA09FB0101F6FB02EF0A07FF1300F102EFF40104FEF605FDFCF4FFF906F706FB05F9FA0AF8FDFF010403F90501140903FBFBF903020A07FDFAEF010F08FAFDF0070912FDFA0100FD0CFB060001F803010800E9FB04FA09000D091AF5F2F5FE0FFCFA0AFDF9FDF90C02FE01F1F8F9FDF3FF02EEFCF503090EF4F406FB0DF4FFFC09F6F007FD0B09F207F108040B1007FD08F3050604FB08F7FFF8F504F305F6F3F0F504FD02FF0CFA04FDF1090BF7FB100603FE061309FCF9FA00EFF805FB0D04FF0C01F6021305F215FB07F003F90104F8FE04170B0001F40D0DFD0508FD02F0FFFE01FF0C0D020DF706030E05FEF8F9080603F4F806000F04FB0100020AF903FB04FF10FA070CFF060700FF02EA01FCF6F00604FD0FFA0EF50FF70801FEF604FA080102F2FEFB04FAFF02FA01FF04FB04011703FC0FF00307FBFCF610F7FBFBF403FC080905000C0D0908FBE60A0A13070B0F0AF60002F4EC0203FDF8FDF9F7FCF301F8FE010912070208FF00FC02F9FF01FD00FDFDFC12F5FEFEF50308FE130000F8F6FF0410FDFD130107F9FD0802110103FD0408FCFFE9F605ECF406FB0C08F9F609F3000C030DFF0A0CF50C02FA0F01FD0902F21500FD01FFF7FB01060005FBF9F4F7050D07000801F6F007090AFEF50C0005FEFEFA0DF902FC08FAF6FE0807F4F901FF12010509FE090D050D0D04070EFF0003FC010AFFFB02F7FB03070507F7FEEF000808060D0FFE04070502060804FA0CFBEEDF03F7F9F3FC1008040207F00CF902F50903FDFE03FD05050709F6FE03F1FAF909000401F5EEFCF2F7FF030F020D0B0EFCF5FFFDFC17FB03F6FA0E05FD0C090F0606F40306050AFFF50804FFF908FCF8FD03FEFEFDFC02EF0DFA00F401FCFA00FC01F7FBFD0A070D050AECF6F7FC080DF4F6F707FBF9F0FE00040401F705010501FC0AF60E03FF0600040B04FA090B05FE0901F4F80400F8F1F506FDFE0EFC01F60E090D01FB0BF90201FDFFF011F50F0203FD01FBFEFDFEF504FF0407F70C07FA07FDFB1206FEFD02F903F1F5F80A0BF2F6FCF8FE0501FDFCFD08FB0F05FAFCF0F8FF0303F0FC0801F70109FD020708FE0A03F3011207030D04F10C090701FF0E05F80205FCFC120601060FF8FFF9040905F908070912000FF2FB090004F9010402F3FDF1080DFEFF0A010EF904FFFD02FBF707FF110D07FEFAF80AF2F7F60AFC0401F804F6FA02FCF9F7F6F10605FE07F8F2F6FAF5FA0000FE040107040EF6FA05FFFEF903FB0502FDF903FB03F9FE010104FD0FFE080304EB0B01E8FD000F0AFC02F5040AFE0702100CFBF4030604050008FFFA0215FA00EB00F8F5F706FE0BF907FD080600FB10FD1104030B06F8040600F8FD0DF60F0B0B0AF0F80201ECF4FEFB020B05EC030100050301F8100A03040C05FA0910FEFB04FCFB000104050DF5EBFEFAF9F7F81201F90912F20BF905FAFAF904FDFE0EF5F8F2FFF6FFFA0103FB0FF5040401F3FC04FBFB1102FEFBF903090B0803020807FA04010BF2FF02F7EF01F409FEFFFD05F90F0616FA0B0307FBFF02FBF703F9FBFE030CF802FCF8F30B14040710F2F3FB0A0BF80BFF040A04F3F702F801FFF5F505EDF6060C00EEF1FF0A0B1006FF03070F09FE07FCFFFD03FE0CEFFF0A080A000900FA070C080003070B0AF5FAF6F0021B0E0C00FFFFFCFC0E0714000902F1F7FAFEF60708F60007F5FAFCFC060501010EFE0BF506F7040500F1F9FE0CFFFC0611FCFBF4FE0D05FCF91205FE010FF4E9FCF7ECFDFCFA0B03FF06FAFDF001ECEB09FDF901010AF5000EFCF6F5FA0109FF150A0306FE04010F000906F6F90EFFFB010706FAEFFB030605FE0BF40502090D02EDF5FF040AF6F4FEF8FDFD07000CF6F302FA04ED05F008FEF106EE1306FD1305F20405000DEC0601FBF8EE07FDFC11FDFA1004F70DFF0AF4F9FE01FC0D03F1FFFFFF0605F7FB03F9FBF4FB01FB01FD01FEEBFFFAFA04F2F90701FFFEF304F90005FFFCFEFFF608F915FAFDFF0406FFF70204F6FFF704F4070909F40EF90102FA02FEFB0505F8080CFCFCF8FEFD05FDF70109F80BF908F9F8FC02FF0503090AE9FEF5FBFA081201F30AF90408FB02F7090101F502FFFEFD06FD0B0C13F6F905FD01020306FA01F7000BF7030E0CF1F7F80AE90FFEFDF8FD03FCFCFA040709FF0900F8FEF204FBFFF3FCF602FB02F80B10FF070C00F50C0A01FA110DF7010EEB0001FAFFFE0001000F130F05020E09F70102FB09EEFBFF0104FEFF05F40BFBFB09FC040405FBF4FD0A00EDFAFEF70011F2F210FE0AFF030F0319FCFFFBF8F4FD01FDFA000408FCF70AEF030005F104F305EEFFF60506F307FA02FF02010AF50DF30BFDFFFC0712FE090CFE0309FE0507FF09F5FBFB0211E9030606F4F201F8F0ECFD03F9060BF10108EB040AEE0310F5FFF7F3F9F5FB03120402EEEE031100FE030FFEFFFF060AFC040004F1FA0B03000D020F00050906EF0B02FFFC0306060201FEF2010E03FF05F9FAFA0300FEF9F9FF09F80BFFF605090D0009F8F606FEFEF8FDFCF0FBFDFEF60D0902F7FAFCFCF50BFE040CFCFE01EC1308FC08080EEE110006F300FE0503FC0408F901FDFDF7F90D0502FA02080DF502FD01FB01050A01F004FF0BFD08FE0C020AFDF4F302F005F9FB12FA00FAFEFAFF060102F1FBF9FC040904FEFF06F8F80CF80A0FFEF707FC06FBF10EFA020D09F90EFF12F8F9F512110300FCFC020EF71206090801ECFFF016FCEEFE0202030900030205F7FBF8EF0306F2050507F7F6F71D02FE040108FD03F0FCF604F807FB04EDFA0B01F70A0803FF0C0D09FC03FFEE0E110105FFFA01FCFF0B03FC09090303FDFEFDF6FC110002030402F80412F7FE050005F9050F040202EFFEFE0A0AF5F8FFFAFFFD0D00FF0106060101060C04FE01FFF7FAF6F612FB020910050E12FDFEF6040D09F6FFF7020301F3F2FCFA0CF404FE11F4F1F3FD05F509FDFE0DFD01FAFCFAFAF9F6F80309FA160105F5FD0C02F8FD01E9FFEDF4F5F507F20609F907F3EBFB03000C0506F8FFFEF6F900060EFE0D0606F8FCEAF1FC0905000C0514FE09FFF800FEF9F60CF9F6FE09FE1502080F0CFAFCF20907F410F20CF7010C0108F6FA050AFD02F8F60B00FDF604FD03F6FBFF03FF0DFCFBF3FF010201F302F2FB08FE0401F400FFFD05FDF5F0F4070202F3FC0008FD00FEFDF80105FF04FD03F902FD02000206FAF8F80409FE00FAF8F8FD09FBF2F8FD0008FEFCFA030107F3090102F8F602F2F8FE1007FBFB09060405F4080C070209FDF307F8FBF80210FC00FCFAF2FAFCF911F3000809FB0E06FE040807FBFBFD000403F8F8FA000BFF0209FFFB0602FAFC0A050AF8EE0BFFF4EFFFFEF309FA0105FBF702F6020B0C02FD0AF2FA0F0401F902FEF2030106FFF20C0410FD07F00304FDF70505F9FB04F800FFF3FA0D0003F8F90D0902020AFC11FFEFFFF5F40DF90703F8130FFB04F30A0602F7FAF4FCF7F6FCFD0606FA0CF90200FEFF1107F3FB0D12FA08F60904F2FBFE010B060010F8F403FB04F70BF8F00DFEF809130607F702F8060A0004FAFC02FDFE0BFFFE12090D00FE00FAF7F4090104F6F209F9FCE6031103060CFDFB080207ECFF010304FEFE0705F9F10800F9F7070AF80704FEFA06F2EAFD0602F7FEF305FDFC070009F103F80303050A04110600FBFBF8FCFAFAF1FDFE0CF2FB080407FDFE0A03FBFE0211FAF409070CFE010CF4100607F6FA0F0E0BF6FFFDED05FC010FFA0200FFF90206FB0305FD0BF507FC04FB021BFC03F5090407EAFF0DF1F2FFF506F906070D0AFA1C0203FE09FF06FC05F007F30302FBF5F8FCF90106F5FA060404FDED07F8F6F7FB02000C18F6F9FF04050D0712FE06000102FD00FE0CFE03F701FDF3010304F9F30A02FF040A0AFAF1010C00FF01FFF7F50A090307FEF703020B03FA0AFEFC080205F6FB06F20214F608F9FF02FFFC010D040B07FEFF00ED0A0500F306040507EC03FD0101F40D00F0FB00FFFD14F603000206EDFFFBF40A01FCF70904090702040602F4F801020506FEFC09080800030201FB06000708F4FE0909F70DF3F0030C0507FEF4FE030503F20902F90B07FBFE0002F60902F6FE05EFF0030509FB0CF9FBF508FB02FF17FBFAFB0513000005050A060F06F6FD070605FB0CFC0710F0F90A0D06FF03FFFAF4F3FE02F605FB0402FF01F807050603F501FD05F7F7F2FDF1EDFF07F6FE00F0FDF50A0905040407FFF708FEF41200FCFCF5FAF5FE07F8FFFEF9F20609F009FDF50606F9FAF2F30B0104FAFA120300FEF9F907FFFA1002F30C0805FEF7E90A0102FFFB02F3FD0CF5FC02FBFCF8FE09FFF8FCFF0E040E0806F908070FF903F300091509F80C0FFCF700F3FAFFF8FC02F6FD0905FD0A1001EC070012F5050809F2F5020A0E08F4F8F3EE030106FB08FFFF00FEF402F703EDF80EFF14F9F902EFFAFD1800FE0107F7FFF106000B06F6FFF4FE020110091000F7FFFB0E0E0302F1F20806FD0501FCFC010105F7070A01FEFBFF03F706FA130207FE12FB0AFE07FDFDFA01FBFFFB0606FB03020BF3FF00FDEFF5F30AF6FDFFF9F9F70307F7010403FA00FF04EF04F70BF3F501FCE8FEFFFFFCF80100090CF300FE03FE0A0C00F0F7FC020DF60207FEF000000B0CF3F5FEF700F7EFFCFF00F903FE0809F9FBFA0BFA0EFAFF07F1F9FB01FE06040510081001F103F90BFDF70700F2F801021001FD08FAF6FB06FBF5FF000CF4FF02F60C00F2FAF20207FCF6FFFC00FEFCFC08FD0FF60002F00104F801011201070203FE090204070203FF11FDFC02FE00FFFC04030CFD00FD030E01EEF6FA08F5F801FA0205FE04FFF800F1F8080C03FEF80809F700F9F7F90A000413FDF4F20E01020A04FD020CFB10EF01F90B04FEFAFB040A03FAFBF70708FA06F5050005FF01000DEF07EC0E05FFF604050C040EF2F70703F8FDF0FAF40C140FFCF4FB0C060008FEF3FDF201E30CFFFD0E0606FBFD0711FAF7F708030304F6050A07F6FCFC0409FD010501FD02FC0B07FD080E0F0D030DFE0E0B1300FDF4F304F2FAF807F6E6FB01F90701FD0907F9FEF50D0EFC0FF9FFFE09FB0516FF060008FA0701F8F0F807FD0E02F3F8EF0109F4010BFFFBF9070BF5FB09F1F7F8FE0E0BF9EDFC080E0DFCF3F4F7F607F903F9FD010AF60302070D08FC05FDFEE9FE09FDF4FB02FFFBF90206F3F21802FF0007F4F203040500FD08F9F602FFFBFF0402FB08FD0006050501F7F70B0801F7FFFAF5000E0BFEFDFF060901F703F30503FDFAFC00FCEE01FE0C100E05050B02FF040E08FBFBFC0BF0F70704FBEFF4F3FC02FEFD0701FF0BFC00040402FEFA050BFCFF040A05F4FEF3F7FB09FC02FB010B020B0906EFFC08F703040111FE06F609FEF4FBFE050BFBFFFCF80010F90A0105FFF3F003021106F405F5F6F5FFF4FFFD0105FD07EF03F101F904FE04030A060703070B04FE09F8F4FDF60009FF000FF504FA0CF6F6F5FAF302F9090506ECF3ED05FCFE09F7FF0009FD02F6020E0BF4FD00F7FAFB01F6F50608FF010508FBFCFCFBF9FA0BFCFFFB0A01040600FDFEF8FC05F5FA0B050BF10000FF020B0F0200FAF9060407F5020A0C05F80DF90706020EFBFD03F61200FAF90DFD1204FF01F702F8FAF4FF0501070012EFF9F8ED09000A020103F9FA030D04FCFE09F9FBFBED02F9F015EDF20808FD0B0AFAFF0001060204F6F00F0404F703FC09F801F4FDF004F700F9FCF8FDFCFA030FFBF7EFF4FDF60800FB04FB04F501FF0B0317FE0C0DFD0E01FD03FB01F50A03010503FBF0F6F902F6F0EF0B08EEFC02F40EF9FE0808F70BFCF6FDF6FB09FAFDFBF3F1F5F3F1110A0205F2050EFCF901FB09FDF1EAECF90707FCFBFCEFF8F702F70AFF0A0C0307F506FF01F7040C0104F9F80805F603F504FD0303FB02F8FBF7F207FA02F507FD06FF020DFD0BF90E07060CEB100A0CF30106FF03060702F716FB0B0607F8080509F2FA0AEF040602FA0501FD0303010605F4FDFD0306FD0A0B04FAFA0101FA0F0313FBEE031109EB13FEFFFEF9090906EF090903FF0F06FA06FE0404F1F9F2F8FDFA0809EE0F070213F605FF0AFFFCFF0302F5F9F60906FF12FFFA06F709040607080B00F705F3F900F50E02F91009F90AFBFAF1FBFEFA0105F602F41201FB01F70DFBF1FF02FFF10109F5FAF8020606F9EDFE0008FCF402010EFCF0F9F1F106FC0402FAF9050109070AF313FEFA01FC090401F7FAFFF808FAFB0209F003F00C08FCF5EF03FAF3FFFD0907FC03FDFCFF0DFEEFFCFF0200F50A0EF7FD07FD0E0604F703070400FE010EFB01FEFB0501F3FAFB06E8070B08FAFC1304FF0404FD11FE0605F70AFE02FB0DFE0307FEFFF6070000F5F5FCFFFD1307FCEA00FCF7FE0D00EAFBFA000AEF0105F60302020F0108FA03F4F7F8F3170E01FFFB0907FA0606FD04FEF508F904000D0CFB08F10904F7FF07F2F501FDFCFCFA0AFD0C010D040C060106F2060F00F708F2FDFAFDF606EBFDFDFDF80201F3090CFBFCFF02FA0B0402FC0901F9F7FE060BF703F8FAF60300FD03010006FEFBF8F7F60401F807FE1707FE08FDFEFD02FC0AFDFEFE0801F9060B14FA070FF506FFFDFBFDF40812FBFA0C0500090C02EEFC02F7F40FFEFCEC02050911FCF6FB19F6FCF2FCF3FC08F506F2FAF9FFF211F3F8F902FDF4080A0CF910FA0D09FCFF09FB0E0BF100FCFAEF0A020AFAF9120403F7FDFFF8020CF800F8FDF8010DF60400FDF80EFDFD14F7F011030213F900FB0507000001F9000C05EF01F5010518020BFCF8F6F70A0401F7FC100C12FBFD0401F6FEFCFAFD04FD05030DF40105FD000CFA03F20600EC0101FB07F7FFF9FB060CFC0E0402FF0B0903FD08F9FEFFF30111050809F5FAFE0DF7F7FF09FF0902F60C01EDFD03FB0403010DF5EFF80901F5F8F500FDF002FB16F1FAF9FB020402F902F6F70BFF07FF09FB06F700FC0606FC03FA0305FC070D0FFE0300FC0808F9F6FD05FC0603F5FC020E0409FCFC010009F90AFFF709F3FB01FFF9F112FCF40001FFFDFE0714FFFF0504F805010B0010010C05FEFC0400040203F8F209F9F00AF303FBFEF400F3F210FCFDF605FD0B080702051403FD000002FE0505FC04000EF8F6FEFB050509EC0802F0F30702FEFAFA03FF0406040300FBFA04130B020608FA0303F7FD130004F2F7FD070702FF0802F809FDFD09FEF8F404000D0105EBFBF7020FEFF610F30800FD04F2FD000804FC070110FDF9FAF902F90EF70009FBFD10EC05040BF7FD0107F90C0401F3F8F80205F901F8FBE8FC08FC0801FBFA0003070004FB0B04FCFDFD0EFBF9FE0FFA0B01F3FD0103F703FCFEFA01F901FAFD000AFE080706F3F6FC06FF04FDF70403F50101F7F9F2F7F30702F80CF3FBFBFBFFF9FA090AF7F801F101F7150D08FD02F9EDFDF30E09F0FEFAFCF8F3FDFB0C03000307F9F501F7FC09080407F10BFE00F80C04F80B12F8F4FDFFFDFCFCF91201F5FCF3FBEE00F300FC0603F9010601F702F5FCFDFCFE0904FAFE0E13FDFB08F80204020407FF0005FDED0FF505F50704FAEE0E0AF9FEFAFC06041601FBF1F513F605F804FD0F0A09F2F9F60202FA070800FA03FA0FFBFEF1FDF8F7F600F807FEF9F207F50210030003070A030FF90EFE01F9FEF8F6F500040209F7FF0006F7F902FBFAF90CFEFFF6030F0A0DFCF7FD06F8F50413EFF8FDF1FEFF0BF8F701F9F1F7FDF91CFA0CFC0609F90C050809FE07FD0805F804F40004FBEE01FFEC0404F20403F9FDFB0AFDF4EF020B08EB0BFAF30B04030BFF0307060706050F0702FA08FB0AFBE40A00F9F405FB00FBFEFBFE0405F3FD06FAF3040A0EFF02F6FD02FDFCFEFF11FDFBF6F70208070701010304FC04F6F90603FC01FAF60708F8FB0300FAFE03F102F7F508FA08FFF8FDFDF0FC020111F6F1F8FDFCF9FC04060EFD04F90A0AFCFBF200F603FFFCFCFEFEFF07FCF40D0900FF0400050BF60107FC06000BF709FB090D01FBFBED0902FEFD04FE04F900FCF7F201FCFB03FD08FBF9F108F7F406FFF7FCF1FDFB01FF06FFFE04F6FB000208F501F905FE05FAFE0AFC04F503FAEEFFFAFCFC010900FB0AFE09EEFFF4FDF7FCFA00FE0A06FD0006FF00040A05FBFA0B11F806FBF1FF0BEE08FFFCF00809F705FDFFF2FC05FE09F60A070AFDFAF7F4080F13FCF6130D080BFD0EFAFAFE06FF06FAF300F508F80903FFFAFEFC09FFF90CFBFF050DFB0300020504FDF9FE080608FA09FBF500F200FA09F8F20109FF0C0DFF030006FB0CF7FA11FB0104FA04FEFE02FE04F6FDFFFDFE000CF70FFC03090004FCF209FCF8F4FF0EFE04EBF2FD04FCF108FBFE080301010907FFF106FCFF04FAFD080BF7FBFCF110F9020B0601FF07FB18030E04F3FE0804FEFF0406F9040C05FB0A1A04FA0306FC020C000D02F905F10400F7000DEDF50108F40309FEF6FDFFFD01FF0AF60402FDFCFB0A02FA06FF04F500FDFC01EFF702FB0004FDFA0609F9020002FF05FBFE0302FB07FB0AF90904F7F20B0F10F9FF0FFC0501F00308FBFEFEECFE0B000301FCF1F405FF06FE03F9F5FAFB05FD060603090710F6F803FCF80FFF03FAF5F4041200FEF9F503FF011413F3FBFFFAFCFDFE04F6FCFFF4F9FFFFF9FF00FA11F5FA04FD0204F9FD03FBFD07F80303F306F2FC10080D000505FCF9F90F04F9FD06F8030304FCFEF3FC0DEE06F609FCF9FB02FF0504F8FFFDFE0600F2F3F60FF5FCF9FFEA0B070C0205F3FDF4030211FE04F40DF608040EF5F80007F301000D04000FFBFFF7FCFF0C041B050201080913FD05FC0304F30909F6FCFF0701F9F6F6FB0AFDF8F40E0507F504FE040F01F200FFFAFF00EFFB02F50404050810030608F8081001ED000BF9FCFF0005FCFBFB020906050C11FEF7F303F502070605FFF6F805FCF8F90CF3FBFD03F10001F9F8FDF4FB07FBFDFAF708F00B0501050301020003FB03FC04FBFE0DF9F6F8FCFEF6F1F9FEED00FF06F3F0FA030205F6FF03FCF6FBFB020302FFF903FBF903FB00010600FF04EFF90803F7F207050F000007FD0BFDFCF1FEF9FAF4FD0208F0071113070209FA05FCF908FE0C0D01F4F3FDFF0EF708FCFE07040305FAF006F00904FCE502EEF601FDF70EF404FFF402F9F8FD0C0602F805FCF00500FB02FF05FDFF0305FF0709FD00031102040EF4FFFEFFFE05FDE5FF100EF002F70406FF080900FBF5F705040313FD0208F4FCFEFEF6130BFE0E08090D04FDFE04FF00000607F3FD00FE00F706F7FD07F605F40006FFF400FE0DFEF200FAFE0B04FEF5F7F90001FFFFF5F4150CF804F200F9FCFDFC05F802EFF2FE04F41101000E0700FCF613FCF30407030311FDFD01FCF504F5FBF90E08FFFB04FAFEF0FD00FD020CF7FE000602F7F9FA04FAFFF70300FBF9F2050C08FE0104F004F7FBFAF4FE0EFFFC09FF00FEF702010403090007FBF006FBF50711F90605000211FFF90B08FFF6F901FC0E050007FAFB0202FB0CFD05EC05EA010B0CF6FE050309FE06FA0C000202F605FDF706FCF4FF0B03020001FFFBFCF60C06001500F1FCFBFCFAF608FAFD02FBFAFA03FE0100FA0C01F404F9F2FC07FB1504FD0809010700F90EF609031001F802F8F4FBF203FB14F705F9F70708F7EFFF0CFBFBFB08080909000404E700FCFC000607F104FD08000803EFFB04030AFA03FF02F4FCFA050E010507FEFEF510ED0905F205080DF5F00002F9F20A0C0CFF080401F9FE01FD05F8000202F1080DFF00FB1404F709FF0C050EF90CF3000A1406F0F9FD02FCFA0513080D03F5F500FDF8F403FF0BF902F6EFFFEE04F70511FE08F70DFFF6F30F0905010007040400000406FDFD0107F910F2FCF9FBF5FD0207F101FBEF07F4FF0209FA030DF90205F50C13FF06F8011609F8F503F8F9F8FC070704040304FD00FBEFFDF2F8F8BC708C44E3D8A298F708196BC66F75F91732E5AF062775A9ACA36CE2DA64BCF62CAA4F63293C7A8F894856E9F263EB9CA4A0648141B4B0EA3A2D3364C36A83");
        int smlen = 5697;
        byte[] sm = Hex.decode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

        doTestKAT(PQCObjectIdentifiers.qTESLA_p_III, pk, sk, seed, msg, sm);
    }
}
