// Berkeley Open Infrastructure for Network Computing
// http://boinc.berkeley.edu
// Copyright (C) 2005 University of California
//
// This is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation;
// either version 2.1 of the License, or (at your option) any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU Lesser General Public License for more details.
//
// To view the GNU Lesser General Public License visit
// http://www.gnu.org/copyleft/lesser.html
// or write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#if defined(_WIN32) && !defined(__STDWX_H__) && !defined(_BOINC_WIN_) && !defined(_AFX_STDAFX_H_)
#include "boinc_win.h"
#endif

#ifndef _WIN32
#include "config.h"
#include <cstring>
#include <string>
#endif

#include "parse.h"
#include "error_numbers.h"
#include "str_util.h"
#include "filesys.h"
#include "miofile.h"
#include "app_ipc.h"

using std::string;

const char* xml_graphics_modes[NGRAPHICS_MSGS] = {
    "<mode_unsupported/>",
    "<mode_hide_graphics/>",
    "<mode_window/>",
    "<mode_fullscreen/>",
    "<mode_blankscreen/>",
    "<reread_prefs/>",
    "<mode_quit/>"
};

GRAPHICS_MSG::GRAPHICS_MSG() {
    memset(this, 0, sizeof(GRAPHICS_MSG));
}

APP_INIT_DATA::APP_INIT_DATA() {
    project_preferences = 0;
}

APP_INIT_DATA::~APP_INIT_DATA() {
    if (project_preferences) {
        free(project_preferences);
        project_preferences=0;      // paranoia
    }
}

APP_INIT_DATA::APP_INIT_DATA(const APP_INIT_DATA& a) {
    copy(a);
}

void APP_INIT_DATA::operator=(const APP_INIT_DATA& a) {
    copy(a);
}

void APP_INIT_DATA::copy(const APP_INIT_DATA& a) {
    memcpy(this, &a, sizeof(APP_INIT_DATA));
    if (a.project_preferences) {
        project_preferences = strdup(a.project_preferences);
    }
}

int write_init_data_file(FILE* f, APP_INIT_DATA& ai) {
    char buf[2048];
    fprintf(f,
        "<app_init_data>\n"
        "<major_version>%d</major_version>\n"
        "<minor_version>%d</minor_version>\n"
        "<release>%d</release>\n"
        "<app_version>%d</app_version>\n",
        ai.major_version,
        ai.minor_version,
        ai.release,
        ai.app_version
    );
    if (strlen(ai.app_name)) {
        fprintf(f, "<app_name>%s</app_name>\n", ai.app_name);
    }
    if (strlen(ai.symstore)) {
        fprintf(f, "<symstore>%s</symstore>\n", ai.symstore);
    }
    if (strlen(ai.acct_mgr_url)) {
        fprintf(f, "<acct_mgr_url>%s</acct_mgr_url>\n", ai.acct_mgr_url);
    }
    if (ai.project_preferences && strlen(ai.project_preferences)) {
        fprintf(f, "<project_preferences>\n%s</project_preferences>\n", ai.project_preferences);
    }
    if (strlen(ai.team_name)) {
        xml_escape(ai.team_name, buf);
        fprintf(f, "<team_name>%s</team_name>\n", buf);
    }
    if (strlen(ai.user_name)) {
        xml_escape(ai.user_name, buf);
        fprintf(f, "<user_name>%s</user_name>\n", buf);
    }
    if (strlen(ai.project_dir)) {
        fprintf(f, "<project_dir>%s</project_dir>\n", ai.project_dir);
    }
    if (strlen(ai.boinc_dir)) {
        fprintf(f, "<boinc_dir>%s</boinc_dir>\n", ai.boinc_dir);
    }
    if (strlen(ai.authenticator)) {
        fprintf(f, "<authenticator>%s</authenticator>\n", ai.authenticator);
    }
    if (strlen(ai.wu_name)) {
        fprintf(f, "<wu_name>%s</wu_name>\n", ai.wu_name);
    }
#ifdef _WIN32
    if (strlen(ai.shmem_seg_name)) {
        fprintf(f, "<comm_obj_name>%s</comm_obj_name>\n", ai.shmem_seg_name);
    }
#else
    fprintf(f, "<shm_key>%d</shm_key>\n", ai.shmem_seg_name);
#endif
    fprintf(f,
        "<slot>%d</slot>\n"
        "<wu_cpu_time>%f</wu_cpu_time>\n"
        "<user_total_credit>%f</user_total_credit>\n"
        "<user_expavg_credit>%f</user_expavg_credit>\n"
        "<host_total_credit>%f</host_total_credit>\n"
        "<host_expavg_credit>%f</host_expavg_credit>\n"
        "<resource_share_fraction>%f</resource_share_fraction>\n"
        "<checkpoint_period>%f</checkpoint_period>\n"
        "<fraction_done_update_period>%f</fraction_done_update_period>\n"
        "<fraction_done_start>%f</fraction_done_start>\n"
        "<fraction_done_end>%f</fraction_done_end>\n"
        "<rsc_fpops_est>%f</rsc_fpops_est>\n"
        "<rsc_fpops_bound>%f</rsc_fpops_bound>\n"
        "<rsc_memory_bound>%f</rsc_memory_bound>\n"
        "<rsc_disk_bound>%f</rsc_disk_bound>\n",
        ai.slot,
        ai.wu_cpu_time,
        ai.user_total_credit,
        ai.user_expavg_credit,
        ai.host_total_credit,
        ai.host_expavg_credit,
        ai.resource_share_fraction,
        ai.checkpoint_period,
        ai.fraction_done_update_period,
        ai.fraction_done_start,
        ai.fraction_done_end,
        ai.rsc_fpops_est,
        ai.rsc_fpops_bound,
        ai.rsc_memory_bound,
        ai.rsc_disk_bound
    );
    MIOFILE mf;
    mf.init_file(f);
    ai.host_info.write(mf, false);
    ai.proxy_info.write(mf);
    ai.global_prefs.write(mf);
    fprintf(f, "</app_init_data>\n");
    return 0;
}

int parse_init_data_file(FILE* f, APP_INIT_DATA& ai) {
    char tag[1024];
    int retval;
    bool flag, is_tag;

    MIOFILE mf;
    mf.init_file(f);
    XML_PARSER xp(&mf);

    if (!xp.parse_start("app_init_data")) {
        fprintf(stderr, "no start tag in app init data\n");
        return ERR_XML_PARSE;
    }

    memset(&ai, 0, sizeof(ai));
    ai.fraction_done_start = 0;
    ai.fraction_done_end = 1;

    while (!xp.get(tag, sizeof(tag), is_tag)) {
        if (!is_tag) {
            fprintf(stderr, "unexpected text in init_data.xml: %s\n", tag);
            continue;
        }
        if (!strcmp(tag, "/app_init_data")) return 0;
        if (!strcmp(tag, "project_preferences")) {
            retval = dup_element_contents(f, "</project_preferences>", &ai.project_preferences);
            if (retval) return retval;
            continue;
        }
        if (!strcmp(tag, "global_preferences")) {
            GLOBAL_PREFS_MASK mask;
            retval = ai.global_prefs.parse(xp, "", flag, mask);
            if (retval) return retval;
            continue;
        }
        if (!strcmp(tag, "host_info")) {
            ai.host_info.parse(mf);
            continue;
        }
        if (!strcmp(tag, "proxy_info")) {
            ai.proxy_info.parse(mf);
            continue;
        }
        if (xp.parse_int(tag, "major_version", ai.major_version)) continue;
        if (xp.parse_int(tag, "minor_version", ai.minor_version)) continue;
        if (xp.parse_int(tag, "release", ai.release)) continue;
        if (xp.parse_int(tag, "app_version", ai.app_version)) continue;
        if (xp.parse_str(tag, "app_name", ai.app_name, sizeof(ai.app_name))) continue;
        if (xp.parse_str(tag, "symstore", ai.symstore, sizeof(ai.symstore))) continue;
        if (xp.parse_str(tag, "acct_mgr_url", ai.acct_mgr_url, sizeof(ai.acct_mgr_url))) continue;
        if (xp.parse_str(tag, "user_name", ai.user_name, sizeof(ai.user_name))) continue;
        if (xp.parse_str(tag, "team_name", ai.team_name, sizeof(ai.team_name))) continue;
        if (xp.parse_str(tag, "project_dir", ai.project_dir, sizeof(ai.project_dir))) continue;
        if (xp.parse_str(tag, "boinc_dir", ai.boinc_dir, sizeof(ai.boinc_dir))) continue;
        if (xp.parse_str(tag, "authenticator", ai.authenticator, sizeof(ai.authenticator))) continue;
        if (xp.parse_str(tag, "wu_name", ai.wu_name, sizeof(ai.wu_name))) continue;
#ifdef _WIN32
        if (xp.parse_str(tag, "comm_obj_name", ai.shmem_seg_name, sizeof(ai.shmem_seg_name))) continue;
#else
        if (xp.parse_int(tag, "shm_key", ai.shmem_seg_name)) continue;
#endif
        if (xp.parse_int(tag, "slot", ai.slot)) continue;
        if (xp.parse_double(tag, "user_total_credit", ai.user_total_credit)) continue;
        if (xp.parse_double(tag, "user_expavg_credit", ai.user_expavg_credit)) continue;
        if (xp.parse_double(tag, "host_total_credit", ai.host_total_credit)) continue;
        if (xp.parse_double(tag, "host_expavg_credit", ai.host_expavg_credit)) continue;
        if (xp.parse_double(tag, "resource_share_fraction", ai.resource_share_fraction)) continue;
        if (xp.parse_double(tag, "rsc_fpops_est", ai.rsc_fpops_est)) continue;
        if (xp.parse_double(tag, "rsc_fpops_bound", ai.rsc_fpops_bound)) continue;
        if (xp.parse_double(tag, "rsc_memory_bound", ai.rsc_memory_bound)) continue;
        if (xp.parse_double(tag, "rsc_disk_bound", ai.rsc_disk_bound)) continue;
        if (xp.parse_double(tag, "wu_cpu_time", ai.wu_cpu_time)) continue;
        if (xp.parse_double(tag, "checkpoint_period", ai.checkpoint_period)) continue;
        if (xp.parse_double(tag, "fraction_done_update_period", ai.fraction_done_update_period)) continue;
        if (xp.parse_double(tag, "fraction_done_start", ai.fraction_done_start)) continue;
        if (xp.parse_double(tag, "fraction_done_end", ai.fraction_done_end)) continue;
        fprintf(stderr, "parse_init_data_file: unrecognized %s\n", tag);
        xp.skip_unexpected(tag);
    }
    fprintf(stderr, "parse_init_data_file: no end tag\n");
    return ERR_XML_PARSE;
}

APP_CLIENT_SHM::APP_CLIENT_SHM() {
    shm = 0;
}
bool MSG_CHANNEL::get_msg(char *msg) {
    if (!buf[0]) return false;
    strlcpy(msg, buf+1, MSG_CHANNEL_SIZE-1);
    buf[0] = 0;
    return true;
}

bool MSG_CHANNEL::has_msg() {
    if (buf[0]) return true;
    return false;
}

bool MSG_CHANNEL::send_msg(const char *msg) {
    if (buf[0]) return false;
    strlcpy(buf+1, msg, MSG_CHANNEL_SIZE-1);
    buf[0] = 1;
    return true;
}

void MSG_CHANNEL::send_msg_overwrite(const char* msg) {
    strlcpy(buf+1, msg, MSG_CHANNEL_SIZE-1);
    buf[0] = 1;
}

int APP_CLIENT_SHM::decode_graphics_msg(char* msg, GRAPHICS_MSG& m) {
    int i;

    parse_str(msg, "<window_station>", m.window_station, sizeof(m.window_station));
    parse_str(msg, "<desktop>", m.desktop, sizeof(m.desktop));
    parse_str(msg, "<display>", m.display, sizeof(m.display));

    m.mode = 0;
    for (i=0; i<NGRAPHICS_MSGS; i++) {
        if (match_tag(msg, xml_graphics_modes[i])) {
            m.mode = i;
        }
    }
    return 0;
}

void APP_CLIENT_SHM::reset_msgs() {
    memset(shm, 0, sizeof(SHARED_MEM));
}

// resolve "symbolic link"
//
int boinc_resolve_filename(const char *virtual_name, char *physical_name, int len) {
    FILE *fp;
    char buf[512], *p;

    if (!virtual_name) return ERR_NULL;
    strlcpy(physical_name, virtual_name, len);

    // Open the link file and read the first line
    //
    fp = boinc_fopen(virtual_name, "r");
    if (!fp) return ERR_FOPEN;

    // must initialize buf since fgets() on an empty file won't do anything
    //
    buf[0] = 0;
    p =fgets(buf, 512, fp);
    fclose(fp);

    // If it's the <soft_link> XML tag, return its value,
    // otherwise, return the original file name
    //
    if (p) parse_str(buf, "<soft_link>", physical_name, len);
    return 0;
}


// same, std::string version
//
int boinc_resolve_filename_s(const char *virtual_name, string& physical_name) {
    char buf[512], *p;
    if (!virtual_name) return ERR_NULL;
    physical_name = virtual_name;
    FILE *fp = boinc_fopen(virtual_name, "r");
    if (!fp) return ERR_FOPEN;
    buf[0] = 0;
    p = fgets(buf, 512, fp);
    fclose(fp);
    if (p) parse_str(buf, "<soft_link>", physical_name);
    return 0;
}

void url_to_project_dir(char* url, char* dir) {
    char buf[256];
    escape_project_url(url, buf);
    sprintf(dir, "%s/%s", PROJECT_DIR, buf);
}

const char *BOINC_RCSID_3add42d20e = "$Id: app_ipc.C 12925 2007-06-13 15:55:14Z davea $";
