//  BMP
//  Copyright (C) 2003-2007 http://beep-media-player.org 
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <time.h>
#include <errno.h>

#include <glib/gstdio.h>
#include <glibmm.h>
#include <glibmm/i18n.h>
#include <gtkmm.h>
#include <libglademm.h>

#include <cstring>
#include <iostream>
#include <sstream>
#include <string>

#include <boost/algorithm/string.hpp>
#include <boost/format.hpp>
#include <boost/shared_ptr.hpp>

#include "main.hh"
#include "network.hh"
#include "paths.hh"
#include "stock.hh"
#include "debug.hh"
#include "util.hh"
#include "util-file.hh"
#include "ui-tools.hh"
#include "uri++.hh"
#include "bmp/library-ops.hh"
#include "dialog-add-podcast.hh"
#include "dialog-simple-progress.hh"
#include "ui-part-podcasts.hh"
#include "x_play.hh"
using namespace Glib;
using namespace Gtk;
using namespace std;
using namespace Bmp::PodcastBackend;
using Bmp::URI;

#define POD_ACTION_ADD_PODCAST      "pod-action-add-podcast"
#define POD_ACTION_DEL_PODCAST      "pod-action-del-podcast"
#define POD_ACTION_UPD_PODCAST      "pod-action-upd-podcast"
#define POD_ACTION_COPY_RSS_LINK    "pod-action-copy-rss-link"
#define POD_ACTION_COPY_WEB_LINK    "pod-action-copy-web-link"
#define POD_ACTION_EXPORT_FEEDLIST  "pod-action-export-feedlist"

namespace
{
  const char * ui_string_casts_popup =
  "<ui>"
  ""
  "<menubar name='popup-podcasts-casts'>"
  ""
  "   <menu action='dummy' name='menu-podcasts-casts'>"
  "     <menuitem action='" POD_ACTION_UPD_PODCAST "'/>"
  "     <menuitem action='" POD_ACTION_DEL_PODCAST "'/>"
  "       <separator name='podcasts-sep2'/>"
  "     <menuitem action='" POD_ACTION_COPY_RSS_LINK "'/>"
  "     <menuitem action='" POD_ACTION_COPY_WEB_LINK "'/>"
  "   </menu>"
  ""
  "</menubar>"
  ""
  "</ui>";

  const char *ui_string_podcasts =
  "<ui>"
  ""
  "<menubar name='MenuBarMain'>"
  "   <menu action='MenuUiPartPodcasts'>"
  "     <menuitem action='" POD_ACTION_ADD_PODCAST "'/>"
  "     <separator/>"
  "     <menuitem action='" POD_ACTION_EXPORT_FEEDLIST "'/>"
  "   </menu>"
  "</menubar>"
  ""
  "</ui>";

  const char * error_messages[] =  
  {
    "A Network Error during loading of the RSS feed: %s",
    "The feed XML (RSS feed/Podcast XML) could not be parsed: %s",
    "The specified Feed URL is invalid: %s",
    "Encountered an invalid URI during parsing: %s",
  };

  ustring
  get_timestr_from_time_t (time_t atime)
  {
    struct tm atm;
    localtime_r (&atime, &atm);

    char bdate[64];
    strftime (bdate, 64, "%d/%m/%Y", &atm);

    char btime[64];

    strftime (btime, 64, "%H:%M", &atm);

    static boost::format
      date_f ("<b>%s</b> <small>%s</small>");

    return ustring ((date_f % bdate % btime).str());
  }

  ustring
  get_timestr_from_time_t_item (time_t atime)
  {
    struct tm atm;
    localtime_r (&atime, &atm);

    char bdate[64];
    strftime (bdate, 64, "%d/%m/%Y", &atm);

    char btime[64];

    strftime (btime, 64, "%H:%M", &atm);

    static boost::format
      date_f ("<b>%s</b> (%s)");

    return ustring ((date_f % bdate % btime).str());
  }


  bool
  video_type (std::string const& type)
  {
    bool is_video (type.substr (0, 6) == std::string("video/"));
    return is_video;
  }

  static boost::format
    podcast_title_f ("%s\n<small>%s</small>");
}

namespace Bmp
{
  namespace UiPart
  {
    guint
    Podcasts::add_ui ()
    {
      if( !Network::check_connected() )
        return 0;
      else
        return m_ui_manager->add_ui_from_string (ui_string_podcasts);
    };

    void
    Podcasts::load_cast_image (Podcast const& cast, TreeIter & iter)
    {
      if( !cast.image_url.empty() )
      {
        try{
          ::Cairo::RefPtr< ::Cairo::ImageSurface> image = Util::cairo_image_surface_from_pixbuf
            (Gdk::Pixbuf::create_from_file (cast_image_filename (cast))->scale_simple (48, 48, Gdk::INTERP_HYPER));
          Util::cairo_image_surface_border (image, 1.); 
          (*iter)[casts.image] = image; 
          }
        catch (...)
          {
            (*iter)[casts.image] = m_feed_default; 
          }
      }
      else
      {
        (*iter)[casts.image] = m_feed_default; 
      }
    }

    void
    Podcasts::upd_cast_internal (Podcast const& cast,
                                 TreeIter     & iter)
    {
      load_cast_image (cast, iter);
      ustring title;

      if( cast.items.empty() )
      {
        title = ((podcast_title_f % Markup::escape_text (cast.title).c_str()
                                  % _("(No Current Items)")).str());
      }
      else
      {
        title = ((podcast_title_f % Markup::escape_text (cast.title).c_str()
                                  % get_timestr_from_time_t (cast.most_recent_item)).str());

      }

      (*iter)[casts.uri] = cast.uri;
      (*iter)[casts.expired] = ::Cairo::RefPtr< ::Cairo::ImageSurface>(0); 
      (*iter)[casts.title] = title;
      (*iter)[casts.key] = cast.title.casefold_collate_key();
      (*iter)[casts.most_recent] = cast.most_recent_item;
      (*iter)[casts.searchtitle] = cast.title;

      m_view_casts->get_selection()->unselect_all ();
      m_view_casts->get_selection()->select (m_store_casts_filter->convert_child_iter_to_iter (iter));
    }

    void
    Podcasts::add_cast_internal (Podcast const& cast)
    {
      TreeIter iter (m_store_casts->append ());
      load_cast_image (cast, iter);

      ustring title;
      if( cast.items.empty() )
      {
        title = ((podcast_title_f % Markup::escape_text (cast.title).c_str()
                                  % _("(No Current Items)")).str());
      }
      else
      {
        title = ((podcast_title_f % Markup::escape_text (cast.title).c_str()
                                  % get_timestr_from_time_t (cast.most_recent_item)).str());
      }

      (*iter)[casts.uri] = cast.uri;
      (*iter)[casts.expired] = ::Cairo::RefPtr< ::Cairo::ImageSurface>(0); 
      (*iter)[casts.title] = title;
      (*iter)[casts.key] = cast.title.casefold_collate_key();
      (*iter)[casts.most_recent] = cast.most_recent_item;
      (*iter)[casts.searchtitle] = cast.title;
      (*iter)[casts.uid] = m_uid;

      m_context_map.insert (std::make_pair (m_uid, iter));
      m_uid++;
    }

    void
    Podcasts::on_remove_podcast ()
    {
      TreeIter iter (m_store_casts_filter->convert_iter_to_child_iter (m_view_casts->get_selection()->get_selected()));
      Podcast const& cast (m_cast_manager->podcast_fetch ((*iter)[casts.uri]));

      static boost::format
        question_f ("Are you sure you want to remove the podcast '%s'?");

      MessageDialog dialog ((question_f % cast.title).str(), false, MESSAGE_QUESTION, BUTTONS_YES_NO, true);
      dialog.set_title (_("Remove Podcast - BMP"));
      if( dialog.run () == GTK_RESPONSE_YES )
      {
        guint64 uid ((*iter)[casts.uid]);
        m_view_casts->get_selection()->unselect (iter);
        m_cast_manager->podcast_delete ((*iter)[casts.uri]);
        m_store_casts->erase (iter);
        m_context_map.erase (uid);
      }
    }

    void
    Podcasts::on_update_podcast ()
    {
      m_view_items->set_sensitive (0);
      while (gtk_events_pending())
        gtk_main_iteration();

      TreeIter iter = m_store_casts_filter->convert_iter_to_child_iter (m_view_casts->get_selection()->get_selected());

      try{
        ustring uri ((*iter)[casts.uri]);
        m_cast_manager->podcast_update (uri);
        upd_cast_internal (m_cast_manager->podcast_fetch (uri), iter);
        }
      catch (PodcastBackend::Exception & cxe)
        {
          boost::format error_f (error_messages[0]);
          MessageDialog dialog ((error_f % cxe.what()).str(), false, MESSAGE_ERROR, BUTTONS_OK, true);
          dialog.set_title (_("Podcast Error - BMP"));
          dialog.run ();
        }

      m_view_items->set_sensitive (1);
      while (gtk_events_pending())
        gtk_main_iteration();
    }

    void
    Podcasts::on_new_podcast ()
    {
      DialogAddPodcast * dialog (DialogAddPodcast::create());
      ustring uri;
      int response = dialog->run (uri);
      delete dialog;

      if( response == RESPONSE_OK )
      {
        m_ref_xml->get_widget ("podcasts-vbox")->set_sensitive (false);
        try{
            Bmp::URI u (uri);
            if( u.get_protocol() == Bmp::URI::PROTOCOL_ITPC )
            {
              u.set_protocol (Bmp::URI::PROTOCOL_HTTP);
              uri = ustring (u);
            }
            m_cast_manager->podcast_cache (uri);
            add_cast_internal (m_cast_manager->podcast_fetch (uri));
          }
        catch (PodcastBackend::Exception & cxe)
          {
            boost::format error_f (error_messages[0]);
            MessageDialog dialog ((error_f % cxe.what()).str(), false, MESSAGE_ERROR, BUTTONS_OK, true);
            dialog.set_title (_("Podcast Error - BMP"));
            dialog.run ();
          }
        m_ref_xml->get_widget ("podcasts-vbox")->set_sensitive (true);
      }
    }

    void
    Podcasts::on_copy_xml_link ()
    {
      TreeIter iter = m_view_casts->get_selection()->get_selected();
      Clipboard::get_for_display (Gdk::Display::get_default())->set_text ((*iter)[casts.uri]);
    }

    void
    Podcasts::on_copy_web_link ()
    {
      TreeIter iter = m_view_casts->get_selection()->get_selected();
      Podcast const& cast (m_cast_manager->podcast_fetch ((*iter)[casts.uri]));
      Clipboard::get_for_display (Gdk::Display::get_default())->set_text (cast.link);
    }

    void
    Podcasts::on_export_feedlist ()
    {
      FileChooserDialog dialog (_("Specfiy target Feedlist Filename - BMP"),
                                     FILE_CHOOSER_ACTION_SAVE);

      dialog.add_button (Stock::CANCEL, RESPONSE_CANCEL);
      dialog.add_button (Stock::SAVE, RESPONSE_OK);
      dialog.set_current_folder (mcs->key_get<string>("bmp", "file-chooser-path"));
      dialog.set_default_response (RESPONSE_CANCEL);
      dialog.set_default_size (750, 570);
      dialog.set_position (WIN_POS_CENTER);

      if( dialog.run() == RESPONSE_OK )
      {
        std::string filename = filename_from_uri (dialog.get_uri());
        if( filename.substr (filename.length() - 5, 5) != ".opml" )
        {
          filename += ".opml";
        }
        m_cast_manager->save_opml (filename);        
      }
    }

    ::Window
    Podcasts::set_da_window_id ()
    {
      return GDK_WINDOW_XID (m_video->get_window()->gobj());
    }

    Podcasts::Podcasts (RefPtr<Gnome::Glade::Xml> const& xml, RefPtr<UIManager> const&    ui_manager)
    : PlaybackSource      (_("Podcasts"), NONE, F_ALWAYS_IMAGE_FRAME)
    , Base                (xml, ui_manager)
    , m_uid               (1)
    {
      m_actions = ActionGroup::create ("ActionsPodcasts");
      m_actions->add (Action::create ("dummy", "dummy"));
      m_actions->add (Action::create ("MenuUiPartPodcasts", _("Po_dcasts")));

      m_actions->add (Action::create (POD_ACTION_ADD_PODCAST,
                                           StockID (BMP_STOCK_FEED_ADD),
                                           _("_Add Podcast")),
                      sigc::mem_fun (*this, &UiPart::Podcasts::on_new_podcast));

      m_actions->add (Action::create (POD_ACTION_DEL_PODCAST,
                                           StockID (BMP_STOCK_FEED_DELETE),
                                           _(": Remove")),
                      sigc::mem_fun (*this, &UiPart::Podcasts::on_remove_podcast));

      m_actions->add (Action::create (POD_ACTION_UPD_PODCAST,
                                           StockID (BMP_STOCK_FEED_UPDATE),
                                           _(": Update")),
                      sigc::mem_fun (*this, &UiPart::Podcasts::on_update_podcast));

      m_actions->add (Action::create (POD_ACTION_COPY_RSS_LINK,
                                           Stock::COPY,
                                           _("_Copy Feed XML Link")),
                                            AccelKey (""),
                      sigc::mem_fun (*this, &UiPart::Podcasts::on_copy_xml_link));

      m_actions->add (Action::create (POD_ACTION_COPY_WEB_LINK,
                                           Stock::COPY,
                                           _("_Copy Podcast Website URL")),
                                            AccelKey (""),
                      sigc::mem_fun (*this, &UiPart::Podcasts::on_copy_web_link));

      m_actions->add (Action::create (POD_ACTION_EXPORT_FEEDLIST,
                                           Stock::SAVE,
                                           _("Export Feedlist (OPML Format)")),
                                            AccelKey (""),
                      sigc::mem_fun (*this, &UiPart::Podcasts::on_export_feedlist));

      m_ui_manager->insert_action_group (m_actions);
      m_ui_manager->add_ui_from_string (ui_string_casts_popup);

      m_actions->get_action (POD_ACTION_COPY_RSS_LINK)->set_sensitive (false);
      m_actions->get_action (POD_ACTION_COPY_WEB_LINK)->set_sensitive (false);
      m_actions->get_action (POD_ACTION_DEL_PODCAST)->set_sensitive (false);
      m_actions->get_action (POD_ACTION_UPD_PODCAST)->set_sensitive (false);

      m_feed_default = Util::cairo_image_surface_from_pixbuf
        (Gdk::Pixbuf::create_from_file (build_filename (BMP_IMAGE_DIR_PODCAST, "feed-default.png"))->scale_simple (48, 48, Gdk::INTERP_HYPER));

      m_feed_expired = Util::cairo_image_surface_from_pixbuf
        (Gdk::Pixbuf::create_from_file (build_filename (BMP_IMAGE_DIR_PODCAST, "feed-expired.png")));

      m_ref_xml->get_widget ("podcasts-textview", m_textview);
      m_ref_xml->get_widget ("podcasts-view-casts", m_view_casts);
      m_ref_xml->get_widget ("podcasts-view-items", m_view_items);
      m_ref_xml->get_widget ("podcasts-da", m_video);

      m_video->modify_bg (STATE_NORMAL, Gdk::Color ("#000000"));
      m_video->modify_base (STATE_NORMAL, Gdk::Color ("#000000"));

      // In case it re-requests, i'm not entirely sure how this works really -- md
      
      if (::play->has_video())
      {
        ::play->signal_request_window_id ()
          .connect (sigc::mem_fun (*this, &Bmp::UiPart::Podcasts::set_da_window_id));
      }

      m_playing = Gdk::Pixbuf::create_from_file (build_filename (BMP_IMAGE_DIR_STOCK, "play.png"));
      m_cast_manager = boost::shared_ptr<PodcastManager> (new PodcastManager());

      // Casts

#define N_CAST_COLUMNS 3

      for (unsigned int n = 0; n < N_CAST_COLUMNS; ++n)
      {
        switch (n)
        {
          case 0:
          {
            CellRendererCairoSurface *cell = manage (new CellRendererCairoSurface());
            cell->property_xalign() = 1.0; 
            cell->property_yalign() = 0.0;
            cell->property_xpad() = 2;
            cell->property_ypad() = 4;
            TreeViewColumn *column = manage (new TreeViewColumn ("", *cell));
            column->add_attribute (*cell, "surface", casts.expired);
            column->set_resizable (false);
            column->set_expand (false);
            column->set_sizing (TREE_VIEW_COLUMN_FIXED);
            column->set_fixed_width (20);
            m_view_casts->append_column (*column);
            break;
          }

          case 1:
          {
            CellRendererCairoSurface *cell = manage (new CellRendererCairoSurface());
            cell->property_xalign() = 0.5;
            cell->property_yalign() = 0.0;
            cell->property_xpad() = 0;
            cell->property_ypad() = 4;
            TreeViewColumn *column = manage (new TreeViewColumn ("", *cell));
            column->add_attribute (*cell, "surface", casts.image);
            column->set_resizable (false);
            column->set_expand (false);
            column->set_sizing (TREE_VIEW_COLUMN_FIXED);
            column->set_fixed_width (60);
            m_view_casts->append_column (*column);
            break;
          }

          case 2:
          {
            CellRendererText *cell = manage (new CellRendererText());
            cell->property_xalign() = 0.0;
            cell->property_yalign() = 0.0;
            cell->property_xpad() = 0;
            cell->property_ypad() = 2;
            cell->property_ellipsize() = Pango::ELLIPSIZE_END;

            TreeViewColumn *column  = manage (new TreeViewColumn ("", *cell));
            column->add_attribute (*cell, "markup", casts.title);
            column->set_sizing (TREE_VIEW_COLUMN_FIXED);
            m_view_casts->append_column (*column);
            break;
          }
        }
      }

      m_store_casts = ListStore::create (casts);
      m_store_casts_filter = TreeModelFilter::create (m_store_casts);

      m_filter_entry = manage (new Sexy::IconEntry());
      Label * label = manage (new Label());
      label->set_markup_with_mnemonic (_("_Search:"));
      label->set_mnemonic_widget (*m_filter_entry);
      label->show();
      m_filter_entry->show();
      m_filter_entry->add_clear_button ();
      m_filter_entry->signal_changed().connect
        (sigc::mem_fun (m_store_casts_filter.operator->(), &TreeModelFilter::refilter));
      dynamic_cast<HBox*>(m_ref_xml->get_widget ("podcasts-hbox-filter"))->pack_start (*label, false, false);
      dynamic_cast<HBox*>(m_ref_xml->get_widget ("podcasts-hbox-filter"))->pack_start (*m_filter_entry, true, true);

      m_filter_entry->signal_changed().connect
        (sigc::mem_fun (m_store_casts_filter.operator->(), &TreeModelFilter::refilter));

      m_store_casts_filter->set_visible_func
        (sigc::mem_fun (*this, &Bmp::UiPart::Podcasts::podcast_visible_func));
      m_view_casts->set_model (m_store_casts_filter);

      m_view_casts->get_selection()->set_mode (SELECTION_SINGLE);
      m_view_casts->get_selection()->signal_changed().connect
        (sigc::mem_fun (*this, &Bmp::UiPart::Podcasts::on_casts_selection_changed));
      m_view_casts->signal_event().connect
        (sigc::mem_fun (*this, &Bmp::UiPart::Podcasts::on_casts_event));

      m_store_casts->set_default_sort_func
        (sigc::mem_fun (*this, &Bmp::UiPart::Podcasts::casts_default_sort_func));
      m_store_casts->set_sort_column (-1, SORT_ASCENDING);

      // Items
      for (unsigned int n = 0; n < N_COLUMNS; ++n)
      {
        switch (n)
        {
          case COLUMN_PLAYING:
          {
            CellRendererPixbuf *cell = manage (new CellRendererPixbuf());
            TreeViewColumn *column = manage (new TreeViewColumn ()); 

            cell->property_xalign() = 0.5;
            cell->property_xpad() = 2;
            cell->property_ypad() = 2;

            column->pack_start (*cell, false);
            column->set_cell_data_func
              (*cell, sigc::bind (sigc::mem_fun (*this, &Bmp::UiPart::Podcasts::cell_data_func_items), COLUMN_PLAYING, CELL_PIXBUF));
            column->set_resizable (false);
            column->set_expand (false);
            column->set_sizing (TREE_VIEW_COLUMN_FIXED);
            column->set_fixed_width (24);
            m_view_items->append_column (*column);
            break;
          }

          case COLUMN_DOWNLOADED:
          {
            CellRendererToggle * cell = 0;
            TreeViewColumn *column = manage (new TreeViewColumn ());

            cell = manage (new CellRendererToggle());
            cell->property_xalign() = 0.5;
            cell->property_ypad() = 2;
            cell->signal_toggled ().connect
              (sigc::mem_fun (*this, &Bmp::UiPart::Podcasts::on_column_download_toggled));

            column->pack_start (*cell, false);
            column->set_cell_data_func
              (*cell, sigc::bind (sigc::mem_fun (*this, &Bmp::UiPart::Podcasts::cell_data_func_items), COLUMN_DOWNLOADED, CELL_TOGGLE));
            column->set_resizable (false);
            column->set_expand (false);

            Image * disk = manage (new Image (m_view_items->render_icon (StockID (BMP_STOCK_DISK), ICON_SIZE_MENU)));
            disk->show_all ();
            column->set_widget (*disk);            

            m_view_items->append_column (*column);
            break;
          }

          case COLUMN_DOWNLOADING:
          {
            CellRendererProgress * cell = 0;
            TreeViewColumn *column = manage (new TreeViewColumn ());

            cell = manage (new CellRendererProgress());
            cell->property_xalign() = 0.0;
            cell->property_ypad() = 2;
            column->pack_start (*cell, false);
            column->set_cell_data_func
              (*cell, sigc::bind (sigc::mem_fun (*this, &Bmp::UiPart::Podcasts::cell_data_func_items), COLUMN_DOWNLOADING, CELL_PROGRESS));
            column->set_resizable (false);
            column->set_expand (false);
            column->property_visible() = false;
            m_view_items->append_column (*column);
            break;
          }

          case COLUMN_NEW:
          {
            CellRendererPixbuf * cell = 0;
            TreeViewColumn *column = manage (new TreeViewColumn ());

            cell = manage (new CellRendererPixbuf());
            cell->property_ypad() = 2;

            column->pack_start (*cell, false);
            column->set_cell_data_func
              (*cell, sigc::bind (sigc::mem_fun (*this, &Bmp::UiPart::Podcasts::cell_data_func_items), COLUMN_NEW, CELL_PIXBUF));
            column->set_resizable (false);
            column->set_expand (false);
            column->set_sizing (TREE_VIEW_COLUMN_FIXED);
            column->set_fixed_width (24);
            m_view_items->append_column (*column);
            break;
          }

          case COLUMN_TITLE:
          {
            CellRendererText *cell = manage (new CellRendererText());
            TreeViewColumn *column  = manage (new TreeViewColumn (_("Title"), *cell));

            cell->property_xpad() = 2;
            cell->property_ypad() = 2;
            cell->property_ellipsize() = Pango::ELLIPSIZE_END;

            column->set_cell_data_func
              (*cell, sigc::bind (sigc::mem_fun (*this, &Bmp::UiPart::Podcasts::cell_data_func_items), COLUMN_TITLE, CELL_TEXT));
            column->set_resizable (true);
            column->set_expand (false);
            column->set_min_width (200);
            m_view_items->append_column (*column);
            break;
          }

          case COLUMN_DATE:
          {
            CellRendererText *cell = manage (new CellRendererText());
            TreeViewColumn *column  = manage (new TreeViewColumn (_("Date"), *cell));

            cell->property_xpad() = 2;
            cell->property_ypad() = 2;
            cell->property_family() = "Monospace";

            column->set_cell_data_func
              (*cell, sigc::bind (sigc::mem_fun (*this, &Bmp::UiPart::Podcasts::cell_data_func_items), COLUMN_DATE, CELL_TEXT));
            column->set_resizable (false);
            column->set_expand (false);
            m_view_items->append_column (*column);
            break;
          }
        }
      }

      m_store_items = ListStore::create (items);

      m_view_items->set_model (m_store_items);
      m_view_items->get_selection()->set_mode (SELECTION_SINGLE);
      m_view_items->get_selection()->signal_changed().connect
          (sigc::mem_fun (*this, &Bmp::UiPart::Podcasts::on_items_selection_changed));
      m_view_items->signal_row_activated().connect
          (sigc::mem_fun (*this, &Bmp::UiPart::Podcasts::activate_default));
      m_store_items->set_default_sort_func
          (sigc::mem_fun (*this, &Bmp::UiPart::Podcasts::items_default_sort_func));
      m_store_items->set_sort_column (-1, SORT_ASCENDING);

      PodcastList list;
      m_cast_manager->podcast_get_list (list);
      for (PodcastList::const_iterator c = list.begin() ; c != list.end () ; ++c)
      {
         try {
            add_cast_internal (m_cast_manager->podcast_fetch (*c));
            }
          catch (PodcastBackend::ParsingError& cxe)
            {
              g_warning ("%s: Parsing Error during reading of podcast: %s",
                G_STRLOC, cxe.what());
              break;
            }
          catch (PodcastBackend::InvalidUriError& cxe)
            {
              g_warning ("%s: URI Is Invalid: %s",
                G_STRLOC, cxe.what());
              break;
            }
      }

      signal_timeout().connect (sigc::mem_fun (*this, &Podcasts::check_casts), 3600U * 1000U  /* check every hour */);
    }

    bool
    Podcasts::podcast_visible_func (TreeIter const& iter)
    {
      ustring filter (ustring (m_filter_entry->get_text()).lowercase());
      if( filter.empty() )
        return true;
      else
        return (ustring ((*iter)[casts.searchtitle]).lowercase().find (filter) != ustring::npos);
    }

    Podcasts::~Podcasts ()
    { 
      for (DownloadMap::iterator i = mDownloads.begin(); i != mDownloads.end(); ++i)
      {
        i->second->mFileRequest.clear();
      }
    }

    bool
    Podcasts::on_casts_event (GdkEvent * ev)
    {
      if( ev->type == GDK_BUTTON_PRESS )
      {
        GdkEventButton * event = reinterpret_cast <GdkEventButton *> (ev);
        if( event->button == 3 )
        {
          TreeViewColumn      * column;
          TreePath              path;
          int                   cell_x,
                                cell_y;

          static boost::format
            action_update_f (_("Podcast '%s': Update"));

          static boost::format
            action_delete_f (_("Podcast '%s': Remove"));

          if( m_view_casts->get_path_at_pos (int (event->x), int (event->y), path, column, cell_x, cell_y) )
          {
            m_view_casts->get_selection()->select (path);
            TreeIter iter = m_store_casts->get_iter (path);
            Podcast const& cast = m_cast_manager->podcast_fetch ((*iter)[casts.uri]);

            m_ui_manager->get_action (("/popup-podcasts-casts/menu-podcasts-casts/" POD_ACTION_UPD_PODCAST))->property_label() =
              (action_update_f % cast.title).str();
            m_ui_manager->get_action (("/popup-podcasts-casts/menu-podcasts-casts/" POD_ACTION_DEL_PODCAST))->property_label() =
              (action_delete_f % cast.title).str();

            bool sensitivity = (mDownloadUids.find (UID ((*iter)[casts.uid])) == mDownloadUids.end());

            m_ui_manager->get_action (("/popup-podcasts-casts/menu-podcasts-casts/" POD_ACTION_UPD_PODCAST))->set_sensitive (sensitivity);
            m_ui_manager->get_action (("/popup-podcasts-casts/menu-podcasts-casts/" POD_ACTION_DEL_PODCAST))->set_sensitive (sensitivity);

            Menu * menu = dynamic_cast <Menu *>(Util::get_popup (m_ui_manager, "/popup-podcasts-casts/menu-podcasts-casts"));
            if (menu) 
            {
              menu->popup (event->button, event->time);
            }
            return true;
          }
          else
          {
            m_ui_manager->get_action (("/popup-podcasts-casts/menu-podcasts-casts/" POD_ACTION_UPD_PODCAST))->property_label() =
              (action_update_f % (_("(none selected)"))).str();
            m_ui_manager->get_action (("/popup-podcasts-casts/menu-podcasts-casts/" POD_ACTION_DEL_PODCAST))->property_label() =
              (action_delete_f % (_("(none selected)"))).str();
          }
        }
      }
      return false;
    }

    void
    Podcasts::on_column_download_toggled (ustring const& tree_path)
    {
      TreeIter iter = m_view_items->get_model()->get_iter (tree_path);
      TreeIter iter_c = m_view_casts->get_selection ()->get_selected();

      PodcastItem item ((*iter)[items.item]);
      const Podcast & cast = m_cast_manager->podcast_fetch ((*iter_c)[casts.uri]);

      Bmp::URI u (item.enclosure_url);
      if( u.get_protocol() != Bmp::URI::PROTOCOL_HTTP )
      {
        MessageDialog dialog (
          (boost::format (_("This item can not be downloaded (non-HTTP URI): %s"))
          % item.enclosure_url.c_str()).str(),
          false, MESSAGE_ERROR, BUTTONS_OK, true);
        dialog.set_title (_("Podcast Download Error - BMP"));
        dialog.run ();
        return;
      }

      DownloadKey key (((*iter_c)[casts.uid]), item.uid_value);

      if( !mDownloads.empty() )
      {
        if( mDownloads.find (key) != mDownloads.end() ) 
        {
          MessageDialog dialog (_("Are you sure you want to abort the download of this item?"),
                                false, MESSAGE_QUESTION, BUTTONS_YES_NO, true);
          dialog.set_title (_("Podcast Download: Abort - BMP"));
          if (dialog.run() == GTK_RESPONSE_YES) 
          {
            download_cancel (key);
            return; 
          }
        }
      } 

      if( m_current_iter && (m_current_iter.get() == iter) )
      {
        MessageDialog dialog (_("You have to stop playback of this item before you can download it."),
                              false, MESSAGE_INFO, BUTTONS_OK, true);
        dialog.set_title (_("Podcast Download: Info - BMP"));
        dialog.run ();
        return;
      }

      if( item.downloaded )
      {
        MessageDialog dialog (_("Are you sure you want to re-download this item?"),
                              false, MESSAGE_QUESTION, BUTTONS_YES_NO, true);
        dialog.set_title (_("Podcast Download - BMP"));
        if( dialog.run () == GTK_RESPONSE_NO )
        {
          return;
        }
      }

      std::string path = cast_item_path (cast);
      std::string file = cast_item_file (cast, item);

      if( g_mkdir_with_parents (path.c_str(), S_IRWXU) == 0 )
      {
        if( file_test (file, FILE_TEST_EXISTS) )
        {
          g_unlink (file.c_str());
        }

        DownloadContextRefp dlrefp = DownloadContextRefp (new DownloadContext());

        dlrefp->mFileRequest = Soup::RequestFile::create (item.enclosure_url, file); 
        dlrefp->mFileRequest->file_progress().connect
          (sigc::bind (sigc::mem_fun (*this, &Bmp::UiPart::Podcasts::download_progress), key));
        dlrefp->mFileRequest->file_done().connect
          (sigc::bind (sigc::mem_fun (*this, &Bmp::UiPart::Podcasts::download_done), key));
        dlrefp->mFileRequest->file_aborted().connect
          (sigc::bind (sigc::mem_fun (*this, &Bmp::UiPart::Podcasts::download_aborted), key));

        dlrefp->mCastUri = cast.uri;
        dlrefp->mDownloadItem = item;
        dlrefp->mDownloadProgress = .0;

        mDownloadUids.insert (key.first);
        mDownloads.insert (std::make_pair (key, dlrefp));
        
        m_view_items->get_column (COLUMN_DOWNLOADING)->property_visible() = true;
        dlrefp->mFileRequest->run ();
        on_items_selection_changed ();
      }
    }

    void
    Podcasts::download_progress (double progress, DownloadKey key)
    {
      DownloadContextRefp & ctx = mDownloads.find (key)->second;
      ctx->mDownloadProgress = progress;
      m_view_items->queue_draw ();
    }

    void
    Podcasts::download_done (std::string const& filename, DownloadKey key)
    {
      DownloadContextRefp & ctx = mDownloads.find (key)->second;
      PodcastItem & item = ctx->mDownloadItem;

      item.downloaded = 1;
      item.filename = filename; 

      m_cast_manager->podcast_item_change (ctx->mCastUri, item.uid_value, item);
      m_cast_manager->podcast_overlay_save (m_cast_manager->podcast_fetch (ctx->mCastUri));

      if( m_view_casts->get_selection()->count_selected_rows() )
      {
            TreeIter iter_c = m_view_casts->get_selection ()->get_selected();
            UID uid ((*iter_c)[casts.uid]);

            if( key.first == uid )
            {
                  TreeNodeChildren nodes = m_store_items->children();
                  for (TreeNodeChildren::iterator i = nodes.begin(); i != nodes.end(); ++i)
                  {
                    PodcastItem const& node_item = (*i)[items.item];
                    if (node_item.uid_value == item.uid_value)
                    {
                      (*i)[items.item] = item; 
                      m_view_items->queue_draw ();
                      break;
                    }
                  }
            }
      }
      download_remove (key);
    }

    void
    Podcasts::download_aborted (DownloadKey key)
    {
      download_remove (key); 
    }

    void
    Podcasts::download_cancel (DownloadKey key)
    {
      DownloadContextRefp & ctx = mDownloads.find (key)->second;
      ctx->mFileRequest->cancel();
      download_remove (key);
    }

    void
    Podcasts::download_remove (DownloadKey key)
    {
      Glib::Mutex::Lock L (mDownloadEraseLock);

      UidSet::iterator i1= mDownloadUids.find (key.first);
      if (i1 != mDownloadUids.end())
        mDownloadUids.erase (i1);

      DownloadMap::iterator i2 = mDownloads.find (key);
      if (i2 != mDownloads.end())
        mDownloads.erase (i2);

      on_items_selection_changed ();
      
      if (mDownloads.empty()) 
      {
        m_view_items->get_column (COLUMN_DOWNLOADING)->property_visible() = false;
      }
    }

    bool
    Podcasts::sel_uid_is_current_uid ()
    {
      if( m_current_uid && m_view_casts->get_selection()->count_selected_rows() )
      {
        TreeIter iter_c (m_view_casts->get_selection ()->get_selected());
        return bool (guint64 ((*iter_c)[casts.uid]) == m_current_uid);
      }

      return false;
    }

    int
    Podcasts::casts_default_sort_func (TreeIter const& iter_a, TreeIter const& iter_b)
    {
      time_t t1 = (*iter_a)[casts.most_recent];
      time_t t2 = (*iter_b)[casts.most_recent];

      if (t1 > t2)
        return -1;

      if (t1 < t2)
        return  1;

      return 0;
    }

    int
    Podcasts::items_default_sort_func (TreeIter const& iter_a, TreeIter const& iter_b)
    {
      PodcastItem const& item_a ((*iter_a)[items.item]);
      PodcastItem const& item_b ((*iter_b)[items.item]);

      time_t t_a (item_a.pub_date_unix);
      time_t t_b (item_b.pub_date_unix);

      if( t_a > t_b )
        return -1;

      if( t_a < t_b )
        return  1;

      return 0;
    }

    // Items
   
    bool
    Podcasts::check_casts ()
    {
      m_view_casts->set_sensitive (0);
      m_view_items->set_sensitive (0);
      while (gtk_events_pending())
        gtk_main_iteration();

      time_t t = time (NULL);

      TreeNodeChildren const& children (m_store_casts->children());
      for (TreeNodeChildren::iterator i = children.begin(); i != children.end(); ++i)
      {
        Podcast const& cast = m_cast_manager->podcast_fetch ((*i)[casts.uri]);

        // if it's being downloaded, skip it in this cycle
        if (mDownloadUids.find ((*i)[casts.uid]) != mDownloadUids.end())
          continue;

        if( cast.ttl != 0 )
        {
          if( t > (cast.last_poll_time + (cast.ttl*60)) )
          {
            time_t previous_time = cast.most_recent_item;

            try{
              TreeIter iter = *i;
              ustring uri = cast.uri;
              m_cast_manager->podcast_update (uri);
              upd_cast_internal (m_cast_manager->podcast_fetch (uri), iter);
              }
            catch (PodcastBackend::Exception & cxe)
              {
                boost::format error_f (error_messages[0]);
                MessageDialog dialog ((error_f % cxe.what()).str(), false, MESSAGE_ERROR, BUTTONS_OK, true);
                dialog.set_title (_("Podcast Error - BMP"));
                dialog.run ();
              }

            if (previous_time < cast.most_recent_item)
              (*i)[casts.expired] = m_feed_expired;
            else
              (*i)[casts.expired] = ::Cairo::RefPtr< ::Cairo::ImageSurface>(0);
          }
        }
      }

      m_view_items->set_sensitive (1);
      m_view_casts->set_sensitive (1);
      while (gtk_events_pending())
        gtk_main_iteration();

      return false;
    }

    void
    Podcasts::cell_data_func_items (CellRenderer * basecell, TreeIter const& iter, ItemColumn column, ItemColumnType type)
    {
      CellRendererText        * _cell_tx = 0;
      CellRendererPixbuf      * _cell_pb = 0;
      CellRendererToggle      * _cell_tg = 0;
      CellRendererProgress    * _cell_pg = 0;

      switch (type)
      {
        case CELL_TEXT:
          _cell_tx = dynamic_cast<CellRendererText *>(basecell);
          break;

        case CELL_PIXBUF:
          _cell_pb = dynamic_cast<CellRendererPixbuf *>(basecell);
          break;

        case CELL_TOGGLE:
          _cell_tg = dynamic_cast<CellRendererToggle *>(basecell);
          break;

        case CELL_PROGRESS:
          _cell_pg = dynamic_cast<CellRendererProgress *>(basecell);
          break;
      }
  
      PodcastItem const& item ((*iter)[items.item]);

      switch (column)
      {
        case COLUMN_PLAYING:
          if( sel_uid_is_current_uid() && m_current_iter && (m_store_items->get_path (m_current_iter.get()) == m_store_items->get_path (iter)) )
            _cell_pb->property_pixbuf() = m_playing;
          else
            _cell_pb->property_pixbuf() = RefPixbuf(0);
          break;

        case COLUMN_DOWNLOADED:
          if( item.downloaded )
            _cell_tg->property_active() = true; 
          else
            _cell_tg->property_active() = false; 
          break;

        case COLUMN_DOWNLOADING:
          _cell_pg->property_visible() = false; 
          if( !mDownloads.empty() ) 
          {
            if( m_view_casts->get_selection()->count_selected_rows() )
            {
              TreeIter iter_c = m_view_casts->get_selection ()->get_selected();
              UID uid ((*iter_c)[casts.uid]);
            
              DownloadKey key (uid, item.uid_value);
              DownloadMap::iterator i = mDownloads.find (key); 
              if (i != mDownloads.end())
              {
                _cell_pg->property_visible() = true; 
                _cell_pg->property_value() = int (i->second->mDownloadProgress * 100);
              }
            }
          }
          break;

        case COLUMN_NEW:
          if( !item.played )
            _cell_pb->property_pixbuf() = m_view_items->render_icon (StockID (BMP_STOCK_NEW), ICON_SIZE_MENU);
          else
            _cell_pb->property_pixbuf() = RefPixbuf(0);
          break;

        case COLUMN_TITLE:
          _cell_tx->property_markup() = Markup::escape_text (item.title);
          break;

        case COLUMN_DATE:
          _cell_tx->property_markup() = get_timestr_from_time_t_item (item.pub_date_unix);
          break;
      }
    }

    ///////////////////////////////////////////////////////////////////////////////////////

    void
    Podcasts::has_next_prev (bool & next, bool & prev)
    {
      next = false;
      prev = false;

      if( sel_uid_is_current_uid() && m_current_iter )
      {
        TreeModel::Children const& children = m_store_items->children();

        unsigned int position;
        position = m_store_items->get_path (m_current_iter.get()).get_indices().data()[0];

        next = (position > 0);
        prev = (position+1 != children.size());

        if( next )
        {
          TreePath path = TreePath (TreePath::size_type (1), TreePath::value_type (position-1));
          TreeIter iter = m_store_items->get_iter (path);
          PodcastItem const& item ((*iter)[items.item]);
          if (!video_type (item.enclosure_type) || (video_type (item.enclosure_type) && item.downloaded)) 
          {
            next = true;
          }
          else
          {
            next = false;
          }
        }

        if( prev )
        {
          TreePath path = TreePath (TreePath::size_type (1), TreePath::value_type (position+1));
          TreeIter iter = m_store_items->get_iter (path);
          PodcastItem const& item ((*iter)[items.item]);
          if (!video_type (item.enclosure_type) || (video_type (item.enclosure_type) && item.downloaded)) 
          {
            prev = true;
          }
          else
          {
            prev = false;
          }
        }
      }
    }

    void
    Podcasts::current_check_played_and_caps ()
    {
      m_current_item.played = true;
      (*m_current_iter.get())[items.item] = m_current_item;
      m_cast_manager->podcast_item_change (m_current_cast.uri, m_current_item.uid_value, m_current_item);

      if( m_current_item.downloaded )
      {
        m_caps = Caps (m_caps | PlaybackSource::CAN_SEEK);
        m_caps = Caps (m_caps | PlaybackSource::CAN_PAUSE);
      }
      else
      {
        m_caps = Caps (m_caps | PlaybackSource::CAN_SEEK);
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_PAUSE);
      }

      bool next, prev;
      has_next_prev (next, prev);

      if( sel_uid_is_current_uid() && m_current_iter )
        {
          if( next )
            m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_NEXT);
          else
            m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);

          if( prev )
            m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_PREV);
          else
            m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);
        }
      else
        {
          m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);
          m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);
        }

      m_caps = Caps (m_caps | PlaybackSource::CAN_PROVIDE_METADATA);
    }

    void
    Podcasts::go_next_prev (Direction direction)
    {
      TreePath path;
      TreeIter iter;
      unsigned int position = 0;

      switch (direction)
      {
        case NEXT:
          if( m_current_iter )
          {
            path = m_store_items->get_path (m_current_iter.get());
            iter = m_current_iter.get();
            position = path.get_indices().data()[0];

            m_current_iter.reset();
            m_store_items->row_changed (path, iter);

            path.prev ();
            m_current_iter = m_store_items->get_iter (path);
            m_store_items->row_changed (path, m_current_iter.get());
            assign_current_item (m_current_iter.get());
          }
          break;

        case PREV:
          if( m_current_iter )
          {
            path = m_store_items->get_path (m_current_iter.get());
            iter = m_current_iter.get();
            position = path.get_indices().data()[0];

            m_current_iter.reset();
            m_store_items->row_changed (path, iter);

            path.next ();
            m_current_iter = m_store_items->get_iter (path);
            m_store_items->row_changed (path, m_current_iter.get());
            assign_current_item (m_current_iter.get());
          }
          break;

        default: break;
      }

    }

    bool
    Podcasts::go_next ()
    {
      go_next_prev  (NEXT);
      return true;
    }

    bool
    Podcasts::go_prev ()
    {
      go_next_prev  (PREV);
      return true;
    }

    void
    Podcasts::stop ()
    {
      clear_current_item ();
      m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);
      m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);
      m_caps = Caps (m_caps & ~PlaybackSource::CAN_SEEK);
      m_caps = Caps (m_caps & ~PlaybackSource::CAN_PAUSE);
      m_caps = Caps (m_caps & ~PlaybackSource::CAN_PROVIDE_METADATA);
      s_caps_.emit (m_caps);
    }

    ustring
    Podcasts::get_uri ()
    {
      if( m_current_item.downloaded )
        return filename_to_uri (m_current_item.filename);
      else
        return m_current_item.enclosure_url;
    }

    ustring
    Podcasts::get_type ()
    {
      return m_current_item.enclosure_type;
    }

    void
    Podcasts::clear_current_item ()
    {
      if( sel_uid_is_current_uid() && m_current_iter )
      {
        TreePath path = m_store_items->get_path(m_current_iter.get());
        TreeIter iter = m_current_iter.get();
        m_current_iter.reset();
        m_store_items->row_changed (path, iter);
      }
      else
      {
        m_current_iter.reset();
      }

      m_current_uid = 0;
      m_current_item_uid = ustring ();

      m_view_items->queue_draw ();
    }

    void
    Podcasts::assign_current_item (TreeIter const& iter)
    {
      clear_current_item ();

      TreeIter const& iter_c (m_view_casts->get_selection ()->get_selected());
      m_current_cast = m_cast_manager->podcast_fetch ((*iter_c)[casts.uri]);
      m_current_item = ((*iter)[items.item]);

      if( !m_current_cast.image_url.empty() )
      {
        try{
            m_current_image = Gdk::Pixbuf::create_from_file (cast_image_filename (m_current_cast)); 
          }
        catch (...)
          {
            m_current_image = Glib::RefPtr<Gdk::Pixbuf>(0);
          }
      } 
      else
      {
        m_current_image = Glib::RefPtr<Gdk::Pixbuf>(0);
      }

      m_current_iter = iter;
      m_current_uid = (*iter_c)[casts.uid];
      m_current_item_uid = m_current_item.uid_value;
    }
    
    void
    Podcasts::send_metadata ()
    {
      TrackMetadata metadata;
      metadata.artist = m_current_cast.title; 
      metadata.album = m_current_cast.description;
      metadata.title = m_current_item.title;
      metadata.duration = m_current_item.enclosure_length / 1000;
      metadata.image = m_current_image;
      m_metadata = metadata;
      s_track_metadata_.emit (m_metadata);
    }

    void
    Podcasts::buffering_done () 
    {
      send_metadata ();
    }

    void
    Podcasts::play ()
    {
      
      if( !m_current_iter )
      {
        TreeIter iter = m_view_items->get_selection()->get_selected();
        if( iter )
        {
          assign_current_item (iter);
        }
        else
          throw UnableToInitiatePlaybackError();
          
        PodcastItem const& item ((*iter)[items.item]);
        if(video_type(item.enclosure_type))
          ::play->set_window_id (GDK_WINDOW_XID (m_video->get_window()->gobj()));
      }
      else
      {
        if(video_type(m_current_item.enclosure_type))
          ::play->set_window_id (GDK_WINDOW_XID (m_video->get_window()->gobj()));
      }
    }

    void
    Podcasts::play_post ()
    {
      TreeIter iter = m_store_casts_filter->convert_iter_to_child_iter (m_view_casts->get_selection()->get_selected());
      (*iter)[casts.expired] = BmpCairoIS(0);

      current_check_played_and_caps ();
      go_next_prev (CHECKONLY);

      if (m_current_item.downloaded)
        send_metadata (); // we won't get called on buffering_done() so now is the time
    }

    void
    Podcasts::next_post ()
    {
      current_check_played_and_caps ();
      send_metadata ();
    }

    void
    Podcasts::prev_post ()
    {
      current_check_played_and_caps ();
      send_metadata ();
    }

    void
    Podcasts::restore_context ()
    {
      if( m_current_uid )
      {
        m_view_casts->get_selection()->select (m_context_map.find(m_current_uid)->second);
      }
    }

    ///////////////////////////////////////////////////////////////////////////////////////

    void
    Podcasts::activate_default (TreePath const& path, TreeViewColumn* column)
    {
      assign_current_item (m_store_items->get_iter (path));
      s_playback_request_.emit (); 
    }

    void
    Podcasts::on_casts_selection_changed()
    {
      m_actions->get_action (POD_ACTION_COPY_WEB_LINK)->set_sensitive (false);
      m_actions->get_action (POD_ACTION_COPY_RSS_LINK)->set_sensitive (false);

      bool selected = (m_view_casts->get_selection ()->count_selected_rows() > 0);

      m_view_items->queue_draw();
      m_current_iter.reset();
      m_store_items->clear ();
      m_textview->get_buffer()->set_text("");

      if( selected )
      {
        TreeIter const& iter (m_store_casts_filter->convert_iter_to_child_iter (m_view_casts->get_selection()->get_selected()));
        Podcast const& cast (m_cast_manager->podcast_fetch ((*iter)[casts.uri]));
        for (PodcastItemsMap::const_iterator i = cast.items.begin(); i != cast.items.end() ; ++i)
        {
          TreeIter iter (m_store_items->append ());
          (*iter)[items.item] = i->second; 

          if( sel_uid_is_current_uid() && (i->second.uid_value == m_current_item_uid) )
          {
            m_current_iter = iter;
          }
        }
          
        m_actions->get_action (POD_ACTION_COPY_WEB_LINK)->set_sensitive (!cast.link.empty());
        m_actions->get_action (POD_ACTION_COPY_RSS_LINK)->set_sensitive (true);
      }
      else
      {
        m_store_items->clear ();
      }

      m_actions->get_action (POD_ACTION_DEL_PODCAST)->set_sensitive (selected);
      m_actions->get_action (POD_ACTION_UPD_PODCAST)->set_sensitive (selected);
      m_view_items->columns_autosize();
      go_next_prev (CHECKONLY);
    }

    void
    Podcasts::on_items_selection_changed()
    {
      if( (m_view_items->get_selection ()->count_selected_rows() > 0) )
      {
        TreeIter iter = m_view_items->get_selection()->get_selected();
        PodcastItem item ((*iter)[items.item]);

        TreeIter iter_c = m_view_casts->get_selection ()->get_selected();
        UID uid ((*iter_c)[casts.uid]);
    
        DownloadKey key (uid, item.uid_value); 

        m_textview->get_buffer()->set_text (Util::sanitize_html (item.description));

        if( ((item.downloaded) || (!item.downloaded && (item.enclosure_type.substr(0,6) != "video/"))) && (mDownloads.find (key) == mDownloads.end()) )
        {
          m_caps = Caps (m_caps | PlaybackSource::CAN_PLAY);
        }
        else
        {
          m_caps = Caps (m_caps & ~PlaybackSource::CAN_PLAY);
        }
      }
      else
      {
        m_textview->get_buffer()->set_text("");
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_PLAY);
      }
      s_caps_.emit (m_caps);
    }
  } // namespace UiPart
} // namespace Bmp
