//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2007 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifndef BMP_UI_PART_LIBRARY_HH
#define BMP_UI_PART_LIBRARY_HH

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#ifdef HAVE_TR1
#include<tr1/unordered_map>
#include<string>
using namespace std;
using namespace std::tr1;
#endif

#include <list>
#include <set>
#include <string>
#include <vector>
#include <queue>
#include <boost/optional.hpp>
#include <boost/shared_ptr.hpp>
#include <gdkmm/pixbuf.h>
#include <glibmm/ustring.h>
#include <gtkmm.h>
#include <libglademm/xml.h>
#include <libsexymm/icon-entry.h>

#include "mcs/mcs.h"
#include "bmp/base-types.hh"
#include "bmp/library-types.hh"
#include "widgets/cell-renderer-cairo-surface.hh"

// BMP Audio
#include "audio/audio.hh"

// BMP Misc
#ifdef HAVE_HAL
#  include "hal.hh"
#endif //HAVE_HAL

#include "dialog-simple-progress.hh"
#include "util.hh"
#include "util-file.hh"

#include "playbacksource.hh"
#include "ui-part-base.hh"

using namespace Bmp::DB;

namespace Bmp
{              
  typedef sigc::signal<void>  SVoidT;
  typedef std::map<UID, RowV::size_type> UidIndexMap;
  typedef UidIndexMap::value_type UidIndexPair;
  typedef UidIndexMap::iterator UidIndexMapIter;

  struct SharedState  
  {
    public:

      void
      lock ()
      {
        mLock1.lock (); 
      }

      void
      unlock ()
      {
        mLock1.unlock ();
      }

      void
      clear ()
      {
        mLock1.lock ();
        r_album_artist_j.clear ();
        r_album_j.clear ();
        r_tracks.clear ();
        r_tracks_index.clear ();
        i_attrs.clear ();
        mLock1.unlock ();
      }

      void
      clear_unlocked ()
      {
        r_album_artist_j.clear ();
        r_album_j.clear ();
        r_tracks.clear ();
        r_tracks_index.clear ();
        i_attrs.clear ();
      }

      bool
      new_artist ()
      {
        bool n = m_new_artist;
        m_new_artist = 0;
        return n;
      }

      RowV                  r_album_artist_j;
      RowV                  r_album_j;
      TrackV                r_tracks;
      UidIndexMap           r_tracks_index;
      Bmp::DB::AttributeVV  i_attrs;
      bool                  m_new_artist;

      bool                  m_querying;
      Glib::Cond            m_reset_cond;
      Glib::Mutex           m_reset;

      SharedState () : m_new_artist (0), m_querying (0) {}

      Glib::Mutex mLock1;
  };

  class ViewArtists
    : public Gtk::TreeView
  {
    public:

      SharedState * m_state;

    public:

      typedef sigc::signal <void, UidSet const&, Glib::ustring const&>  SignalArtistSelected;
      typedef sigc::signal <void, uint64_t>                             SignalAllArtistsSelected;

    protected:

      virtual void on_row_activated (Gtk::TreeModel::Path const&, Gtk::TreeViewColumn*);

    private:

#ifdef HAVE_TR1
      typedef std::tr1::unordered_map<std::string, Gtk::TreeIter> IndexMapT;
#else //!HAVE_TR1
      typedef std::map<std::string, Gtk::TreeIter> IndexMapT;
#endif //HAVE_TR1

      typedef IndexMapT::iterator IndexMapIter;
      typedef IndexMapT::value_type IndexMapPair;

      enum NodeType
      {
        NODE_BRANCH,
        NODE_LEAF
      };

      class ArtistCR
        : public Gtk::TreeModel::ColumnRecord
      {
        public:

          Gtk::TreeModelColumn<Glib::ustring>   value;
          Gtk::TreeModelColumn<AlbumArtist>     artist;
          Gtk::TreeModelColumn<UID>             uid;
          Gtk::TreeModelColumn<NodeType>        type;

          ArtistCR ()
          {
            add (value);
            add (artist);
            add (uid);
            add (type);
          }
      };

      ArtistCR                      m_artist_cr;
      Glib::RefPtr<Gtk::TreeStore>  m_store;
      IndexMapT                     m_index_map;

      void  put_artist (Bmp::AlbumArtist const& artist);
      void  cell_data_func (Gtk::CellRenderer * basecell, Gtk::TreeIter const& iter, int cell);
      int   default_sort_func (Gtk::TreeIter const& iter_a, Gtk::TreeIter const& iter_b);

      void  on_selection_changed ();
      bool  m_have_various_artist;

      UidIterMap  m_UidIterMap;
      
      SignalArtistSelected        signal_artist_selected_;
      SignalAllArtistsSelected    signal_artist_all_selected_;
      SVoidT                      signal_artist_cleared_;
      SVoidT                      signal_activated_;
    
      sigc::connection conn_changed;

      Glib::Mutex m_update_mutex;

    public:

      ViewArtists (BaseObjectType*                        obj,
                  Glib::RefPtr<Gnome::Glade::Xml> const& xml);
      virtual ~ViewArtists () {}

      void  clear ();
      void  display ();

      SignalArtistSelected&
      signal_artist_selected ()
      {
        return signal_artist_selected_;
      }

      SignalAllArtistsSelected&
      signal_artist_all_selected ()
      {
        return signal_artist_all_selected_;
      }

      SVoidT&
      signal_artist_cleared ()
      {
        return signal_artist_cleared_;
      }

      SVoidT&
      signal_activated ()
      {
        return signal_activated_;
      }
  };

  //////////////////////////////////////////////////////////////////

  class ViewAlbums
    : public Gtk::TreeView
  {
    private:
      Glib::RefPtr<Gnome::Glade::Xml> m_ref_xml;

    public:
      SharedState * m_state;

    private:
      void  album_cell_data_func (Gtk::CellRenderer * basecell, Gtk::TreeIter const& iter);
      bool  slot_select (Glib::RefPtr <Gtk::TreeModel> const& model, Gtk::TreeModel::Path const& path, bool was_selected);
      void  insert (Album const& album);
      void  get_cover (UidList const& uid_list);

      class AlbumCR
        : public Gtk::TreeModel::ColumnRecord
      {
        public:

          Gtk::TreeModelColumn<BmpCairoIS>      cover;
          Gtk::TreeModelColumn<Album>           album;
          Gtk::TreeModelColumn<UID>             uid;
          Gtk::TreeModelColumn<Glib::ustring>   searchKey;
          AlbumCR ()
          {
            add (cover);
            add (album);
            add (uid);
            add (searchKey);
          }
      };

      AlbumCR                         m_album_cr;
      Glib::RefPtr<Gtk::ListStore>    m_store;
      BmpCairoIS                      m_cover;

      UidIterMap  m_uid_album_map;
      UidList     m_selected_uids;
      bool        m_all_artists;

    protected:

      virtual void on_row_activated (Gtk::TreeModel::Path const&, Gtk::TreeViewColumn*);

    private:

      SVoidT signal_changed_;
      SVoidT signal_activated_;

    public:

      SVoidT&
      signal_changed()
      {
        return signal_changed_;
      }

      SVoidT&
      signal_activated()
      {
        return signal_activated_;
      }

    public:

      ViewAlbums (BaseObjectType                       * obj,
                 Glib::RefPtr<Gnome::Glade::Xml> const& xml);
      virtual ~ViewAlbums () {}

      // API
      UidList
      get_albums ();

      void
      clear ();

      void
      change ();

      // Handlers
      void    on_artist_selected      (UidSet const& x, Glib::ustring const& name);
      void    on_artist_cleared       ();
      void    on_artist_all_selected  (uint64_t n_artists);
  };

  //////////////////////////////////////////////////////////////////

  class ViewPlaylist
    : public Gtk::TreeView
  {
    public:

      SharedState * m_state;

    private:

      Glib::RefPtr <Gdk::Pixbuf> m_pb_playing;
      Glib::RefPtr <Gnome::Glade::Xml> m_ref_xml;

      // History
      class History
      {
        public:

          History () : m_position (m_queue.begin()) {}
          ~History () {}

          bool
          boundary ()
          {
            return ( (std::distance (m_queue.begin(), m_position) == 0) ||
                     (std::distance (m_position, (m_queue.end()-1)) == 0));
          }
    
          bool         
          have_prev ()
          {
            return ((m_queue.size() > 1) && (std::distance (m_queue.begin(), m_position) > 0));
          }

          bool
          have_next ()
          {
            return ((m_queue.size() > 1) && (std::distance (m_position, (m_queue.end()-1)) > 0));
          }

          UID
          get_prev ()
          {
            --m_position;
            return UID (*m_position);
          }

          UID
          get_next ()
          {
            ++m_position;
            return UID (*m_position);
          }

          void
          append (UID uid)
          {
            if (m_queue.size() == 1024)      
            {
              Position p (m_queue.begin());
              m_queue.erase (p);
            }
            m_queue.push_back (uid);
            m_position = m_queue.end() - 1; 
          }

          void
          prepend (UID uid)
          {
            if (m_queue.size() == 1024)      
            {
              Position p (m_queue.end() - 1);
              m_queue.erase (p);
            }
            m_queue.push_front (uid);
            m_position = m_queue.begin ();
          }

          void
          erase  (UID uid)
          {
            Position p = m_queue.begin();
            for ( ; p != m_queue.end() ; )
            {
              if (*p == uid)
                p = m_queue.erase (p);
              else
                ++p;
            }
          } 

          bool
          has_uid (UID uid)
          {
            for (Position p = m_queue.begin() ; p != m_queue.end() ; ++p)
            {
              if (*p == uid)
                return true;
            }
            return false;
          }

          void
          clear ()
          {
            m_queue.clear ();
            m_position = m_queue.begin();
          }

          void
          set (UID uid)
          {
            if (!boundary())
              m_queue.erase (m_position+1, m_queue.end());
            append (uid);
          }

          void
          rewind ()
          {
            m_position = m_queue.begin();
          }

          bool
          get (UID & uid)
          {
            if (m_queue.size())
            {
              uid = *m_position;
              return true;
            }
            return false;
          }

          bool
          empty ()
          {
            return m_queue.empty();
          }

          void
          print (const char* func)
          {
            printf ("Func: %s\nSize: %llu\n", func, uint64_t (m_queue.size()));
            for (Queue::const_iterator i = m_queue.begin(); i != m_queue.end(); ++i)
            {
              if (i == m_position)
              {
                printf ("%llu\t*\n", *i);
              }
              else
              {
                printf ("%llu\t-\n", *i);
              }
            }
            printf ("\n");
          }

        private:

          typedef std::deque <UID>  Queue;
          typedef Queue::iterator   Position;

          Queue     m_queue; 
          Position  m_position;
      };

      UidIterSetMap                   m_UidIterMap;
      UidIterMap                      m_localUidIterMap;
      UID                             m_localUid;
      History                         m_history;

      // Main Datastore
      class ColumnsTrack
        : public Gtk::TreeModel::ColumnRecord
      {
        public:
        
          ColumnTrack   track;
          ColumnUID     uid;
          ColumnUID     localUid;
          ColumnString  searchKey;
          ColumnBool    playTrack;
          ColumnBool    present;

          ColumnsTrack ()
          {
            add (track);
            add (uid);
            add (localUid);
            add (searchKey);
            add (playTrack);
            add (present);
          }
      };

      ColumnsTrack
      m_track_cr;

      Glib::RefPtr <Gtk::ListStore>
      m_store;

      boost::optional <Gtk::TreeIter>
      m_current_iter;

      uint64_t
      m_bmpx_track_id;

      void  put_track_at_iter (Track const& track, Gtk::TreeIter & iter);
      void  put_track_at_iter (Track const& track); // this will append the track

      void  cell_data_func (Gtk::CellRenderer * basecell, Gtk::TreeIter const& iter, int column, int renderer);
      void  cell_play_track_toggled (Glib::ustring const& path);
      bool  slot_select (Glib::RefPtr <Gtk::TreeModel> const& model, Gtk::TreePath const& path, bool was_selected);

      void  assign_current_iter (Gtk::TreeIter const& iter);
      void  clear_current_iter ();

      void  on_tracks_retag ();
      void  on_playlist_export ();
      void  on_delete_files ();

      Glib::RefPtr <Gtk::UIManager>   m_ui_manager;
      Glib::RefPtr <Gtk::ActionGroup> m_actions;

#ifdef HAVE_HAL
      void  hal_volume_del (HAL::Volume const& volume);
      void  hal_volume_add (HAL::Volume const& volume);
      void  unselect_missing ();
#endif //HAVE_HAL

      void  on_library_track_modified (Track const& track);

    protected:

      virtual bool on_event (GdkEvent * event);
      virtual bool on_motion_notify_event (GdkEventMotion * event);
      virtual void on_row_activated (Gtk::TreeModel::Path const&, Gtk::TreeViewColumn*);

    public:

      Glib::ustring get_uri ();
      Track get_track ();

      bool  notify_next ();
      bool  notify_prev ();
      bool  notify_play ();
      void  notify_stop ();
      bool  check_play ();

      void  has_next_prev (bool & next, bool & prev);
      bool  has_playing ();
      void  set_first_iter ();

      void  append_album (uint64_t bmpx_album_id);
      void  display ();
      void  clear ();

      bool  m_playing;

      typedef sigc::signal <void> SignalRecheckCaps;
      typedef sigc::signal <void> SignalUpdated;
      typedef sigc::signal <void> Signal;
      
    private:

      Signal            signal_activated_;
      SignalRecheckCaps signal_recheck_caps_;
      SignalUpdated     signal_updated_;

    public:

      Signal&
      signal_activated()
      {
        return signal_activated_;
      }

      SignalRecheckCaps&
      signal_recheck_caps ()
      {
        return signal_recheck_caps_;
      }

      SignalUpdated&
      signal_updated ()
      {
        return signal_updated_;
      }

      ViewPlaylist (BaseObjectType                       * obj,
                    Glib::RefPtr<Gnome::Glade::Xml> const& xml);
      void  set_ui_manager (Glib::RefPtr<Gtk::UIManager> const& ui_manager);
      virtual ~ViewPlaylist ();
  };

  //////////////////////////////////////////////////////////////////

  class LibrarySearch
  {
    public:

      SharedState * m_state;
    
      Sexy::IconEntry&
      entry ()
      {
        return *m_search_entry;
      }

    private:

      Glib::RefPtr <Gnome::Glade::Xml>  m_ref_xml;
      Glib::RefPtr <Gtk::UIManager>     m_ui_manager;
      Glib::RefPtr <Gtk::ActionGroup>   m_actions;

      Sexy::IconEntry       * m_search_entry;
      int                     m_current_attribute;

      sigc::connection        sort_entry_changed_timeout_conn;
      Glib::Timer             entry_changed_timer;

      bool  on_search_entry_timeout ();
      void  on_search_entry_changed ();
      void  on_search_entry_icon_pressed (Sexy::IconEntryPosition pos, int button);

      void  select_attribute (); 
      void  menu_position (int & x, int & y, bool & push_in);

      void  run_query_thread ();
      bool  run_query_result_idle ();

      bool m_dissociated;
      bool m_change_block;

      Gdk::Color m_saved_color;

    private:

      SVoidT  signal_search_result_;
      SVoidT  signal_search_restore_;
      SVoidT  signal_search_clear_;

    public:

      SVoidT&
      signal_search_result ()
      {
        return signal_search_result_;
      }

      SVoidT&
      signal_search_clear ()
      {
        return signal_search_clear_;
      }

      SVoidT&
      signal_search_restore ()
      {
        return signal_search_restore_;
      }

      LibrarySearch (Glib::RefPtr <Gnome::Glade::Xml> const& xml,
                     Glib::RefPtr<Gtk::UIManager> const& ui_manager);
      virtual ~LibrarySearch ();

      void  run_query ();
      void  dissociate ();
      void  associate ();
   };

  //////////////////////////////////////////////////////////////////

  namespace UiPart
  {
    class Library
      : public  PlaybackSource,
        public  Base
    {
      public:

        Library (Glib::RefPtr<Gnome::Glade::Xml> const& xml, Glib::RefPtr<Gtk::UIManager> ui_manager);
        virtual ~Library ();

        void  update ();

      private:

        ViewArtists     * m_view_artists;
        ViewAlbums      * m_view_albums;
        ViewPlaylist    * m_view_playlist;

        LibrarySearch   * m_library_search;
        Gtk::Notebook   * m_library_notebook;

        void  on_playlist_selection_changed ();
        void  on_playlist_activated ();
        void  on_playlist_rows_reordered ();

        void  on_albums_activated ();
        void  on_albums_changed ();

        void  on_artists_activated ();

        void  on_search_result ();
        void  on_search_clear ();
        void  on_search_restore ();

        void  query_playlist_caps ();
        void  send_metadata ();

        bool  m_playing;
        bool  m_queue_playback;
        bool  m_queue_block_changes;

        void  on_shuffle_repeat_toggled (MCS_CB_DEFAULT_SIGNATURE);
        void  on_library_backend_modified ();

        Glib::RefPtr <Gtk::ActionGroup> m_actions;
    
        SharedState m_state;

      protected:

        virtual guint
        add_ui ();

        virtual Glib::ustring
        get_uri ();

        virtual Glib::ustring
        get_type () { return Glib::ustring(); }

        virtual bool
        go_next ();

        virtual bool
        go_prev ();

        virtual void
        stop ();

        virtual void
        play ();

        virtual void
        play_post ();

        virtual void
        restore_context () {}

        virtual GHashTable*
        get_metadata ();
    };
  }
}
#endif //!BMP_UI_PART_LIBRARY_HH
