//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2007 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifndef BMP_PLAYBACK_SOURCE_HH
#define BMP_PLAYBACK_SOURCE_HH

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <glib/ghash.h>
#include <glibmm/ustring.h>
#include <sigc++/signal.h>

#include "bmp/base-types.hh"
#include "bmp/ui-types.hh"

namespace Bmp
{
    class PlaybackSource
    {
      public:

#include "exception.hh"

      EXCEPTION(UnableToInitiatePlaybackError)
  
        enum Flags
        {
          F_NONE                = 0,
          F_HANDLE_STREAMINFO   = 1 << 0,
          F_PHONY_NEXT          = 1 << 1,
          F_PHONY_PREV          = 1 << 2,
          F_ALWAYS_IMAGE_FRAME  = 1 << 3,
          F_HANDLE_LASTFM       = 1 << 4,
          F_USES_REPEAT         = 1 << 5,
          F_USES_SHUFFLE        = 1 << 6,
        };

        enum Caps
        {
          NONE                  = 0,
          CAN_GO_NEXT           = 1 << 0,
          CAN_GO_PREV           = 1 << 1,
          CAN_PAUSE             = 1 << 2,
          CAN_PLAY              = 1 << 3,
          CAN_SEEK              = 1 << 4,
          CAN_RESTORE_CONTEXT   = 1 << 5, 
          CAN_PROVIDE_METADATA  = 1 << 6,
          CAN_BOOKMARK          = 1 << 7,
          PROVIDES_TIMING       = 1 << 8,
        };

        typedef sigc::signal<void, Caps>                          SignalCaps;
        typedef sigc::signal<void, Flags>                         SignalFlags;

        typedef sigc::signal<void, TrackMetadata const&>          SignalTrackMetadata;

        typedef sigc::signal<void>                                SignalSegment;

        typedef sigc::signal<void, unsigned int>                  SignalPositionChanged;

        typedef sigc::signal<void, Glib::ustring const&>          SignalMessage;
        typedef sigc::signal<void>                                SignalMessageClear;

        typedef sigc::signal<void>                                SignalPlaybackRequest;
        typedef sigc::signal<void>                                SignalStopRequest;
        typedef sigc::signal<void>                                SignalNextRequest;

        PlaybackSource (Glib::ustring const& name, Caps caps = NONE, Flags flags = F_NONE)

          : m_caps        (caps),
            m_flags       (flags),
            m_name        (name) {}

        virtual ~PlaybackSource () {}

        SignalPositionChanged &
        signal_position_changed ()
        {
          return s_position_changed_;
        }

        SignalCaps &
        signal_caps ()
        {
          return s_caps_;
        }

        SignalFlags &
        signal_flags ()
        {
          return s_flags_;
        }

        SignalTrackMetadata &
        signal_track_metadata ()
        {
          return s_track_metadata_;
        }

        SignalPlaybackRequest &
        signal_playback_request ()
        {
          return s_playback_request_;
        }

        SignalStopRequest &
        signal_stop_request ()
        {
          return s_stop_request_;
        }

        SignalNextRequest &
        signal_next_request ()
        {
          return s_next_request_;
        }

        SignalSegment &
        signal_segment ()
        {
          return s_segment_;
        }

        SignalMessage &
        signal_message ()
        {
          return s_message_;
        }

        SignalMessageClear &
        signal_message_clear ()
        {
          return s_message_clear_;
        }

        virtual Glib::ustring
        get_uri () = 0; 
    
        virtual Glib::ustring
        get_type () = 0;

        virtual bool
        go_next () = 0;

        virtual bool
        go_prev () = 0;

        virtual void
        stop () = 0;
  
        virtual void
        play () = 0;


        virtual void
        play_post () = 0;

        virtual void
        next_post () {}; 

        virtual void
        prev_post () {};


        virtual void
        restore_context () = 0;

        virtual void
        skipped () {}


        virtual void
        send_caps ()
        {
          s_caps_.emit (m_caps);
        }

        virtual void
        send_flags ()
        {
          s_flags_.emit (m_flags);
        }


        virtual GHashTable *
        get_metadata () { return NULL; } 


        virtual void
        segment () {}; //FIXME: Should be pure virtual

        Glib::ustring
        get_name ()
        {
          return m_name;
        }

        virtual void
        buffering_done () {};

      protected:

        SignalCaps            s_caps_;
        SignalFlags           s_flags_;

        SignalTrackMetadata   s_track_metadata_;

        SignalSegment         s_segment_;
        SignalPositionChanged s_position_changed_;

        SignalMessage         s_message_;
        SignalMessageClear    s_message_clear_;

        SignalPlaybackRequest s_playback_request_;
        SignalStopRequest     s_stop_request_;
        SignalNextRequest     s_next_request_;

        Caps                  m_caps;
        Flags                 m_flags;
        Glib::ustring         m_name;

    }; // end class PlaybackSource 
} // end namespace Bmp 
  
#endif
