//  "BMP" Copyright (C) 2003-2007 BMP development
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include "bmp/bmp.hh"

#include "bookmark-manager.hh"
#include "paths.hh"
#include "xml.hh"

#include <boost/format.hpp>
#include <boost/algorithm/string.hpp>

using namespace Glib;

namespace Bmp
{
    BookmarkManager::BookmarkManager ()
    {
    }

    BookmarkManager::~BookmarkManager ()
    {
      save_bookmarks ();
    }

    uint64_t
    BookmarkManager::add_bookmark (Bookmark const& bookmark)
    {
      m_bookmarks.push_back (bookmark); 
      m_indexmap.insert (std::make_pair (bookmark.m_url, m_bookmarks.size()-1));
      return uint64_t (m_bookmarks.size());
    }

    void
    BookmarkManager::save_bookmarks () // private
    {
      static boost::format bookmark_f ("%s\n%d\n%s\n");
      std::string data;

      for (BookmarkV::const_iterator i = m_bookmarks.begin(); i != m_bookmarks.end(); ++i)
      {
        data += (bookmark_f % i->m_title % i->m_id % i->m_url).str();
      }
      
      g_file_set_contents ( build_filename (BMP_PATH_DATA_DIR, "bookmarks.txt").c_str(),
                            data.c_str(),
                            (gssize)(strlen (data.c_str())),
                            NULL);


      xmlDocPtr   doc = xmlNewDoc   (BAD_CAST "1.0");
      xmlNodePtr  bkm = xmlNewNode  (0, BAD_CAST "bookmarks");
      xmlNsPtr    bmp = xmlNewNs    (bkm, BAD_CAST XML_NS_BMP, BAD_CAST "bmp");
      xmlNodePtr  q   = xmlNewChild (bkm, bmp, BAD_CAST "bookmark-list", 0);

      xmlDocSetRootElement (doc, bkm);
      xmlSetProp (bkm, BAD_CAST "version", BAD_CAST "1.0");

      for (BookmarkV::const_iterator i = m_bookmarks.begin(); i != m_bookmarks.end(); ++i)
      {
        static boost::format int_f ("%d");

        xmlNodePtr n = xmlNewChild (q, bmp, BAD_CAST "bookmark", 0);

        xmlSetProp (n, BAD_CAST "title",
                       BAD_CAST i->m_title.c_str());

        xmlSetProp (n, BAD_CAST "href",
                       BAD_CAST i->m_url.c_str());

        xmlSetProp (n, BAD_CAST "id",
                       BAD_CAST (int_f % i->m_id).str().c_str());
      }

      xmlCreateIntSubset (doc,
                          BAD_CAST "bkm",
                          NULL,
                          BAD_CAST "http://bmpx.beep-media-player.org/dtd/bookmarks-1.dtd" );

      std::string filename = build_filename (BMP_PATH_DATA_DIR, "bookmarks.xml");
      xmlThrDefIndentTreeOutput (1);
      xmlKeepBlanksDefault (0);
      g_message ("Saving Bookmarks to '%s'.", filename.c_str());
      int ret = xmlSaveFormatFileEnc (filename.c_str(), doc, "utf-8", 1);
      g_message ("Saved %d characters.", ret);
      xmlFreeDoc (doc);
    }

    void
    BookmarkManager::load_bookmarks () // private
    {
      std::string filename = build_filename (BMP_PATH_DATA_DIR, "bookmarks.txt");
      if (file_test (filename, FILE_TEST_EXISTS))
      {
        g_message ("%s: Loading Bookmarks: %s", G_STRLOC, filename.c_str());

            using boost::algorithm::split;
            using boost::algorithm::is_any_of;

            std::string data = file_get_contents (filename);
    
            StrV resp;
            split (resp, data, boost::algorithm::is_any_of("\n"));
      
            for (StrV::const_iterator i = resp.begin(); i < resp.end(); )
            {
              std::string title = *i;
              if (++i == resp.end())
                break;

              int id = int (g_ascii_strtoull ((*i).c_str(), NULL, 10));
              if (++i == resp.end())
                break;
      
              std::string url = *i;
              if (++i == resp.end())
                break;

              Bookmark b (title, id, url);
              signal_bookmark_loaded_.emit (b);              
            }
      }
    }

} // end namespace Bmp 
