//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#include <iostream>
#include <string>
#include <vector>
#include <cstdlib>
#include <cstring>

#include <boost/format.hpp>
#include <boost/regex.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/algorithm/string/regex.hpp>

#include <glibmm.h>
#include <glibmm/i18n.h>

#include <src/util_string.hh>
#include <src/util_file.hh>
#include <src/uri++.hh>

#include <src/vfs.hh>

namespace
{
  typedef std::map < std::string, std::string > StringMap;

  void
  parse_to_map (StringMap& map, std::string const& buffer)
  {
      using boost::algorithm::split;
      using boost::algorithm::split_regex;
      using boost::algorithm::is_any_of;

      std::vector<std::string> lines;
      split_regex (lines, buffer, boost::regex ("\\\r?\\\n"));

      for (unsigned int n = 0; n < lines.size(); ++n)
        {
          char **line = g_strsplit (lines[n].c_str(), "=", 2);
          if (line[0] && line[1] && strlen(line[0]) && strlen(line[1]))
            {
              map[line[0]] = line[1];
            }
          g_strfreev (line);
        }
  }
}

namespace Bmp
{
  namespace VFS
  {
      class PluginContainerPLS
        : public Bmp::VFS::PluginContainerBase
      {
        public:

          virtual bool
          can_process (Glib::ustring const& uri)
          {
            Bmp::URI u (uri);
            return Bmp::Util::str_has_suffix_nocase (u.path.c_str(), "pls"); 
          }

          virtual bool 
          handle_read	(Handle & handle,
                       VUri & list)
          {

            if (!handle.get_buffer()) 
                throw ProcessingError ("Empty Buffer"); 

            std::string buffer = (const char*)(handle.get_buffer());

            StringMap map;
            parse_to_map (map, buffer);

            if (map.empty()) 
                throw ProcessingError (_("Malformed PLS: No key/value pairs (Invalid Playlist?)")); 

            if (map.find("numberofentries") == map.end())
                throw ProcessingError (_("Malformed PLS: No 'numberofentries' key")); 

            unsigned int n = strtol (map.find("numberofentries")->second.c_str(), NULL, 10);
            static boost::format File ("File%d");
            for (unsigned int a = 1; a <= n ; ++a)
              {
                StringMap::iterator const& i = map.find ((File % a).str());
                if (i != map.end())
                {
                  list.push_back (i->second);
                }
              }

            if (list.empty()) 
                throw ProcessingError (_("Malformed PLS: No 'File' keys found")); 

            if (list.size() != n)
                g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, _("Malformed PLS: Number of Files doesn't match specified number"));

            return true;
          }

          virtual bool
          can_write ()
          {
            return true;
          }

          virtual bool
          handle_write (Handle & handle, VUri const& uri_list)
          {
            std::stringstream out;
            int n = 1;

            out << "[playlist]\n";
            out << "numberofentries=" << uri_list.size () << "\n";

            for (VUri::const_iterator iter = uri_list.begin (); iter != uri_list.end (); ++iter)
            {
              out << "File" << n << "=" << (*iter) << "\n";
              out << "Title" << n << "=\n";
              out << "Length" << n << "=-1\n";
              n++;
            }

            out << "Version=2\n";
            
            handle.set_buffer((const unsigned char *)strdup(out.str().c_str()), strlen(out.str().c_str())+1);
            return true;
          }

          virtual Bmp::VFS::ExportData
          get_export_data ()
          {
            static Bmp::VFS::ExportData export_data ("PLS Playlist", "pls"); 
            return export_data;
          }
      };
  }
}
  
extern "C" Bmp::VFS::PluginContainerBase* plugin_create (void)
{
  return new Bmp::VFS::PluginContainerPLS;
}

extern "C" void plugin_delete (Bmp::VFS::PluginContainerPLS* p)
{
  delete p;
}
