// Neon++ Library, (C) 2006 M. Derezynski
// A C++ wrapper library for neon <http://www.webdav.org/neon>, (C) Joe Orton

#include <iostream>
#include <fstream>
#include <exception>
#include <string>

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/fcntl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <stdio.h>
#include <unistd.h>
#include <cstring>

#include <glib/gstdio.h>

#include <ne_session.h>
#include <ne_request.h>
#include <ne_uri.h>

#include "session.hh" 
#include "request.hh" 

#include "uri++.hh"
#include "network.hh"

namespace Neon
{
  Request::~Request ()
  {
    if (m_request) ne_request_destroy (m_request);
    if (m_session) delete m_session;
  }

  Request::Request (std::string const&  hostname,
                    std::string const&  path,
                    Neon::HttpPort      port,
                    ResponseRead        read,
                    Method              method,
                    std::string const&  scheme,
                    bool                direct_file_output,
                    std::string const&  output_file)

  throw (Neon::Session::SessionCreateError, InvalidRequestError) 

            : m_o                   (0),
              m_request             (0),
              m_session             (0),
              m_dispatched          (false),
              m_read                (read),
              m_method              (method),
              m_check_server        (true),
              m_direct_file_output  (direct_file_output),
              m_output_file         (output_file)
  {
    m_session = new Neon::Session (hostname, scheme, port);
    ne_set_progress (m_session->session, Neon::Request::progress, this);
    m_request = ne_request_create (m_session->session, (m_method == Neon::Request::METHOD_GET) ? "GET" : "POST",
                                   path.c_str());

    if (!m_request)
      throw InvalidRequestError (ne_get_error (m_session->session));

    if (m_read == Neon::Request::RESPONSE_READ)
    {
      if (m_direct_file_output)
        ne_add_response_body_reader (m_request, ne_accept_2xx, read_block_file, this); 
      else
        ne_add_response_body_reader (m_request, ne_accept_2xx, read_block, &m_vec);
    }
  }

  void
  Request::set_check_server (bool check)
  {
    m_check_server = check;
  }
  
  void 
  Request::set_request_data (const char * data, size_t data_size)
  {
    ne_set_request_body_buffer (m_request, data, data_size); 
  }

  const ne_status*
  Request::get_status () const
  {
    return ne_get_status (m_request);
  }

  int
  Request::read_block (void * udata, const char * data, size_t len)
  {
    std::vector<unsigned char> *vec = (std::vector<unsigned char>*)(udata);
    for (unsigned int n = 0; n < len; ++n)
      vec->push_back ((unsigned char)data[n]);
    return 0;
  }

  int
  Request::read_block_file (void * udata, const char * data, size_t len)
  {
    Neon::Request * req = reinterpret_cast<Neon::Request *>(udata);
    req->m_o->write (data, len);
    return 0;
  }

  void
  Request::progress (void * udata, off_t cur, off_t siz)
  {
    double cur_d = cur;
    double siz_d = siz;

    Neon::Request * request = reinterpret_cast<Neon::Request *>(udata);
    request->s_request_progress_.emit (cur_d/siz_d);
  }

  void
  Request::clear ()
  {
    m_dispatched = false;
    m_vec.clear(); 
  }

#ifdef NEONPP_EXCEPTIONS_ENABLED
  void
#else
  int
#endif //NEONPP_EXCEPTIONS_ENABLED

  Request::dispatch ()
  {
    if (m_dispatched)
      return NE_OK;

    ne_uri test_uri;
    ne_fill_server_uri (m_session->session, &test_uri);
  
    if (m_check_server && !Bmp::Network::check_host (test_uri.host, test_uri.port))
      {

#ifdef NEONPP_EXCEPTIONS_ENABLED
        throw NeUnableToConnectError ();
#else
        return NE_CONNECT;
#endif //NEONPP_EXCEPTIONS_ENABLED

      }

    if (m_direct_file_output)
      {
        std::cerr  << " Opening Output File at '" << m_output_file << "'" << std::endl;
        m_o = new std::ofstream ();
        try {
            m_o->exceptions ( std::ios_base::failbit ); 
            m_o->open (m_output_file.c_str(), std::ios::binary );
          }
        catch (std::ios_base::failure& cxe)
          {
            std::cerr << " Exception trying to open the file occured: '" << cxe.what() <<  "'" << std::endl;
    
            if (m_o->bad())
              std::cerr << " File is 'bad' " << std::endl; 

            if (m_o->eof())
              std::cerr << " File is EOF " << std::endl; 

            if (m_o->fail())
              std::cerr << " File open has failed " << std::endl; 

            delete m_o;
            m_o = 0;
            m_dispatched = true;
#ifdef NEONPP_EXCEPTIONS_ENABLED
            throw NeGenericError (cxe.what());
#else
            return NE_ERROR;
#endif //NEONPP_EXCEPTIONS_ENABLED
          }
      }

    int result = ne_request_dispatch (m_request);
    int status = ne_get_status (m_request)->code;

    if ((status == 302) || (status == 301))
      {
        const char *redir = ne_get_response_header (m_request, "Location");
        //const char *cctrl = ne_get_response_header (m_request, "Cache-Control");
        //const char *exprd = ne_get_response_header (m_request, "Expires");

        m_o->close ();
        g_unlink (m_output_file.c_str());
        delete m_o; 
        m_o = 0;

        if (redir != NULL)
          {
            Glib::ustring redir_u (redir);

            ne_request_destroy (m_request);
            delete m_session;

            Bmp::URI uri (redir_u);

            m_session = new Neon::Session (uri.hostname, uri.scheme, uri.port);
            ne_set_progress (m_session->session, Neon::Request::progress, this);

            m_request = ne_request_create (m_session->session,
                            (m_method == Neon::Request::METHOD_GET) ? "GET" : "POST", uri.fullpath().c_str());

            if (!m_request)
              throw InvalidRequestError (ne_get_error (m_session->session));

            if (m_read == Neon::Request::RESPONSE_READ)
            {
              if (m_direct_file_output)
                ne_add_response_body_reader (m_request, ne_accept_2xx, read_block_file, this); 
              else
                ne_add_response_body_reader (m_request, ne_accept_2xx, read_block, &m_vec);
            }
            dispatch ();
          }
      }

    if (m_o)
      {
        m_o->flush ();
        m_o->close ();  
        delete m_o;
        m_o = 0;
      }

    m_dispatched = true;

    if (!m_direct_file_output) m_vec.push_back ((unsigned char)'\0');

#ifdef NEONPP_EXCEPTIONS_ENABLED

    switch (result)
    {
      case NE_OK:
        return;

      case NE_AUTH:
        throw NeAuthError (ne_get_error (ne_get_session (m_request)));

      case NE_PROXYAUTH:
        throw NeProxyAuthError (ne_get_error (ne_get_session (m_request)));

      case NE_CONNECT:  
        throw NeConnectError (ne_get_error (ne_get_session (m_request)));

      case NE_ERROR:
        throw NeGenericError (ne_get_error (ne_get_session (m_request)));
    }

#else
    return result;
#endif

  }

  void
  Request::add_request_header (std::string const& name, std::string const& value)
  {
    ne_add_request_header (m_request, name.c_str(), value.c_str());
  }

  void
  Request::add_request_header (Neon::Request::HeaderType type)
  {
    switch (type)
    {
      case HEADER_WWWFORM:
        ne_add_request_header (m_request, "Content-Type", "application/x-www-form-urlencoded");
        break;

      case HEADER_XMLCONTENT:
        ne_add_request_header (m_request, "Content-Type", "text/xml");
        break;
    }
  }

  std::ostream&
  operator<<(std::ostream& os, Request & r)
  {
    r.dispatch ();
    os.write (reinterpret_cast<const char*>(&(r.m_vec[0])), r.m_vec.size());
    return os;
  }

  std::ostream&
  Request::operator<<(std::ostream& os)
  {
    dispatch ();
    os.write (reinterpret_cast<const char*>(&(m_vec[0])), m_vec.size());
    return os;
  }

  void
  Request::operator>> (std::string& out)
  {
    dispatch ();
    if (m_vec.size()) out.append (reinterpret_cast<const char*>(&m_vec[0])); 
  }

  const unsigned char *
  Request::get_data ()
  {
    return &m_vec[0];
  }

  size_t
  Request::get_data_size ()
  {
    return m_vec.size();
  }

  std::string
  Request::get_error () const
  {
    return std::string (ne_get_error (ne_get_session (m_request)));
  }

}
