// Neon++ Library, (C) 2006 M. Derezynski
// A C++ wrapper library for neon <http://www.webdav.org/neon>, (C) Joe Orton

#ifndef _NEON_PP_REQUEST_
#define _NEON_PP_REQUEST_

#include <string>
#include <vector>
#include <iostream>
#include <fstream>

#include <ne_session.h>
#include <ne_request.h>
#include <sigc++/sigc++.h>

#include "session.hh"

namespace Neon
{
  typedef unsigned int HttpPort;
  class Request
  {
    public:

      typedef sigc::signal<void, double> SignalRequestProgress;

      /** The method to use for a request. Currently GET and POST are supported.
       *
       */
      enum Method
      {
        METHOD_POST,
        METHOD_GET
      };

      /** Whether to read the response from the request, or discard it
       *
       */
      enum ResponseRead
      {
        RESPONSE_READ,
        RESPONSE_IGNORE
      };

      /** Enumeration to use with the Neon::Request::add_request_header() variant that takes in a
       *  HeaderType
       *
       *  HEADER_WWWFORM: Adds Content-type: application/x-www-form-urlencoded to the headers
       *
       */
      enum HeaderType
      {
        HEADER_WWWFORM,
        HEADER_XMLCONTENT,
      };

#include "neonpp-exception.hh"
      
      NEONPPEXCEPTION(InvalidRequestError)

#ifdef NEONPP_EXCEPTIONS_ENABLED
      NEONPPEXCEPTION(NeAuthError)
      NEONPPEXCEPTION(NeProxyAuthError)
      NEONPPEXCEPTION(NeConnectError)
      NEONPPEXCEPTION(NeTimeoutError)
      NEONPPEXCEPTION(NeGenericError)
      NEONPPEXCEPTION(NeUnableToConnectError)
#endif //NEONPP_EXCEPTIONS_ENABLED

      /** Creates a request object
       *
       * @param hostname The hostname to use. Note that this shall not contain the preceding http:// or https://
       * @param path The path to get from the host, defaults to '/'
       * @param port The port to use, defaults to port 80
       * @param read Whether to read the response from this request. Defaults to Neon::Request::RESPONSE_READ
       * @param method The method to use for the request. Defaults to Neon::Request::METHOD_GET
       * @param scheme The scheme to use. Can be one of "http" or "https"
       *
       */
      Request ( std::string const&    hostname,
                std::string const&    path                = "/", 
                Neon::HttpPort        port                = 80,
                ResponseRead          read                = RESPONSE_READ,
                Method                method              = METHOD_GET,
                std::string const&    scheme              = "http",
                bool                  direct_file_output  = false,
                std::string const&    output_filename     = std::string())

      throw (Neon::Session::SessionCreateError, InvalidRequestError);

      ~Request ();      

      /** Puts a request body in case of a POST request
       *
       * @param data The data to be pushed. Thsi is const char * to accomodate for binary
                     data that might contain zeros in between
       * @param data_size The size of the data to be pushed out
       *
       */
      void set_request_data (const char * data, size_t data_size);

      void set_check_server (bool check);

      /** Adds a request header giving the name and value of the header, e.g. "Content-type" and "text/xml"
       *
       * @param name The name of the header
       * @param value The content of this header
       *
       */
      void add_request_header (std::string const& name, std::string const& value);

      /** Adds a request header using one of the predefined types in Neon::Request::HeaderType
       *
       * @param Neon::Request::HeaderType
       *
       */
      void add_request_header (HeaderType type);

      /** Dispatches the request manually. Note that this is usually not needed if you use one of the operators << and >>,
       *  as they will call dispatch internally themselves
       *
       */
#ifdef NEONPP_EXCEPTIONS_ENABLED
      void dispatch ();
#else
      int dispatch ();
#endif

      /** Resets the request, that is, clears the (potentially read) response data, and allows for re-fetching using
       *  Neon::Request::dispatch(), or operator << or >>
       *
       */
      void clear ();

      /** Gets the ne_status of the current request
       *
       */
      const ne_status * get_status () const;

      /** Gets the current error string, if any
       *
       */
      std::string get_error () const;

      /** Gets the read data
       *
       */
      const unsigned char * get_data ();

      /** Gets the data size
       *
       */
      size_t get_data_size ();

      /** Allows for easily outputting the read context to an std::string
       *
       */
      void operator>> (std::string&);

      /** Allows for appending the data content into an ostream, like an ofstream or a stringstream
       *
       */
      std::ostream& operator<<(std::ostream& os);

      /** Allows for appending the data content into an ostream, like an ofstream or a stringstream
       *
       */
      friend std::ostream& operator<<(std::ostream& os, Request & r);

      SignalRequestProgress&
      signal_request_progress () { return s_request_progress_; }

    private:

      SignalRequestProgress s_request_progress_;

      std::ofstream * m_o;

      Request ()
          : m_request (0),
            m_session (0),
            m_check_server (false) {}

      static void progress (void * udata, off_t progress, off_t total);

      static int read_block (void * udata, const char * data, size_t len);
      static int read_block_file (void * udata, const char * data, size_t len);

      ne_request   *m_request;
      Session      *m_session;
      bool          m_dispatched;
      ResponseRead  m_read; 
      Method        m_method;      
      bool          m_check_server;

      std::vector<unsigned char> m_vec;

      bool          m_direct_file_output;
      std::string   m_output_file;
  };
}
#endif
