/*
*
*  BlueZ - Bluetooth protocol stack for Linux
*
*  Copyright (C) 2004-2005  Marcel Holtmann <marcel@holtmann.org>
*
*
*  This library is free software; you can redistribute it and/or
*  modify it under the terms of the GNU Lesser General Public
*  License as published by the Free Software Foundation; either
*  version 2.1 of the License, or (at your option) any later version.
*
*  This library is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*  Lesser General Public License for more details.
*
*  You should have received a copy of the GNU Lesser General Public
*  License along with this library; if not, write to the Free Software
*  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <errno.h>
#include <malloc.h>
#include <signal.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <syslog.h>

#include <alsa/asoundlib.h>
#include <alsa/pcm_external.h>
#include <alsa/timer.h>

#include "../a2dp.h"
#include "a2dpd_protocol.h"
#include "a2dpd_protocol.h"
#include "a2dpd_timer.h"
#include "a2dpd_ipc.h"

char* g_prefix="PCM.A2DPD2";
FILE* g_fdout = DEFAULTFDOUT;
int g_bdebug = DEFAULTDEBUG;

typedef struct snd_pcm_a2dp {
	snd_pcm_ioplug_t io;
	int opened_for;
	int streamid;
	int rateconstraint;
	int rate;
	int sk;
	int channels;
	snd_pcm_sframes_t num;
	unsigned int frame_bytes;
	TIMERINFO TimerInfos;
} snd_pcm_a2dp_t;

static int a2dp_disconnect(snd_pcm_a2dp_t * a2dp)
{
	close_socket(&a2dp->sk);
	return 0;
}

static int a2dp_connect(snd_pcm_a2dp_t * a2dp)
{
	if (a2dp->sk <= 0) {
		int sockfd = make_client_socket();
		a2dp->sk = -1;
		if (sockfd > 0) {
			int32_t client_type = a2dp->opened_for;
			//if(fcntl(sockfd, F_SETFL, O_NONBLOCK)<0) {
			//	DBG("Failed to set socket %d non blocking", sockfd);
			//}
			if (send_socket(sockfd, &client_type, sizeof(client_type)) == sizeof(client_type)) {
				// Fill stream informations
				AUDIOSTREAMINFOS StreamInfos = INVALIDAUDIOSTREAMINFOS;
				StreamInfos.rate = a2dp->rate;
				StreamInfos.channels = a2dp->channels;
				StreamInfos.bitspersample = a2dp->frame_bytes/a2dp->channels;
				StreamInfos.streamid = a2dp->streamid;
				StreamInfos.buffer_size = a2dp->io.buffer_size;
				switch(a2dp->io.format) {
				case SND_PCM_FORMAT_S8:     StreamInfos.format = A2DPD_PCM_FORMAT_S8; break;
				case SND_PCM_FORMAT_U8:     StreamInfos.format = A2DPD_PCM_FORMAT_U8; break;
				case SND_PCM_FORMAT_S16_LE: StreamInfos.format = A2DPD_PCM_FORMAT_S16_LE; break;
				default: StreamInfos.format = A2DPD_PCM_FORMAT_UNKNOWN; break;
				}
				if (send_socket(sockfd, &StreamInfos, sizeof(StreamInfos)) == sizeof(StreamInfos)) {
					a2dp->sk = sockfd;
					DBG("Connected a2dp %p, sk %d, fps %ld", a2dp, a2dp->sk, (long)(a2dp->TimerInfos.fpsX/TIMERFACTOR));
				} else {
					a2dp_disconnect(a2dp);
				}
			} else {
				close_socket(&sockfd);
			}
		}
	}
	return 0;
}

static inline snd_pcm_a2dp_t *a2dp_alloc(void)
{
	snd_pcm_a2dp_t *a2dp;
	a2dp = malloc(sizeof(*a2dp));
	if (a2dp) {
		memset(a2dp, 0, sizeof(*a2dp));
		a2dp->sk = -1;
	}
	return a2dp;
}

static inline void a2dp_free(snd_pcm_a2dp_t * a2dp)
{
	a2dp_disconnect(a2dp);
	free(a2dp);
}

static int a2dp_start(snd_pcm_ioplug_t * io)
{
	//snd_pcm_a2dp_t *a2dp = io->private_data;
	//FIXME
	return 0;
}

static int a2dp_stop(snd_pcm_ioplug_t * io)
{
	//snd_pcm_a2dp_t *a2dp = io->private_data;
	return 0;
}

static snd_pcm_sframes_t a2dp_pointer(snd_pcm_ioplug_t * io)
{
	snd_pcm_a2dp_t *a2dp = io->private_data;
	//DBG("returning a2dp->num = %lu", a2dp->num);z
	return a2dp->num;
}

// This is the main transfer func which does the transfer and sleep job
static snd_pcm_sframes_t a2dp_transfer2(snd_pcm_ioplug_t * io, char *buf, int32_t datatoread)
{
	snd_pcm_a2dp_t *a2dp = io->private_data;
	int transfer = 0;
	int delay = 0;
	AUDIOPACKETHEADER hdr;
	gettimeofday(&hdr.packet_date, NULL);
	hdr.pcm_buffer_size = datatoread;

	// Connect if needed and send
	a2dp_connect(a2dp);
	if (transfer >= 0)
		transfer = send_socket(a2dp->sk, &hdr, sizeof(hdr));
	if (transfer >= 0)
		transfer = send_socket(a2dp->sk, buf, hdr.pcm_buffer_size);

	// Disconnect if error detected
	if (transfer < 0) {
		DBG2("Error while sending");
		a2dp_disconnect(a2dp);
	}

	// The data are sent to the daemon that act as a proxy thus we double transfer delay to compensate latency
	a2dp_timer_notifyframe(&a2dp->TimerInfos);
	delay = a2dp_timer_sleep(&a2dp->TimerInfos, 4*A2DPTIMERPREDELAY);
	if(delay) {
		// Beware : usleep 0 does wait!!!
		usleep(delay);
	}

	// Stats
	if (a2dp->TimerInfos.display > 0) {
		if (errno != 0 || transfer <= 0) {
			//syslog(LOG_INFO, "send_socket(%d bytes)=%d (errno=%d:%s)", datatoread, transfer, errno, strerror(errno));
		}
	}
	// update pointer, tell alsa we're done
	a2dp->num += datatoread / a2dp->frame_bytes;

	return datatoread / a2dp->frame_bytes;
}

// also works but sleeps between transfers
// This is the main transfer func which does the transfer and sleep job
static snd_pcm_sframes_t a2dp_transfer(snd_pcm_ioplug_t * io, const snd_pcm_channel_area_t * areas, snd_pcm_uframes_t offset, snd_pcm_uframes_t nframes)
{
	snd_pcm_a2dp_t *a2dp = io->private_data;
	int i = 0;
	snd_pcm_sframes_t totaltransfered = 0;
	while (i++ < 1 && totaltransfered < nframes) {
		char *buf = (char *) areas->addr + (areas->first + areas->step * offset) / 8;
		int datatoread = min(A2DPD_BLOCK_SIZE, nframes * a2dp->frame_bytes);
		snd_pcm_sframes_t transfered = a2dp_transfer2(io, buf, datatoread);
		if (transfered > 0) {
			offset += transfered;
			totaltransfered += transfered;
		} else {
			break;
		}
	}
	return totaltransfered;
}

// This is the main transfer func which does the transfer and sleep job
static snd_pcm_sframes_t a2dp_read2(snd_pcm_ioplug_t * io, char *buf, int32_t datatoread)
{
	snd_pcm_a2dp_t *a2dp = io->private_data;
	int transfer = 0;
	uint32_t pkt_size = 0;
	int dummy = 0;
	int delay = 0;

	// Connect if needed and send
	a2dp_connect(a2dp);

	DBG("writing 2");
	if(transfer>=0)
		transfer = send_socket(a2dp->sk, &dummy, sizeof(dummy));

	DBG("reading 2 = %d", transfer);
	if(transfer>=0)
		transfer = recv_socket(a2dp->sk, &pkt_size, sizeof(pkt_size));
	DBG("got size %d:%d", transfer, pkt_size);
	if(transfer>=0) {
		if(pkt_size<datatoread) {
			transfer = recv_socket(a2dp->sk, buf, min(datatoread, pkt_size));
			DBG("got data %d/%d", transfer, pkt_size);
		} else {
			DBG("invalid pkt size");
		}
	}

	// update pointer, tell alsa we're done
	if(transfer>=0)
		a2dp->num += transfer / a2dp->frame_bytes;

	// Disconnect if error detected
	if (transfer < 0) {
		a2dp_disconnect(a2dp);
		transfer = 0;
	}

	a2dp_timer_notifyframe(&a2dp->TimerInfos);
	delay = a2dp_timer_sleep(&a2dp->TimerInfos, 0);
	//usleep(delay);

	return transfer / a2dp->frame_bytes;
}

static snd_pcm_sframes_t a2dp_read(snd_pcm_ioplug_t *io,
				  const snd_pcm_channel_area_t *areas,
				  snd_pcm_uframes_t offset,
				  snd_pcm_uframes_t nframes)
{
	snd_pcm_a2dp_t *a2dp = io->private_data;

	DBG("reading");

	char *buf = (char *) areas->addr + (areas->first + areas->step * offset) / 8;
	int datatoread = min(A2DPD_BLOCK_SIZE, nframes * a2dp->frame_bytes);
	snd_pcm_uframes_t totaltransfered = a2dp_read2(io, buf, datatoread);
	return totaltransfered;
}


static int a2dp_close(snd_pcm_ioplug_t * io)
{
	snd_pcm_a2dp_t *a2dp = io->private_data;

	DBG("");

	a2dp_disconnect(a2dp);
	a2dp_free(a2dp);
	DBG("OK");
	return 0;
}

static int a2dp_params(snd_pcm_ioplug_t * io, snd_pcm_hw_params_t * params)
{
	snd_pcm_a2dp_t *a2dp = io->private_data;
	unsigned int period_bytes;

	a2dp->frame_bytes = (snd_pcm_format_physical_width(io->format) * io->channels) / 8;

	period_bytes = io->period_size * a2dp->frame_bytes;

	DBG("format %s rate %d channels %d", snd_pcm_format_name(io->format), io->rate, io->channels);

	return 0;
}

static int a2dp_prepare(snd_pcm_ioplug_t * io)
{
	snd_pcm_a2dp_t *a2dp = io->private_data;
	long block = ((long)A2DPD_BLOCK_SIZE);

	a2dp->num = 0;
	a2dp->rate = io->rate;
	a2dp->channels = io->channels;

	if(a2dp->opened_for == A2DPD_PLUGIN_PCM_READ) {
		block = 48;
		// ALSA library is really picky on the fact num is not null. If it is, capture won't start 
		a2dp->num = io->period_size;
	}

	a2dp->TimerInfos.fpsX = a2dp->rate * a2dp->frame_bytes * TIMERFACTOR / block;

	DBG("block %ld, %ld fps", block, (long)(a2dp->TimerInfos.fpsX / TIMERFACTOR));

	return 0;
}

static snd_pcm_ioplug_callback_t a2dp_write_callback = {
	.close = a2dp_close,
	.start = a2dp_start,
	.stop = a2dp_stop,
	.prepare = a2dp_prepare,
	.pointer = a2dp_pointer,
	.transfer = a2dp_transfer,
	.hw_params = a2dp_params,
};

static snd_pcm_ioplug_callback_t a2dp_read_callback = {
	.close = a2dp_close,
	.start = a2dp_start,
	.stop = a2dp_stop,
	.prepare = a2dp_prepare,
	.pointer = a2dp_pointer,
	.transfer = a2dp_read,
	.hw_params = a2dp_params,
};

// Alsa can convert about any format/channels/rate to any other rate
// However, since we added some code in the daemon to convert, why not do it ourselves!!!
// Moreover some player like aplay won't play a wav file if the device that do not natively support the requested format
// If you want alsa to do the conversion, just remove the value you want to see converted
static int a2dp_constraint(snd_pcm_a2dp_t * a2dp)
{
	snd_pcm_ioplug_t *io = &a2dp->io;
	snd_pcm_access_t access_list[] = { SND_PCM_ACCESS_RW_INTERLEAVED, SND_PCM_ACCESS_MMAP_INTERLEAVED };
	unsigned int formats[] = { SND_PCM_FORMAT_U8, SND_PCM_FORMAT_S8, SND_PCM_FORMAT_S16_LE };
	unsigned int channels[] = { 1, 2 };
	unsigned int rates[] = { 8000, 11025, 22050, 32000, 44100, 48000 };
	int formats_nb = ARRAY_SIZE(formats);
	int channels_nb = ARRAY_SIZE(channels);
	int rates_nb = ARRAY_SIZE(rates);
	int rate_daemon = 0;
	int rate_prefered = 0;
	int period_bytes = 8192;
	char srcfilename[512];
	int err;

	if(a2dp->opened_for == A2DPD_PLUGIN_PCM_WRITE) {
		get_config_filename(srcfilename, sizeof(srcfilename));
		// Default is same as the daemon
		rate_daemon = read_config_int(srcfilename, "a2dpd", "rate", A2DPD_FRAME_RATE);
		// If a value is specified, use it
		rate_prefered = read_config_int(srcfilename, "a2dpd", "plugin-rate", rate_daemon);

		// If a constraint is defined for the plugin use it
		if(a2dp->rateconstraint != -1) {
			rate_prefered = a2dp->rateconstraint;
		}

		// If this value is not 0, alsa will convert to plugin-rate
		if(rate_prefered != 0) {
			// use defaults settings the rate specified + 16 bits stereo
			rates[0] = rate_prefered;
			rates_nb = 1;
			formats[0] = SND_PCM_FORMAT_S16_LE;
			formats_nb = 1;
			channels[0] = 2;
			channels_nb = 1;
			period_bytes = 8192;
			DBG("%d", rate_prefered);
		} else {
			// If this value is 0, the daemon will do most conversions
		}
		DBG("A2DPD_PLUGIN_PCM_WRITE %d", rate_prefered);
	} else if(a2dp->opened_for == A2DPD_PLUGIN_PCM_READ) {
		// Capture is only available using SCO
		// 8000 hz, 16bits, Mono
		rates[0] = 8000;
		rates_nb = 1;
		formats[0] = SND_PCM_FORMAT_S16_LE;
		formats_nb = 1;
		channels[0] = 1;
		channels_nb = 1;
		period_bytes = 48;
		DBG("A2DPD_PLUGIN_PCM_READ %d", 8000);
	}

	err = snd_pcm_ioplug_set_param_list(io, SND_PCM_IOPLUG_HW_ACCESS, ARRAY_SIZE(access_list), access_list);
	if (err < 0)
		return err;

	err = snd_pcm_ioplug_set_param_list(io, SND_PCM_IOPLUG_HW_FORMAT, formats_nb, formats);
	if (err < 0)
		return err;

	err = snd_pcm_ioplug_set_param_list(io, SND_PCM_IOPLUG_HW_CHANNELS, channels_nb, channels);
	if (err < 0)
		return err;

	err = snd_pcm_ioplug_set_param_list(io, SND_PCM_IOPLUG_HW_RATE, rates_nb, rates);
	if (err < 0)
		return err;

	err = snd_pcm_ioplug_set_param_minmax(io, SND_PCM_IOPLUG_HW_PERIOD_BYTES, 8192, 8192);
	if (err < 0)
		return err;

	err = snd_pcm_ioplug_set_param_minmax(io, SND_PCM_IOPLUG_HW_PERIODS, 2, 2);
	if (err < 0)
		return err;

	return 0;
}

SND_PCM_PLUGIN_DEFINE_FUNC(a2dpd2)
{
	snd_pcm_a2dp_t *a2dp = NULL;
	snd_config_iterator_t i, next;
	int err = 0;
	long rateconstraint = -1;
	long streamid = 0;
	long debug = 0;

	DBG("Open mode is for %s", stream == SND_PCM_STREAM_PLAYBACK ? "Playback" : "Capture");

	snd_config_for_each(i, next, conf) {
		snd_config_t *n = snd_config_iterator_entry(i);
		const char *id;

		if (snd_config_get_id(n, &id) < 0)
			continue;

		// Alsa specific options
		if (!strcmp(id, "comment") || !strcmp(id, "type"))
			continue;

		// Ignore old options
		if (strstr("ipaddr bdaddr port src dst use_rfcomm", id))
			continue;

		// rate of the plugin (overwrite plugin-rate in .a2dprc)
		if (!strcmp(id, "rateconstraint")) {
			if (snd_config_get_integer(n, &rateconstraint) < 0) {
				SNDERR("Invalid type for %s", id);
				return -EINVAL;
			}
			continue;
		}

		// streamid (to be used later)
		if (!strcmp(id, "streamid")) {
			if (snd_config_get_integer(n, &streamid) < 0) {
				SNDERR("Invalid type for %s", id);
				return -EINVAL;
			}
			continue;
		}

		// streamid (to be used later)
		if (!strcmp(id, "debug")) {
			if (snd_config_get_integer(n, &debug) < 0) {
				SNDERR("Invalid type for %s", id);
				return -EINVAL;
			}
			continue;
		}

		SNDERR("Unknown field %s", id);
		return -EINVAL;
	}

	g_bdebug = debug;

	a2dp = a2dp_alloc();
	if (!a2dp) {
		SNDERR("Can't allocate plugin data");
		return -ENOMEM;
	}

	// Notify plugin
	a2dp->rateconstraint = rateconstraint;
	a2dp->streamid = streamid;
	a2dp->io.version = SND_PCM_IOPLUG_VERSION;
	a2dp->io.name = "Bluetooth Advanced Audio Distribution";
	a2dp->io.mmap_rw = 0;
	a2dp->io.poll_fd     =  1; /* Do not use poll !! */
	a2dp->io.poll_events =  POLLOUT; /* Do not use poll !! */
	a2dp->io.callback = (stream == SND_PCM_STREAM_PLAYBACK)?(&a2dp_write_callback):(&a2dp_read_callback);
	a2dp->io.private_data = a2dp;
	a2dp->opened_for = (stream == SND_PCM_STREAM_PLAYBACK)?(A2DPD_PLUGIN_PCM_WRITE):(A2DPD_PLUGIN_PCM_READ);

	err = snd_pcm_ioplug_create(&a2dp->io, name, stream, mode);
	if (err < 0)
		goto error;

	err = a2dp_constraint(a2dp);
	if (err < 0) {
		snd_pcm_ioplug_delete(&a2dp->io);
		goto error;
	}

	*pcmp = a2dp->io.pcm;
	return 0;

      error:
	a2dp_disconnect(a2dp);
	a2dp_free(a2dp);

	return err;
}

SND_PCM_PLUGIN_SYMBOL(a2dpd2);
